﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.Linq;
using System.Threading.Tasks;
using System.Windows;
using System.Xml.Linq;
using EffectMaker.BusinessLogic;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Properties;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the CustomActionUserDataViewModel.
    /// </summary>
    public class CustomActionUserDataGroupViewModel : PropertyGroupViewModel<CustomActionData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "AvailableSettingViewModels",
            "SelectedSettingViewModel",
        };

        /// <summary>
        /// The display names to show on the UI for the custom action items.
        /// (the first for using no custom action, and 8 custom action items)
        /// </summary>
        private KeyValuePair<string, object>[] customActionNames =
            new KeyValuePair<string, object>[9];

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public CustomActionUserDataGroupViewModel(
            HierarchyViewModel parent,
            CustomActionData dataModel)
            : base(parent, dataModel)
        {
            // Generate view model for the custom shader user data models.
            this.UpdateUserData(true);

            UserDataManager.UserDataUpdated += this.OnUserDataUpdated;

            // Always create the modification flag view model IN THE END of the constructor,
            // in case any initialization triggers the modification events.
            this.ModificationFlagViewModel = new ModificationFlagViewModel(this);
            this.ModificationFlagViewModel.AddIgnoreProperties(this.ignorePropertyNames);
        }

        /// <summary>
        /// Get or set the selected custom setting index.
        /// </summary>
        [UseDataModelOriginalValue]
        public int SelectedSettingIndex
        {
            get
            {
                return this.GetDataModelValue(() => this.SelectedSettingIndex);
            }

            set
            {
                this.SetDataModelValue(value, () => this.SelectedSettingIndex);
                this.OnPropertyChanged(() => this.SelectedSettingViewModel);
            }
        }

        /// <summary>
        /// Get the available custom shader setting view models.
        /// </summary>
        public IEnumerable<object> AvailableSettingViewModels
        {
            get
            {
                return from ch in this.Children
                       where ch is CustomActionUserSettingDataViewModel
                       select (CustomActionUserSettingDataViewModel)ch;
            }
        }

        /// <summary>
        /// Get the selected custom shader setting data view model.
        /// </summary>
        public object SelectedSettingViewModel
        {
            get
            {
                int index = this.DataModel.SelectedSettingIndex;
                if (index < 0 || index >= this.DataModel.Settings.Count)
                {
                    return null;
                }

                var dataModel = this.DataModel.Settings[index];
                if (dataModel == null)
                {
                    return null;
                }

                foreach (IHierarchyObject child in this.Children)
                {
                    var vm = child as CustomActionUserSettingDataViewModel;
                    if (vm != null && vm.DataModel == dataModel)
                    {
                        return vm;
                    }
                }

                return null;
            }
        }

        /// <summary>
        /// コールバックIDタイプを取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> CallbackTypeItems
        {
            get { return this.customActionNames; }
        }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// Dispose this object.
        /// </summary>
        public override void Dispose()
        {
            UserDataManager.UserDataUpdated -= this.OnUserDataUpdated;

            base.Dispose();
        }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            this.UpdateUserData(true);
            base.UpdateChildViewModels();
        }

        /// <summary>
        /// Create a data model proxy.
        /// This method is called in the constructor.
        /// If you need a specific type of data model proxy,
        /// override this method and return the desired data model proxy.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The created data model proxy.</returns>
        protected override DataModelProxy CreateDataModelProxy(DataModelBase dataModel)
        {
            return new CustomActionDataProxy(dataModel);
        }

        /// <summary>
        /// インデックスだけをエクスポート
        /// </summary>
        /// <param name="container">Container to export values to.</param>
        protected override void ExportValues(IPushOnlyContainer<XElement> container)
        {
            var root = new XElement("CustomActionSettings");
            root.Add(new XAttribute("index", this.SelectedSettingIndex));
            container.Add(root);
        }

        /// <summary>
        /// インデックスだけをインポート
        /// </summary>
        /// <param name="elements">Sequence of XML elements to import from.</param>
        protected override void ImportValues(IEnumerable<XElement> elements)
        {
            if (elements == null)
            {
                throw new ArgumentNullException("elements");
            }

            XElement root = elements.FirstOrDefault(x => x.Name == "CustomActionSettings");
            if (root == null)
            {
                throw new FormatException("CustomActionSettings");
            }

            int index;
            if (int.TryParse(root.Attribute("index").Value, out index))
            {
                this.SelectedSettingIndex = index;
            }
        }

        /// <summary>
        /// Handle UserDataUpdated event from the user data manager.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnUserDataUpdated(
            object sender,
            UserDataUpdatedEventArgs e)
        {
            if (e.IsUserDataTypeUpdated(UpdatedUserDataTypes.CustomAction) == false)
            {
                // The updated user data is not for custom shader, bail out.
                return;
            }

            this.UpdateUserData(false);
        }

        /// <summary>
        /// Helper method for updating user data.
        /// </summary>
        /// <param name="onlyUpdateViewModels">
        /// True to update only the view models for to the existing user data models.
        /// Otherwise, remove all the user data models and update them.
        /// </param>
        private void UpdateUserData(bool onlyUpdateViewModels)
        {
            // First remove all the user page view models.
            var viewModels = this.Children.Where(
                ch => ch is CustomActionUserSettingDataViewModel).ToArray();
            foreach (var vm in viewModels)
            {
                this.Children.Remove(vm);
            }

            if (onlyUpdateViewModels == false)
            {
                // Update user data models.
                (this.Proxy as CustomActionDataProxy).UpdateUserData();
            }

            // Have to create the array again, otherwise the UIComboBox rejects
            // the modified array because the object reference is the same.
            // See UIComboBox.AvailableItems property, the if statement below :
            // if (this.controlExtender.SetValue(ref this.availableItems, value))
            this.customActionNames = new KeyValuePair<string, object>[8];

            // Create user page view models with the new user data models.
            int index = 0;
            foreach (CustomActionSettingData dm in this.DataModel.Settings)
            {
                var vm = new CustomActionUserSettingDataViewModel(this, dm, index);
                this.Children.Add(vm);

                CustomActionDefinition def =
                    CustomActionUserDataManager.GetCustomActionDefinition(index);

                if (def == null)
                {
                    this.customActionNames[index] = new KeyValuePair<string, object>(
                        string.Format(Properties.Resources.CustomActionCallbackTypeItem, index),
                        index);
                }
                else
                {
                    this.customActionNames[index] = new KeyValuePair<string, object>(
                        def.Caption,
                        index);
                }

                // Advance the index.
                ++index;
            }

            // Deselect custom action.
            if (this.SelectedSettingIndex >= index)
            {
                this.SelectedSettingIndex = -1;
            }

            // Clear modification flags.
            if (this.ModificationFlagViewModel != null)
            {
                this.ModificationFlagViewModel.ClearModificationFlags();
                this.ModificationFlagViewModel.ClearChildModificationFlags();
            }

            // Notify UI to update.
            this.OnPropertyChanged(() => this.CallbackTypeItems);
            this.OnPropertyChanged(() => this.AvailableSettingViewModels);
            this.OnPropertyChanged(() => this.SelectedSettingIndex);
            this.OnPropertyChanged(() => this.SelectedSettingViewModel);
        }
    }
}
