﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.UILogic.TypeSerializers
{
    /// <summary>
    /// An IXmlTypeSerializer that support all the C# language primitives.
    /// </summary>
    public class LanguagePrimitiveXmlTypeSerializer : IXmlTypeSerializer
    {
        /// <summary>
        /// Array of supported types.
        /// </summary>
        private Type[] supportedTypes = new Type[]
        {
            typeof(bool),
            typeof(byte),
            typeof(sbyte),
            typeof(char),
            typeof(short),
            typeof(ushort),
            typeof(int),
            typeof(uint),
            typeof(long),
            typeof(ulong),
            typeof(decimal),
            typeof(float),
            typeof(double),
            typeof(string),
            typeof(Guid),
        };

        /// <summary>
        /// Checks whether a given type is supported or not.
        /// </summary>
        /// <param name="type">Type to check upon.</param>
        /// <returns>Returns true of the type is supported, false otherwise.</returns>
        public bool IsSupportedType(Type type)
        {
            return this.supportedTypes.Contains(type);
        }

        /// <summary>
        /// Serializes the value.
        /// </summary>
        /// <param name="name">Name of the variable.</param>
        /// <param name="value">Value of the variable.</param>
        /// <returns>Returns a XElement containing the serialized value.</returns>
        public XElement Serialize(string name, object value)
        {
            if (value == null)
            {
                return null;
            }

            Type valueType = value.GetType();

            return new XElement(
                "lp",
                new XAttribute("name", name),
                new XAttribute("type", valueType.FullName),
                new XAttribute("value", value));
        }

        /// <summary>
        /// Deserializes the XML.
        /// </summary>
        /// <param name="xmlElement">The XML content to deserialize.</param>
        /// <param name="name">Name of the variable.</param>
        /// <param name="value">Value of the variable.</param>
        /// <returns>Returns true if deserialization was successful, false otherwise.</returns>
        public bool Deserialize(XElement xmlElement, out string name, out object value)
        {
            name = null;
            value = null;

            Type valueType = null;

            try
            {
                name = (string)xmlElement.Attribute("name");
                if (string.IsNullOrWhiteSpace(name))
                {
                    return false;
                }

                valueType = Type.GetType((string)xmlElement.Attribute("type"));
                if (valueType == null)
                {
                    return false;
                }

                value = Convert.ChangeType((string)xmlElement.Attribute("value"), valueType);
                return true;
            }
            catch
            {
                return false;
            }
        }
    }
}
