﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Runtime.InteropServices;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.Properties;
using EffectMaker.UILogic.ViewModels;
using Microsoft.Win32;

namespace EffectMaker.UILogic.Manager
{
    /// <summary>
    /// ヘルプマネージャです。
    /// </summary>
    public class HelpManager
    {
        /// <summary>
        /// トップページのIDです。
        /// </summary>
        private const string TopPageId = "83955553";

        /// <summary>
        /// ニュースページのIDです。
        /// </summary>
        private const string NewsPageId = "113826765";

        /// <summary>
        /// トラブルシューティングページのIDです。
        /// </summary>
        private const string TroubleshootingPageId = "162201368";

        /// <summary>
        /// ヘルプを表示する.
        /// </summary>
        /// <param name="pageId">表示するページID</param>
        /// <param name="pageAnchor">表示するページ内のアンカー</param>
        public static void ShowHelp(string pageId = "", string pageAnchor = "")
        {
            // ユーザー設定からドキュメントパスを絶対パス化して取得
            var helpPath = PathUtility.GetDirectoryName(PathUtility.ToAbsolutePath(OptionStore.RootOptions.Basic.DocumentPath, IOConstants.ExecutableFolderPath));

            // プロジェクト設定でドキュメントパスが設定されていたらそちらで上書き
            if (IOConstants.DefaultDocumentPath == OptionStore.RootOptions.Basic.DocumentPath ||
                String.IsNullOrEmpty(helpPath))
            {
                if (!String.IsNullOrEmpty(OptionStore.ProjectConfig.DocumentPath))
                {
                    helpPath = PathUtility.GetDirectoryName(PathUtility.ToAbsolutePath(OptionStore.ProjectConfig.DocumentPath, IOConstants.ExecutableFolderPath));
                }
            }

            // 無効なパスだった場合は警告を表示してドキュメントの配置を促す
            if (!Directory.Exists(helpPath))
            {
                var errMsg = new string[2];
                errMsg[0] = Resources.HelpNintendoSDKDocumentText;
                errMsg[1] = Resources.HelpNintendoSDKDocumentCaption;
                WorkspaceRootViewModel.Instance.Dialogs.OnShowWarningDialogExecutable.Execute(errMsg);
                return;
            }

            // ページの指定がない場合はトップページを表示する.
            if (string.IsNullOrEmpty(pageId))
            {
                pageId = TopPageId;
                pageAnchor = string.Empty;
            }

            // Internet Explorerの場合はURL文字列のエンコーディングを変えないといけないので
            // 実行ファイル名の名前で分岐(ブラウザパスが空の場合はIEの前提で動く)
            var browserPath = GetDefaultBrowserExePath();
            bool isInternetExplorer = string.IsNullOrEmpty(browserPath) || browserPath.Contains("iexplore.exe");

            // URL文字列生成
            string url;
            if (string.IsNullOrEmpty(pageAnchor))
            {
                // 指定されたページを表示する
                string pagePath = System.IO.Path.GetFullPath(helpPath).Replace('\\', '/');
                url = string.Format(
                    "file:///{0}/Package/contents/Pages/Page_{1}.html",
                    isInternetExplorer ? pagePath : Uri.EscapeUriString(pagePath),
                    pageId);
            }
            else
            {
                // 指定されたページ内のアンカーを表示する
                string pagePath = System.IO.Path.GetFullPath(helpPath).Replace('\\', '/');
                url = string.Format(
                    "file:///{0}/Package/contents/Pages/Page_{1}.html#{2}",
                    isInternetExplorer ? pagePath : Uri.EscapeUriString(pagePath),
                    pageId,
                    pageAnchor);
            }

            // ヘルプを表示する
            try
            {
                if (string.IsNullOrEmpty(browserPath) || !File.Exists(browserPath))
                {
                    // ブラウザパスが空だった、不正だった場合はファイル関連付けを信じてファイルパスのみで開く
                    System.Diagnostics.Process.Start(url);
                }
                else
                {
                    // ブラウザパスが取得できた場合は、明示的に引数を渡して開く
                    System.Diagnostics.Process.Start(browserPath, url);
                }
            }
            catch
            {
                Logger.Log("LogView", LogLevels.Error, Resources.ErrorOpenHelp);
            }
        }

        /// <summary>
        /// 既定ブラウザのexeファイルパスを取得します.
        /// </summary>
        /// <returns>既定ブラウザのexeファイルパス</returns>
        private static string GetDefaultBrowserExePath()
        {
            const string userChoice = @"Software\Microsoft\Windows\Shell\Associations\UrlAssociations\http\UserChoice";
            try
            {
                using (var key = Registry.CurrentUser.OpenSubKey(userChoice))
                {
                    if (key != null)
                    {
                        var progIdValue = key.GetValue("ProgId");
                        if (progIdValue != null)
                        {
                            using (var pathKey = Registry.ClassesRoot.OpenSubKey(progIdValue.ToString() + @"\shell\open\command"))
                            {
                                if (pathKey != null)
                                {
                                    var command = pathKey.GetValue(null).ToString();

                                    // 引数がついているかもしれないのでコマンド部分だけ取得
                                    int argc;
                                    var argv = CommandLineToArgvW(command, out argc);
                                    if (argv != IntPtr.Zero)
                                    {
                                        try
                                        {
                                            // 0 番目の文字列を取得
                                            var arg = Marshal.ReadIntPtr(argv, 0);
                                            return Marshal.PtrToStringUni(arg);
                                        }
                                        finally
                                        {
                                            Marshal.FreeHGlobal(argv);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch
            {
            }

            return string.Empty;
        }

        /// <summary>
        /// ニュースを表示します。
        /// </summary>
        public static void ShowNews()
        {
            ShowHelp(NewsPageId);
        }

        /// <summary>
        /// トラブルシューティングを表示します。
        /// </summary>
        public static void ShowTroubleshooting()
        {
            ShowHelp(TroubleshootingPageId);
        }

        /// <summary>
        /// Unicode コマンドライン文字列から Unicode 版の argv と argc の値を取得します。
        /// </summary>
        /// <param name="lpCmdLine">コマンドライン文字列へのポインタ</param>
        /// <param name="pNumArgs">引数の数を受け取る変数へのポインタ</param>
        /// <returns>
        /// 関数が成功すると、組み立てられた引数リストへの NULL 以外のポインタが返ります。
        /// この引数リストは、Unicode ワイド文字引数文字列の集合です。
        /// 関数が失敗すると、NULL が返ります。
        /// </returns>
        [DllImport("shell32.dll", SetLastError = true)]
        private static extern IntPtr CommandLineToArgvW([MarshalAs(UnmanagedType.LPWStr)] string lpCmdLine, out int pNumArgs);
    }
}
