﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Dynamic;
using System.Linq;
using System.Text;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// SetDataModelPropertyCommand
    /// </summary>
    [Alias("SetPropertyValue")]
    public class SetPropertyCommand : CommandBase
    {
        /// <summary>ビューモデルのインスタンス</summary>
        private DynamicObject propertyOwner = null;

        /// <summary>ビューモデルのインスタンス</summary>
        private IHierarchyObject target = null;

        /// <summary>変更対象のプロパティ名</summary>
        private string propertyName = null;

        /// <summary>元の値</summary>
        private object originalValue = null;

        /// <summary>新しい値</summary>
        private object newValue = null;

        /// <summary>複数ノード編集時でリロードが必要なプロパティか否か</summary>
        private bool isReloadOnMultiEdit = false;

        /// <summary>複数ノード編集用のビット演算を管理するオブジェクト</summary>
        private MultiNodeEditUtil.BitOperator bitOperator = null;

        /// <summary>
        /// Stores the root view model.
        /// </summary>
        private WorkspaceRootViewModel root;

        /// <summary>
        /// Stores the selected workspace node view model.
        /// </summary>
        private WorkspaceNodeViewModelBase workspaceNodeViewModel;

        /// <summary>
        /// Stores the selected property page view model.
        /// </summary>
        private HierarchyViewModel propertyPageViewModel;

        /// <summary>
        /// 変更したプロパティ情報を格納する文字列
        /// </summary>
        private string commandInfo;

        /// <summary>
        /// 複数選択時のアンドゥ操作です。
        /// </summary>
        private readonly List<Action> multiUndoActions;

        /// <summary>
        /// 複数選択時のリドゥ操作です。
        /// </summary>
        private readonly List<Action> multiRedoActions;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public SetPropertyCommand()
        {
            this.multiUndoActions = new List<Action>();
            this.multiRedoActions = new List<Action>();
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="propertyOwner">ビューモデルのインスタンス</param>
        /// <param name="propertyName">変更対象のプロパティ名</param>
        /// <param name="originalValue">元の値</param>
        /// <param name="newValue">新しい値</param>
        /// <param name="isReloadOnMultiEdit">複数ノード編集でリロードが必要か</param>
        public SetPropertyCommand(
            DynamicObject propertyOwner,
            string propertyName,
            object originalValue,
            object newValue,
            bool isReloadOnMultiEdit)
            : this()
        {
            object[] args = new object[] { propertyOwner, propertyName, originalValue, newValue, isReloadOnMultiEdit };

            this.Initialize(args);
        }

        /// <summary>
        /// 呼び出し元を返すようにオーバーライド
        /// </summary>
        /// <returns>呼び出し元を含むクラス名</returns>
        public override string ToString()
        {
            return "SetProperty : " + this.commandInfo;
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">The arguments for the command.</param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 5)
            {
                return false;
            }

            this.propertyOwner = args[0] as DynamicObject;
            this.propertyName = args[1] as string;
            this.originalValue = args[2].CloneIfCloneable();
            this.newValue = args[3].CloneIfCloneable();
            this.isReloadOnMultiEdit = (bool)args[4];

            // propertyOwnerから最上位のtargetを取得する
            var targetObj = this.propertyOwner as IHierarchyObject;
            this.target = CommandBase.FindCommandStackTargetFromHierarchyObject(targetObj) as IHierarchyObject;

            Debug.Assert(
                this.target != null,
                "SetDataModelPropertyCommand.Initialize : The target is not found.");

            Debug.Assert(
                this.propertyName != null,
                "SetDataModelPropertyCommand.Initialize : The propertyName is empty.");

            IHierarchyObject page = targetObj.FindNearestParentOfType<PropertyPageViewModel>();
            page = page ?? this.target;

            this.propertyPageViewModel = page as HierarchyViewModel;

            this.workspaceNodeViewModel = targetObj
                .FindNearestParentOfType<WorkspaceNodeViewModelBase>();

            this.root = targetObj
                .FindFarthestParentOfType<WorkspaceRootViewModel>();

            this.commandInfo = string.Format(
                "{0}.{1} => {2}",
                this.propertyOwner.GetType().Name,
                this.propertyName,
                this.newValue.ToString());

            // ビット操作であるかを検出し、そうであればオペレータを生成します。
            var viewModel = this.propertyOwner as PropertyGroupViewModel;
            if (viewModel != null && viewModel.PropertyAttributes.ContainsKey(this.propertyName))
            {
                if (viewModel.PropertyAttributes[this.propertyName] == "BitField")
                {
                    this.bitOperator =
                        new MultiNodeEditUtil.BitOperator(this.originalValue, this.newValue);
                }
            }

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得する
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return this.target;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            // 現在選択中のノードについてはこのコマンドの呼び出し元で値をセット済みです

            // 一部のプロパティでは複数ノード同時編集を無効化する（今のところカラーの交換のみ）
            if (!MultiNodeEditUtil.DisableMultiNodeEdit)
            {
                if (this.isReloadOnMultiEdit)
                {
                    var topViewModel = this.target as ViewModelBase;
                    using (new MessageBlockerWithSendBinaryOnce(topViewModel.Proxy.DataModel))
                    {
                        this.DoCore();
                    }
                }
                else
                {
                    this.DoCore();
                }
            }

            // メニューの状態を更新
            WorkspaceRootViewModel.Instance.UpdateUIStates();

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            this.SetupView();

            // 元の値に戻す
            bool result;
            if (this.isReloadOnMultiEdit)
            {
                var topViewModel = this.target as ViewModelBase;
                using (new MessageBlockerWithSendBinaryOnce(topViewModel.Proxy.DataModel))
                {
                    result = this.UndoOrRedoCore(this.originalValue, this.multiUndoActions);
                }
            }
            else
            {
                result = this.UndoOrRedoCore(this.originalValue, this.multiUndoActions);
            }

            // メニューの状態を更新
            WorkspaceRootViewModel.Instance.UpdateUIStates();

            // TODO:ビューアノードのUNDO/REDO対応暫定措置
            if (this.propertyOwner is ViewerBackgroundGroupViewModel ||
                this.propertyOwner is ViewerCameraViewProjectionViewModel)
            {
                var viewer = ViewModelBase.GetParent<ViewerViewModel>((ViewModelBase)this.propertyOwner);
                if (viewer != null)
                {
                    ViewerMessageHelper.SendViewer(viewer.DataModel);
                }
            }

            return result;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            this.SetupView();

            // 新しい値をセットする
            bool result;
            if (this.isReloadOnMultiEdit)
            {
                var topViewModel = this.target as ViewModelBase;
                using (new MessageBlockerWithSendBinaryOnce(topViewModel.Proxy.DataModel))
                {
                    result = this.UndoOrRedoCore(this.newValue, this.multiRedoActions);
                }
            }
            else
            {
                result = this.UndoOrRedoCore(this.newValue, this.multiRedoActions);
            }

            // メニューの状態を更新
            WorkspaceRootViewModel.Instance.UpdateUIStates();

            // TODO:ビューアノードのUNDO/REDO対応暫定措置
            if (this.propertyOwner is ViewerBackgroundGroupViewModel ||
                this.propertyOwner is ViewerCameraViewProjectionViewModel)
            {
                var viewer = ViewModelBase.GetParent<ViewerViewModel>((ViewModelBase)this.propertyOwner);
                if (viewer != null)
                {
                    ViewerMessageHelper.SendViewer(viewer.DataModel);
                }
            }

            return result;
        }

        /// <summary>
        /// 複数ノード編集処理リストをセットアップしつつRedo処理を実行します。
        /// </summary>
        private void DoCore()
        {
            // 複数選択対象の同じプロパティに対して同じ値をセットする
            this.multiUndoActions.Clear();
            this.multiRedoActions.Clear();

            foreach (var node in WorkspaceRootViewModel.Instance.MultiSelectedNodes)
            {
                var sameTarget = HierarchyObjectExtensions.FindChildrenRecursive(
                    this.propertyOwner as IHierarchyObject,
                    node,
                    MultiNodeEditUtil.IsSameTarget) as ViewModelBase;
                if (sameTarget != null)
                {
                    object oldValue = null;
                    sameTarget.TryGetMember(
                        new EffectMakerGetMemberBinder(this.propertyName, true), out oldValue);
                    object clonedValue = oldValue.CloneIfCloneable();
                    Action undoAction = () =>
                    {
                        var binder = new EffectMakerSetMemberBinder(this.propertyName, true, false);
                        sameTarget.TrySetMember(binder, clonedValue);
                    };
                    this.multiUndoActions.Add(undoAction);

                    // ビット操作の場合とそれ以外とでRedoActionに設定する値を分岐します。
                    Action redoAction;
                    if (this.bitOperator != null)
                    {
                        var redoValue = this.bitOperator.GetOperatedValue(oldValue);
                        redoAction = () =>
                        {
                            var binder = new EffectMakerSetMemberBinder(this.propertyName, true, false);
                            sameTarget.TrySetMember(binder, redoValue);
                        };
                    }
                    else
                    {
                        redoAction = () =>
                        {
                            var binder = new EffectMakerSetMemberBinder(this.propertyName, true, false);
                            sameTarget.TrySetMember(binder, this.newValue);
                        };
                    }

                    this.multiRedoActions.Add(redoAction);

                    redoAction();
                }
            }
        }

        /// <summary>
        /// アンドゥまたはリドゥ処理を実行します。
        /// </summary>
        /// <param name="value">値</param>
        /// <param name="actions">アクションリスト</param>
        /// <returns>成功すればtrue,失敗すればfalse.</returns>
        private bool UndoOrRedoCore(object value, List<Action> actions)
        {
            // 新しい値をセットする
            var binder = new EffectMakerSetMemberBinder(this.propertyName, true, false);
            bool result = this.propertyOwner.TrySetMember(binder, value);

            // アクションリストに保存された同時編集をRedo
            foreach (var editAction in actions)
            {
                try
                {
                    editAction();
                }
                catch (Exception e)
                {
                    // エミッタセットを削除された場合には対応しきれないのでログだけ吐いておく
                    Logger.Log(LogLevels.Error, "Multi Edit Error:" + e.Message);
                }
            }

            return result;
        }

        /// <summary>
        /// Select the proper node and page.
        /// </summary>
        private void SetupView()
        {
            // カーブエディタウィンドウに切り替えターゲットを通知するための処理
            var emitterViewModel = this.workspaceNodeViewModel as EmitterViewModel;
            if (emitterViewModel != null)
            {
                emitterViewModel.CurrentEditingCurveOwener = this.propertyOwner;
            }

            this.root.SelectedNodeViewModel = this.workspaceNodeViewModel;
            this.workspaceNodeViewModel.SelectedPropertyPage = this.propertyPageViewModel;
        }
    }
}
