﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// RemoveEmitterCommand.
    /// </summary>
    [Alias("RemoveEmitter")]
    public class RemoveEmitterCommand : CommandBase
    {
        /// <summary>親ビューモデルです.</summary>
        private IEmitterOwnerViewModel parentViewModel = null;

        /// <summary>エミッタビューモデルです.</summary>
        private EmitterViewModel emitterViewModel = null;

        /// <summary>The index of the emitter view model.</summary>
        private int origViewModelIndex = -1;

        /// <summary>The index of the emitter data.</summary>
        private int origDataModelIndex = -1;

        /// <summary>The command stack target.</summary>
        private object commandStackTarget = null;

        /// <summary>エミッタの削除後に選択するノードです.</summary>
        private WorkspaceNodeViewModelBase selectNode = null;

        /// <summary>以前選択していたエミッタの処理タイプ</summary>
        private int previousUsingCpu = -1;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RemoveEmitterCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="emitterViewModel">削除するエミッタのビューモデル</param>
        public RemoveEmitterCommand(EmitterViewModel emitterViewModel)
        {
            object[] args = new object[] { emitterViewModel };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">
        /// The arguments for the command.
        /// [0] : 削除するエミッタのビューモデル
        /// </param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            // 引数の数をチェック
            if (args == null || args.Length != 1)
            {
                return false;
            }

            // 削除するエミッタのビューモデルを取得
            this.emitterViewModel = args[0] as EmitterViewModel;
            if (this.emitterViewModel == null)
            {
                return false;
            }

            // Get the parent view model.
            this.parentViewModel = this.emitterViewModel.Parent as IEmitterOwnerViewModel;
            if (this.parentViewModel == null || this.parentViewModel is WorkspaceNodeViewModelBase == false)
            {
                return false;
            }

            // Find and save the index of the emitter.
            this.parentViewModel.GetEmitterIndex(
                this.emitterViewModel,
                out this.origViewModelIndex,
                out this.origDataModelIndex);
            if (this.origViewModelIndex < 0 || this.origDataModelIndex < 0)
            {
                return false;
            }

            // エミッタの削除後に選択するノードを取得
            var parentNode = this.parentViewModel as WorkspaceNodeViewModelBase;
            this.selectNode = RemoveNodeCommandUtil.GetSelectNode(parentNode, this.emitterViewModel);
            if (this.selectNode == null)
            {
                return false;
            }

            // 遡ってCommandStackOwnerAttributeを持つインスタンスをチェックする
            var cmdStackTarget = this.parentViewModel as IHierarchyObject;
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(cmdStackTarget) as IHierarchyObject;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            // 削除前に親エミッタセットを取得
            var emitterSetData = ViewModelBase.GetParent<EmitterSetViewModel>(
                this.parentViewModel as ViewModelBase);

            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(emitterSetData.DataModel))
            {
                // 親ノードを選択する.
                // (削除したはずのノードページが操作できてしまうことを防ぐため)
                // この操作は、ノードを削除する前に実行する必要がある.
                // またノードの削除処理は、一時的に親ノードから全てのチャイルドを削除するので
                // ここで選択するのは親ノードにする必要がある.
                ((WorkspaceNodeViewModelBase) this.parentViewModel).IsSelected = true;

                this.emitterViewModel.DisconnectEmitterAnimation();

                // エミッタを削除
                bool removeResult = this.parentViewModel.RemoveEmitter(this.emitterViewModel);
                if (removeResult == false)
                {
                    return false;
                }

                // ノードの選択を変更
                this.selectNode.IsSelected = true;

                // チャイルドエミッタを削除した場合はRaiseCanUsingGpu
                var parentEmitter = this.parentViewModel as EmitterViewModel;
                if (parentEmitter != null)
                {
                    this.previousUsingCpu =
                        parentEmitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();
                    parentEmitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.IsUsingCpu = false;
                }
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
             // 削除前に親エミッタセットを取得
            var emitterSetData = ViewModelBase.GetParent<EmitterSetViewModel>(
                this.parentViewModel as ViewModelBase);

            // using構文内でエミッタセットの転送を抑制し、ディスポーザーでエミッタセットの転送を行います。
            using (new MessageBlockerWithSendBinaryOnce(emitterSetData.DataModel))
            {
                // 元あったインデックスにエミッタを戻す.
                bool insertResult = this.parentViewModel.InsertEmitter(
                    this.emitterViewModel,
                    this.origViewModelIndex,
                    this.origDataModelIndex);
                if (insertResult == false)
                {
                    return false;
                }

                // ノードの選択を変更
                this.emitterViewModel.IsSelected = true;

                this.emitterViewModel.UpdateEmitterAnimation();

                // チャイルドエミッタを復帰した場合はRaiseCanUsingGpu
                var parentEmitter = this.parentViewModel as EmitterViewModel;
                if (parentEmitter != null)
                {
                    parentEmitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu(
                        this.previousUsingCpu);
                }

                OptionStore.TriggerOptionChangedEvent(null);
            }

            return true;
        }
    }
}
