﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModelLogic.Utilities;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// バイナリ転送を抑制しつつ、解除するタイミングで1回だけリロードを行うクラスです。using構文と合わせて使います。
    /// </summary>
    public class MessageBlockerWithSendBinaryOnce : IDisposable
    {
        /// <summary>
        /// リロードの対象となるデータモデルです。
        /// </summary>
        private readonly DataModelBase dataModel;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="data">転送を抑制したいバイナリに含まれるデータモデル</param>
        public MessageBlockerWithSendBinaryOnce(DataModelBase data)
        {
            ViewerMessageHelper.BeginDisableBinaryConvertSection();
            this.dataModel = data;
        }

        /// <summary>
        /// アクションを実行し、必要に応じてリロードを行います。
        /// </summary>
        /// <param name="act">実行処理</param>
        /// <param name="eval">リロードが必要か否か</param>
        /// <param name="data">リロードの基点となるデータモデル</param>
        public static void ExecuteOnDemandReload(Action act, bool eval, DataModelBase data)
        {
            if (eval)
            {
                using (new MessageBlockerWithSendBinaryOnce(data))
                {
                    act();
                }
            }
            else
            {
                act();
            }
        }

        /// <summary>
        /// ディスポーザー
        /// 転送抑制を解除し、バイナリをリロードします.
        /// </summary>
        public void Dispose()
        {
            ViewerMessageHelper.EndDisableBinaryConvertSection();
            if (ViewerMessageHelper.DisableBinaryConvertNestCount == 0 && this.dataModel != null)
            {
                using (new ForceRefreshBinary())
                {
                    ViewerMessageHelper.FindPropertyParentDataAndSend(this.dataModel);
                }
            }
        }
    }
}
