﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.BusinessLogic.Commands
{
    /// <summary>
    /// CreateEmitterSetCommand.
    /// </summary>
    [Alias("CreateEmitterSet")]
    public class CreateEmitterSetCommand : CommandBase
    {
        /// <summary>
        /// ワークスペースのビューモデルです.
        /// </summary>
        private WorkspaceViewModel workspaceViewModel = null;

        /// <summary>
        /// エミッタセットのビューモデルです.
        /// </summary>
        private EmitterSetViewModel emitterSetViewModel = null;

        /// <summary>
        /// エミッタセットの名前です.
        /// </summary>
        private string emitterSetName = string.Empty;

        /// <summary>
        /// エミッタの名前です.
        /// </summary>
        private string emitterName = string.Empty;

        /// <summary>
        /// プレビューの名前です.
        /// </summary>
        private string previewName = string.Empty;

        /// <summary>
        /// プリセットとして使用するesetのパスです.
        /// </summary>
        private string presetPath = string.Empty;

        /// <summary>以前選択していたノード.</summary>
        private WorkspaceNodeViewModelBase previousSelectedNode = null;

        /// <summary>
        /// 挿入するインデックス
        /// </summary>
        private int insertIndex = -1;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public CreateEmitterSetCommand()
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="workspaceViewModel">ワークスペースビューモデル</param>
        /// <param name="emitterSetName">エミッタセット名</param>
        /// <param name="emitterName">エミッタ名</param>
        /// <param name="previewName">プレビュー名</param>
        /// <param name="insertIndex">挿入するインデックス</param>
        public CreateEmitterSetCommand(WorkspaceViewModel workspaceViewModel, string emitterSetName, string emitterName, string previewName, int insertIndex = -1)
        {
            object[] args = new object[] { workspaceViewModel, emitterSetName, emitterName, previewName, insertIndex };

            this.Initialize(args);
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="workspaceViewModel">ワークスペースビューモデル</param>
        /// <param name="emitterSetName">エミッタセット名</param>
        /// <param name="presetInfo">プリセット情報</param>
        /// <param name="insertIndex">挿入するインデックス</param>
        public CreateEmitterSetCommand(WorkspaceViewModel workspaceViewModel, string emitterSetName, PresetItem presetInfo, int insertIndex)
        {
            object[] args = new object[] { workspaceViewModel, emitterSetName, presetInfo, string.Empty, insertIndex };

            this.Initialize(args);
        }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">The arguments for the command.</param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || (args.Length != 4 && args.Length != 5))
            {
                return false;
            }

            // ワークスペースビューモデルを記録
            this.workspaceViewModel = args[0] as WorkspaceViewModel;
            if (this.workspaceViewModel == null)
            {
                return false;
            }

            // エミッタセット名を記録
            this.emitterSetName = args[1] as string;
            if (string.IsNullOrEmpty(this.emitterSetName))
            {
                return false;
            }

            if (args[2] is PresetItem)
            {
                // プリセットからの新規作成
                this.presetPath = ((PresetItem)args[2]).Value as string;
                if (string.IsNullOrEmpty(this.presetPath))
                {
                    return false;
                }
            }
            else
            {
                // 完全新規作成

                // エミッタ名を記録
                this.emitterName = args[2] as string;
                if (string.IsNullOrEmpty(this.emitterName))
                {
                    return false;
                }

                // プレビュー名を記録
                this.previewName = args[3] as string;
                if (string.IsNullOrEmpty(this.previewName))
                {
                    return false;
                }
            }

            // 挿入するインデックスを記録
            if (args.Length == 5)
            {
                this.insertIndex = (int)args[4];
            }

            this.previousSelectedNode = this.workspaceViewModel as WorkspaceNodeViewModelBase;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return null;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            if (!string.IsNullOrEmpty(this.presetPath))
            {
                // プリセットをもとにエミッタセットとエミッタを作成
                this.emitterSetViewModel = this.workspaceViewModel.AdaptPresetForEmitterSet(
                    this.presetPath, this.emitterSetName);
                if (this.emitterSetViewModel == null)
                {
                    return false;
                }

                this.previewName = this.emitterSetName;
            }
            else
            {
                // エミッタセットを作成
                this.emitterSetViewModel = this.workspaceViewModel.CreateNewEmitterSet(this.emitterSetName);
                if (this.emitterSetViewModel == null)
                {
                    return false;
                }

                // エミッタを作成
                EmitterViewModel emitterViewModel = this.emitterSetViewModel.CreateNewEmitter(
                    this.emitterName, null);
                if (emitterViewModel == null)
                {
                    return false;
                }

                // Temporarily disable the modification flag.
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;

                // エミッタをエミッタセットに追加
                bool addEmitterResult = this.emitterSetViewModel.AddEmitter(emitterViewModel);
                if (addEmitterResult == false)
                {
                    return false;
                }

                // Restore the flag.
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;
            }

            // プレビューを作成
            PreviewViewModel previewViewModel = this.emitterSetViewModel.CreateNewPreview(this.previewName);
            if (previewViewModel == null)
            {
                return false;
            }

            // Temporarily disable the modification flag.
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;

            // プレビューをエミッタセットに追加
            bool addPreviewResult = this.emitterSetViewModel.AddPreview(previewViewModel);
            if (addPreviewResult == false)
            {
                return false;
            }

            // Restore the flag.
            ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;

            // エミッタセットを親ノードに追加
            bool addEmitterSetResult = this.workspaceViewModel.AddEmitterSet(this.emitterSetViewModel);
            if (addEmitterSetResult == false)
            {
                return false;
            }

            // デフォルト値を更新
            WorkspaceRootViewModel.Instance.UpdateViewModelDescriptors(this.emitterSetViewModel);

            // データ変更フラグをクリア。
            this.emitterSetViewModel.ModificationFlagViewModel.ClearModificationFlags();
            this.emitterSetViewModel.ModificationFlagViewModel.ClearChildModificationFlags();

            this.SwapLastToIndex();

            // 作成したエミッタセットを選択状態に.
            var node = this.emitterSetViewModel as WorkspaceNodeViewModelBase;
            if (node != null)
            {
                node.IsSelected = true;
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            // エミッターセットノードを選択している場合、ワークスペースノードを選択する.
            // (削除したはずのエミッターセットページが操作できてしまうことを防ぐため)
            // この操作は、ノードを削除する前に実行する必要がある.
            if (this.previousSelectedNode != null)
            {
                var node = this.emitterSetViewModel as WorkspaceNodeViewModelBase;
                if (node != null && node.IsSelected)
                {
                    this.previousSelectedNode.IsSelected = true;
                }
            }

            // ノードの変更を元に戻す.
            bool removeResult = this.workspaceViewModel.RemoveEmitterSet(this.emitterSetViewModel);
            if (removeResult == false)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            // ノードの変更をやり直す
            bool addResult = this.workspaceViewModel.AddEmitterSet(this.emitterSetViewModel);
            if (addResult == false)
            {
                return false;
            }

            this.SwapLastToIndex();
            OptionStore.TriggerOptionChangedEvent(null);

            return true;
        }

        /// <summary>
        /// 挿入先が指定されている場合、末尾から挿入先にフィールドを移動します。
        /// </summary>
        private void SwapLastToIndex()
        {
            if (this.insertIndex != -1)
            {
                var typeOrdered = this.workspaceViewModel.Children
                    as TypeOrderedObservableCollection<IHierarchyObject>;

                if (typeOrdered != null)
                {
                    typeOrdered.SuspendTypeOrdering();
                }

                int sourceIndex = this.workspaceViewModel.Children.IndexOf(this.emitterSetViewModel);

                this.workspaceViewModel.Children.RemoveAt(sourceIndex);
                this.workspaceViewModel.Children.Insert(this.insertIndex + 1, this.emitterSetViewModel);

                this.workspaceViewModel.Proxy.MoveChildDataModel(
                    this.emitterSetViewModel.DataModel,
                    this.insertIndex + 1 - sourceIndex);

                if (typeOrdered != null)
                {
                    typeOrdered.ResumeTypeOrdering();
                }
            }
        }
    }
}
