﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Command;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// using構文のスコープ内で発行されたSetPropertyコマンドを連結します。
    /// </summary>
    public class CommandCombiner : IDisposable
    {
        /// <summary>
        /// コマンド連結のネスト数をカウント
        /// </summary>
        private static int nestCounter = 0;

        /// <summary>
        /// コマンド開始時のUndoスタックの一番上
        /// </summary>
        private readonly CommandBase startCommand = null;

        /// <summary>
        /// コマンドスタック
        /// </summary>
        private readonly CommandStack commandStack = null;

        /// <summary>
        /// コンストラクタで開始時点のコマンドを記憶
        /// </summary>
        public CommandCombiner()
        {
            ++nestCounter;
            if (nestCounter > 1)
            {
                return;
            }

            this.commandStack = CommandManager.GetActiveStack();
            this.startCommand = this.GetLastCommand();
        }

        /// <summary>
        /// ディスポーザーで積まれた履歴をチェックし、最後の履歴の以外を連続実行にする
        /// </summary>
        public virtual void Dispose()
        {
            --nestCounter;
            if (nestCounter > 0)
            {
                return;
            }

            if (this.commandStack == null)
            {
                return;
            }

            var lastCommand = this.GetLastCommand();
            if (this.startCommand != lastCommand)
            {
                foreach (var command in this.commandStack.UndoBuffer)
                {
                    if (command == lastCommand)
                    {
                        continue;
                    }

                    if (command == this.startCommand)
                    {
                        break;
                    }

                    command.KeepExecuting = true;
                }

                CommandManager.UpdateCommandHistoryView();
            }
        }

        /// <summary>
        /// 最後に積まれたコマンドを取得します。
        /// </summary>
        /// <returns>最後に積まれたコマンド</returns>
        private CommandBase GetLastCommand()
        {
            return this.commandStack != null ? this.commandStack.GetNextUndoCommand() : null;
        }
    }
}
