﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// A Command that can add and remove a node to a parent.
    /// </summary>
    [Alias("AddHierarchicalNode")]
    public class AddHierarchicalNodeCommand : CommandBase
    {
        /// <summary>
        /// Stores the command stack target.
        /// </summary>
        private object commandStackTarget;

        /// <summary>
        /// Stores the parent node.
        /// </summary>
        private HierarchyViewModel parent;

        /// <summary>
        /// Stores the child node to add to the parent.
        /// </summary>
        private HierarchyViewModel node;

        /// <summary>
        /// プレビューの場合はフラグ更新をブロックする
        /// </summary>
        private bool isPreview = false;

        /// <summary>以前選択していたエミッタの処理タイプ</summary>
        private int previousUsingCpu = -1;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public AddHierarchicalNodeCommand()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">親ノード</param>
        /// <param name="node">子ノード</param>
        public AddHierarchicalNodeCommand(HierarchyViewModel parent, HierarchyViewModel node)
        {
            object[] args = new object[] { parent, node };

            this.Initialize(args);
        }

        /// <summary>
        /// Initializes the command.
        /// </summary>
        /// <param name="args">Command arguments, two HierarchyViewModel instances.</param>
        /// <returns>Returns true on success, false otherwise.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 2)
            {
                return false;
            }

            this.parent = args[0] as HierarchyViewModel;
            if (this.parent == null)
            {
                return false;
            }

            this.node = args[1] as HierarchyViewModel;
            if (this.node == null)
            {
                return false;
            }

            this.commandStackTarget =
                CommandBase.FindCommandStackTargetFromHierarchyObject(this.parent);

            this.isPreview = this.node is PreviewViewModel;

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// Gets the command target.
        /// </summary>
        /// <returns>Returns the command target.</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// Adds the child to the parent.
        /// </summary>
        /// <returns>Returns true or throw an exception.</returns>
        public override bool Do()
        {
            var proxy = this.parent.Proxy as DataModelProxy;
            proxy.AddChildDataModel(this.node.Proxy.DataModel);

            if (this.isPreview)
            {
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
                this.parent.Children.Add(this.node);
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;

                var esetVM = this.node.Parent as EmitterSetViewModel;
                if (esetVM != null)
                {
                    ViewerMessageHelper.SendEmitterSet(esetVM.DataModel);
                }
            }
            else
            {
                this.parent.Children.Add(this.node);
                var emitter = this.parent as EmitterViewModel;
                if (emitter != null)
                {
                    this.previousUsingCpu =
                        emitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu();
                    WorkspaceRootViewModel.Instance.UpdateUIStates();
                }
            }

            return true;
        }

        /// <summary>
        /// Removes the child from the parent.
        /// </summary>
        /// <returns>Returns true or throw an exception.</returns>
        public override bool Undo()
        {
            var nodeVM = this.parent as WorkspaceNodeViewModelBase;
            if (nodeVM != null)
            {
                nodeVM.IsSelected = true;
            }

            DataModelProxy proxy = this.parent.Proxy;
            proxy.RemoveChildDataModel(this.node.Proxy.DataModel);

            if (this.isPreview)
            {
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = true;
                this.parent.Children.Remove(this.node);
                ModificationFlagViewModel.IgnoreParentPropertyChangedEvents = false;

                var esetVM = this.node.Parent as EmitterSetViewModel;
                if (esetVM != null)
                {
                    ViewerMessageHelper.SendEmitterSet(esetVM.DataModel);
                }
            }
            else
            {
                this.parent.Children.Remove(this.node);
                var emitter = this.parent as EmitterViewModel;
                if (emitter != null)
                {
                    emitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.RaiseCanUsingGpu(this.previousUsingCpu);
                    WorkspaceRootViewModel.Instance.UpdateUIStates();
                }
            }

            return true;
        }

        /// <summary>
        /// Redo時に復帰したノードを選択する。
        /// </summary>
        /// <returns>Redoが成功したか否か</returns>
        public override bool Redo()
        {
            var result = base.Redo();
            OptionStore.TriggerOptionChangedEvent(null);

            var nodeVM = this.node as WorkspaceNodeViewModelBase;
            if (nodeVM != null)
            {
                nodeVM.IsSelected = true;
            }

            return result;
        }
    }
}
