﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Texture;
using EffectMaker.TextureManager.Cache;
using EffectMaker.TextureManager.Loader;

namespace EffectMaker.TextureManager
{
    /// <summary>
    /// テクスチャマネージャ
    /// </summary>
    public class TextureManager
    {
        /// <summary>
        /// テクスチャデータキャッシュ
        /// </summary>
        private readonly TextureDataCache textureDataCache = new TextureDataCache();

        /// <summary>
        /// テクスチャローダー
        /// </summary>
        private readonly ITextureLoader[] textureLoaders = { new FtxTextureLoader() };

        /// <summary>
        /// ファイルリロード後イベント
        /// </summary>
        public event EventHandler<FileReloadedEventArgs> FileReloaded;

        /// <summary>
        /// DLLロードのベースパスを渡しつつ初期化
        /// </summary>
        /// <param name="basePaths">ベースパスのリスト</param>
        public static void InitializeOnStartUp(string[] basePaths)
        {
            foreach (var path in basePaths)
            {
                TexUtilsProxy.AddBasePath(path);
            }

            TexUtilsProxy.SafeInitialize();
        }

        /// <summary>
        /// ファイルが読み込めるか調査する
        /// </summary>
        /// <param name="filePath">調査するファイルパス</param>
        /// <returns>読み込めるかどうか？</returns>
        public bool CanLoad(string filePath)
        {
            if (string.IsNullOrEmpty(filePath))
            {
                return false;
            }

            if (File.Exists(filePath) == false)
            {
                return false;
            }

            return this.textureLoaders.Any(x => x.CanLoad(filePath));
        }

        /// <summary>
        /// キャッシュされているか
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>キャッシュされているか.</returns>
        public bool IsCached(string filePath)
        {
            return this.textureDataCache.Get(filePath) != null;
        }

        /// <summary>
        /// キャッシュをフラッシュする
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        public void FlushCache(string filePath)
        {
            this.textureDataCache.Remove(filePath);
        }

        /// <summary>
        /// リロードする
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        public void Reload(string filePath)
        {
            if (this.IsCached(filePath) == false)
            {
                return;
            }

            this.FlushCache(filePath);
            this.LoadTexture(filePath, true);

            if (this.FileReloaded != null)
            {
                var args = new FileReloadedEventArgs
                {
                    FilePath = filePath
                };

                this.FileReloaded(this, args);
            }
        }

        /// <summary>
        /// キャッシュサイズの最大値を設定します。
        /// </summary>
        /// <param name="maxCapacity">キャッシュサイズの最大値(Byte単位)</param>
        public void SetMaxCapacity(int maxCapacity)
        {
            this.textureDataCache.SetMaxCapacity(maxCapacity);
        }

        /// <summary>
        /// テクスチャファイルを読み込みます。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="enableCache">キャッシュからロードするかどうか</param>
        /// <returns>読み込み結果を返します。</returns>
        public LoadTextureResult LoadTexture(string filePath, bool enableCache)
        {
            LoadTextureResult result;

            if (this.CanLoad(filePath) == false)
            {
                result = new LoadTextureResult(LoadTextureResultCode.FailedLoadingTexture);
            }
            else
            {
                if (enableCache)
                {
                    var textureData = this.textureDataCache.Get(filePath);

                    // キャッシュ済みでも日付が違っていたら読み直す
                    if (textureData != null)
                    {
                        if (textureData.UpdateTimestamp != File.GetLastWriteTime(filePath))
                        {
                            this.textureDataCache.Remove(filePath);
                            textureData = null;
                        }
                    }

                    if (textureData != null)
                    {
                        result = new LoadTextureResult
                        {
                            ResultCode = LoadTextureResultCode.Success,
                            TextureData = textureData.Data
                        };
                    }
                    else
                    {
                        // 新規に読み込む
                        result = this.LoadTexture(filePath);

                        // 成功時キャッシュに収める
                        if (result.ResultCode == LoadTextureResultCode.Success)
                        {
                            var data = new TextureDataCache.TextureFileData
                            {
                                Data = result.TextureData,
                                UpdateTimestamp = File.GetLastWriteTime(filePath)
                            };

                            this.textureDataCache.Add(filePath, data);
                        }
                    }
                }
                else
                {
                    // 新規に読み込む
                    result = this.LoadTexture(filePath);
                }
            }

            Debug.Assert(result != null, "不正な状態。要調査");

            return result;
        }

        /// <summary>
        /// テクスチャファイルを読み込みます。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>読み込み結果を返します。</returns>
        public LoadTextureResult LoadTexture(string filePath)
        {
            // ローダを探す
            var textureLoader = this.textureLoaders.FirstOrDefault(x => x.CanLoad(filePath));

            // ローダが見つからないので失敗を返す
            if (textureLoader == null)
            {
                return new LoadTextureResult(LoadTextureResultCode.UnknowTextureType);
            }

            // 実際に読み込んで結果を返す
            return textureLoader.Load(filePath);
        }

        /// <summary>
        /// 事前JITする
        /// </summary>
        /// <param name="textureData">テクスチャデータ</param>
        /// <param name="extName">テクスチャファイル拡張子</param>
        public void PreJit(byte[] textureData, string extName)
        {
            var filePath = Path.GetTempFileName() + extName;

            try
            {
                using (var file = new FileStream(filePath, FileMode.Create, FileAccess.Write))
                {
                    file.Write(textureData, 0, textureData.Length);
                }

                // 空読みする
                this.LoadTexture(filePath);
            }
            finally
            {
                if (File.Exists(filePath))
                {
                    File.Delete(filePath);
                }
            }
        }

        /// <summary>
        /// 読み込んだテクスチャファイルの変更日時を取得します。
        /// </summary>
        /// <param name="filePath">テクスチャファイルパス</param>
        /// <returns>
        /// 読み込んだテクスチャファイルの変更日時を返します。
        /// テクスチャファイルが読み込まれていないときはデフォルト値を返します。
        /// </returns>
        public DateTime GetLoadedFileModifyTime(string filePath)
        {
            var cacheData = this.textureDataCache.Get(filePath);

            if (cacheData == null)
            {
                return new DateTime();
            }

            return cacheData.UpdateTimestamp;
        }
    }
}
