﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Runtime.Remoting;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Channels.Ipc;

namespace EffectMaker.ProcessBridge
{
    /// <summary>
    /// EffectMakerBridge通信サーバーです。
    /// </summary>
    public class BridgeServer
    {
        /// <summary>
        /// ファイナライザです。
        /// </summary>
        ~BridgeServer()
        {
            Debug.Assert(this.serverChannel == null, "未破棄");
            Debug.Assert(this.remoteObject == null, "未破棄");
        }

        /// <summary>
        /// コマンド実行デリゲートです。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="remoteObject">リモートオブジェクト</param>
        public delegate void ProcessDelegate(object sender, BridgeRemoteObject remoteObject);

        /// <summary>
        /// リモートからコマンド実行要求があったとき発生します。
        /// </summary>
        public ProcessDelegate Process { get; set; }

        /// <summary>
        /// 通信サーバーを初期化します。
        /// </summary>
        /// <returns>初期化に成功したときtrue、それ以外はfalseを返します。</returns>
        public bool Initialize()
        {
            Debug.Assert(this.serverChannel == null, "初期化済み");

            // サーバーチャンネルを作成
            try
            {
                this.serverChannel = new IpcServerChannel(BridgeRemoteObject.IpcChannelName);

                ChannelServices.RegisterChannel(this.serverChannel, true);
            }
            catch (RemotingException)
            {
                this.Release();
                return false;
            }

            // リモートオブジェクトを作成
            {
                this.remoteObject = new BridgeRemoteObject();
                this.remoteObject.Process += this.OnProcess;

                RemotingServices.Marshal(this.remoteObject, BridgeRemoteObject.IpcUrl, typeof(BridgeRemoteObject));
            }

            return true;
        }

        /// <summary>
        /// 通信サーバーを破棄します。
        /// </summary>
        public void Release()
        {
            if (this.remoteObject != null)
            {
                this.remoteObject.Process -= this.OnProcess;
                this.remoteObject = null;
            }

            if (this.serverChannel != null)
            {
                ChannelServices.UnregisterChannel(this.serverChannel);
                this.serverChannel = null;
            }
        }

        /// <summary>
        /// コマンド実行イベントを処理します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnProcess(object sender, EventArgs e)
        {
            if (this.Process != null)
            {
                this.Process(this, this.remoteObject);
            }
        }

        /// <summary>
        /// 通信ポート名
        /// </summary>
        public const string BridgePortName = "EffectMakerBridge";

        /// <summary>
        /// サーバーチャンネル
        /// </summary>
        private IpcServerChannel serverChannel;

        /// <summary>
        /// リモートオブジェクト
        /// </summary>
        private BridgeRemoteObject remoteObject;
    }
}
