﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.BinaryConversionInfo;
using CompileShaderAssemblyCallback = System.Action<EffectMaker.DataModelLogic.ShaderCompileResult, string, string, string, string>;
using CompileShaderCompleteCallback = System.Action<System.Collections.Generic.List<long>, System.Action<bool>, EffectMaker.Foundation.Debugging.Profiling.ProfileTimer, EffectMaker.Foundation.Debugging.Profiling.ProfileTimer, bool, bool, byte[], long, int, long, int, System.Action<System.IO.Stream>>;

namespace EffectMaker.DataModelLogic.Utilities
{
    /// <summary>
    /// A helper class for compiling shader binary.
    /// </summary>
    public class ShaderBinaryHelper
    {
        /// <summary>
        /// マネージャのアセンブリ
        /// </summary>
        private static Assembly ManagerAssembly = null;

        /// <summary>
        /// シェーダーマネージャ
        /// </summary>
        private readonly IShaderManager shaderManager = null;

        /// <summary>
        /// OpenGLのコンテキスト生成用にMainFormのハンドルを登録する
        /// </summary>
        public static IntPtr Handle { get; set; }

        /// <summary>
        /// Default constructor.
        /// </summary>
        public ShaderBinaryHelper()
        {
            // シェーダーマネージャのパスが空ではない定義を読み込む
            var spec = SpecManager.CurrentSpec;
            if (spec != null && !string.IsNullOrEmpty(spec.ShaderConversionOption.ManagerPath))
            {
                ManagerAssembly = Assembly.LoadFrom(Path.Combine(IOConstants.CoreModulesPath, spec.ShaderConversionOption.ManagerPath));
            }
            else
            {
                ManagerAssembly = null;
            }

            if (ManagerAssembly != null && spec != null)
            {
                var type = ManagerAssembly.GetType(spec.ShaderConversionOption.ManagerTypeName);
                this.shaderManager = Activator.CreateInstance(type) as IShaderManager;
            }
            else
            {
                // ひとまず、genericのdllを読み込んでしまう.
                ManagerAssembly = Assembly.LoadFrom(Path.Combine(IOConstants.CoreModulesPath, "EffectMaker.ShaderManagerGeneric.dll"));
                var type = ManagerAssembly.GetType("EffectMaker.SpecGeneric.Shader.ShaderManager");
                this.shaderManager = Activator.CreateInstance(type) as IShaderManager;
            }
        }

        /// <summary>
        /// ユーザー定義文字列の何番目を利用するかを設定・取得
        /// </summary>
        public uint UserDefineIndex
        {
            get { return this.shaderManager.UserDefineIndex; }
            set { this.shaderManager.UserDefineIndex = value; }
        }

        /// <summary>
        /// エミッタデータに対応するシェーダコードを生成します。
        /// </summary>
        /// <param name="emitterData">エミッタデータ</param>
        /// <param name="vertexShader">バーテックスシェーダのソースコード</param>
        /// <param name="fragmentShader">フラグメントシェーダのソースコード</param>
        /// <param name="computeShader">コンピュートシェーダのソースコード</param>
        /// <returns>処理が成功したときはtrue、失敗したときはfalseを返します。</returns>
        public bool GenerateShaderSource(
            EmitterData emitterData,
            out string vertexShader,
            out string fragmentShader,
            out string computeShader)
        {
            // コード生成前にシェーダキーをリフレッシュするためエミッタバイナリをコンバートし直す
            {
                var emitterBinary = GetEmitterBinaryData(emitterData);

                if (emitterBinary == null)
                {
                    vertexShader = null;
                    fragmentShader = null;
                    computeShader = null;

                    return false;
                }

                emitterBinary.ConvertBinary();
            }

            return this.shaderManager.GenerateShaderSource(emitterData, out vertexShader, out fragmentShader, out computeShader);
        }

        /// <summary>
        /// エミッタデータに対応するシェーダアセンブリを生成します。
        /// </summary>
        /// <param name="emitter">エミッタデータ</param>
        /// <param name="completeCallback">The callback function when completed.</param>
        public void GenrateShaderAssemblyAsync(
            EmitterData emitterData,
            CompileShaderAssemblyCallback completeCallback)
        {
            // アセンブリ生成前にシェーダキーをリフレッシュするためエミッタバイナリをコンバートし直す
            {
                var emitterBinary = GetEmitterBinaryData(emitterData);

                if (emitterBinary == null)
                {
                    return;
                }

                emitterBinary.ConvertBinary();
            }

            this.shaderManager.GenrateShaderAssemblyAsync(emitterData, completeCallback);
        }

        /// <summary>
        /// Write shader binary data.
        /// </summary>
        /// <param name="binaryName">バイナリー名</param>
        /// <param name="stream">バイナリーストリーム</param>
        /// <param name="enableAsync">True to compile shader asynchronously.</param>
        /// <param name="completeCallback">The callback to execute when finished.</param>
        /// <param name="emitterContexts">エミッタのバイナリーコンテキスト</param>
        /// <param name="handleShaderCompileComplete"></param>
        /// <returns>True on success.</returns>
        public bool WriteShaderResourcesImpl(
            string binaryName,
            Stream stream,
            string shaderBinaryOutputFilePath,
            string computeShaderBinaryOutputFilePath,
            bool enableAsync,
            Action<bool> completeCallback,
            IEnumerable<WriteBinaryDataContext> emitterContexts,
            CompileShaderCompleteCallback handleShaderCompileComplete)
        {
            return this.shaderManager.WriteShaderResourcesImpl(
                binaryName,
                stream,
                shaderBinaryOutputFilePath,
                computeShaderBinaryOutputFilePath,
                enableAsync,
                completeCallback,
                emitterContexts,
                handleShaderCompileComplete);
        }

        /// <summary>
        /// エミッタデータモデルから対応するバイナリーインスタンスを取得します。
        /// </summary>
        /// <param name="emitter">エミッタデータモデル</param>
        /// <returns>対応するバイナリーインスタンス</returns>
        private static BinaryData.BinaryStructInstance GetEmitterBinaryData(EmitterData emitter)
        {
            var dataModel = (DataModelBase)emitter;
            var binaryData = BinaryConversionInfoManager.FindBinaryData(dataModel);
            while (binaryData == null && dataModel.Parent != null)
            {
                dataModel = dataModel.Parent;
                binaryData = BinaryConversionInfoManager.FindBinaryData(dataModel);
            }

            return binaryData;
        }
    }

    /// <summary>
    /// シェーダマネージャのインタフェースです。
    /// </summary>
    public interface IShaderManager
    {
        /// <summary>
        /// ユーザー定義文字列の何番目を利用するかを設定・取得
        /// </summary>
        uint UserDefineIndex { get; set; }

        /// <summary>
        /// Generate shader source code in the emitter.
        /// </summary>
        /// <param name="emitter">The emitter.</param>
        /// <param name="vertexShader">The generated vertex shader.</param>
        /// <param name="fragmentShader">The generated fragment shader.</param>
        /// <param name="computeShader">The generated compute shader.</param>
        /// <returns>True on success.</returns>
        bool GenerateShaderSource(
            EmitterData emitter,
            out string vertexShader,
            out string fragmentShader,
            out string computeShader);

        /// <summary>
        /// Compile shader in the emitter and generate shader assembly.
        /// </summary>
        /// <param name="emitter">The emitter.</param>
        /// <param name="completeCallback">The callback function when completed.</param>
        void GenrateShaderAssemblyAsync(
            EmitterData emitter,
            CompileShaderAssemblyCallback completeCallback);

        /// <summary>
        /// Write shader binary data.
        /// </summary>
        /// <param name="binaryName">バイナリー名</param>
        /// <param name="stream">バイナリーストリーム</param>
        /// <param name="enableAsync">True to compile shader asynchronously.</param>
        /// <param name="completeCallback">The callback to execute when finished.</param>
        /// <param name="emitterContexts">エミッタのバイナリーコンテキスト</param>
        /// <param name="handleShaderCompileComplete"></param>
        /// <returns>True on success.</returns>
        bool WriteShaderResourcesImpl(
            string binaryName,
            Stream stream,
            string shaderBinaryOutputFilePath,
            string computeShaderBinaryOutputFilePath,
            bool enableAsync,
            Action<bool> completeCallback,
            IEnumerable<WriteBinaryDataContext> emitterContexts,
            CompileShaderCompleteCallback handleShaderCompileComplete);
    }
}
