﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

namespace EffectMaker.DataModelLogic.DataModelProxies
{
    /// <summary>
    /// A specific DataModelProxy that manage workspace related functionalities.
    /// </summary>
    public class WorkspaceDataProxy : DataModelProxy
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="dataModel">The data model for the proxy.</param>
        public WorkspaceDataProxy(DataModelBase dataModel) :
            base(dataModel)
        {
        }

        /// <summary>
        /// Add the given data model to the children data models.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        public override void AddChildDataModel(DataModelBase dataModel)
        {
            if (dataModel is EmitterSetData)
            {
                this.AddEmitterSet((EmitterSetData)dataModel);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Add the given data model to the children data models.
        /// </summary>
        /// <param name="index">The index of the data model.</param>
        /// <param name="dataModel">The data model.</param>
        public override void InsertChildDataModel(int index, DataModelBase dataModel)
        {
            if (dataModel is EmitterSetData)
            {
                this.InsertEmitterSet(index, (EmitterSetData)dataModel);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Remove the specified data model from the children data models.
        /// </summary>
        /// <param name="dataModel">The dat model to be removed.</param>
        public override void RemoveChildDataModel(DataModelBase dataModel)
        {
            if (dataModel is EmitterSetData)
            {
                this.RemoveEmitterSet((EmitterSetData)dataModel);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Move the specified data model to the specified index.
        /// </summary>
        /// <param name="dataModel">The data model to be moved.</param>
        /// <param name="distance">
        /// The distance to move the child.
        /// E.g. 2 means move down 2 items, -3 means move up 3 items.
        /// </param>
        public override void MoveChildDataModel(DataModelBase dataModel, int distance)
        {
            if (dataModel is EmitterSetData)
            {
                this.MoveEmitterSet((EmitterSetData)dataModel, distance);
            }
            else
            {
                throw new ArgumentException();
            }
        }

        /// <summary>
        /// Create a new emitter as a child of the emitter set.
        /// </summary>
        /// <param name="name">The name of the emitter set.</param>
        /// <returns>The created emitter set.</returns>
        public EmitterSetData CreateEmitterSet(string name)
        {
            var ownerData = this.DataModel as WorkspaceData;
            if (ownerData == null)
            {
                return null;
            }

            // Create the emitter.
            EmitterSetData emitterSetData = DataModelCreator.CreateEmitterSetDataModel();
            emitterSetData.Name = name;

            return emitterSetData;
        }

        /// <summary>
        /// Add the given emitter set data to the workspace.
        /// </summary>
        /// <param name="emitterSetData">The emitter set data.</param>
        public void AddEmitterSet(EmitterSetData emitterSetData)
        {
            var workspaceData = this.DataModel as WorkspaceData;
            if (workspaceData == null)
            {
                return;
            }

            // Check if the emitter set has already been added.
            if (workspaceData.EmitterSetList.IndexOf(emitterSetData) >= 0)
            {
                return;
            }

            // Add the emitter set to the workspace.
            workspaceData.EmitterSetList.Add(emitterSetData);
            workspaceData.AddChild(emitterSetData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterSetList");
        }

        /// <summary>
        /// Add the given emitter set data to the workspace.
        /// </summary>
        /// <param name="index">The index of the emitter set.</param>
        /// <param name="emitterSetData">The emitter set data.</param>
        public void InsertEmitterSet(int index, EmitterSetData emitterSetData)
        {
            var workspaceData = this.DataModel as WorkspaceData;
            if (workspaceData == null)
            {
                return;
            }

            // Check if the emitter set has already been added.
            if (workspaceData.EmitterSetList.IndexOf(emitterSetData) >= 0)
            {
                return;
            }

            // Add the emitter set to the workspace.
            workspaceData.EmitterSetList.Insert(index, emitterSetData);
            workspaceData.InsertChild(index, emitterSetData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterSetList");
        }

        /// <summary>
        /// Remove the specified emitter set from the workspace.
        /// </summary>
        /// <param name="emitterSetData">The emitter set to be removed.</param>
        public void RemoveEmitterSet(EmitterSetData emitterSetData)
        {
            var workspaceData = this.DataModel as WorkspaceData;
            if (workspaceData == null)
            {
                return;
            }

            // Remove the emitter set from the workspace.
            workspaceData.EmitterSetList.Remove(emitterSetData);
            workspaceData.RemoveChild(emitterSetData);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterSetList");
        }

        /// <summary>
        /// Move the specified child to the specified index.
        /// </summary>
        /// <param name="child">The child to be moved.</param>
        /// <param name="distance">
        /// The distance to move the child.
        /// E.g. 2 means move down 2 items, -3 means move up 3 items.
        /// </param>
        public void MoveEmitterSet(EmitterSetData child, int distance)
        {
            var ownerData = this.DataModel as WorkspaceData;
            if (ownerData == null)
            {
                return;
            }

            // Find the original index.
            int originalIndex = ownerData.EmitterSetList.IndexOf(child);
            int newIndex = originalIndex + distance;

            // Remove the emitter set from the workspace.
            ownerData.EmitterSetList.Remove(child);

            // Insert the emitter set back to the specified index.
            ownerData.EmitterSetList.Insert(newIndex, child);

            // Fire event.
            this.TriggerPropertyModifiedEvent("EmitterSetList");
        }
    }
}
