﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Events;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Extensions;

namespace EffectMaker.DataModelLogic.DataModelProxies
{
    /// <summary>
    /// Data model proxy for emitter custom shader data.
    /// </summary>
    public class EmitterCustomShaderDataProxy : DataModelProxy
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="dataModel">The data model for the proxy.</param>
        public EmitterCustomShaderDataProxy(DataModelBase dataModel) :
            base(dataModel)
        {
            DataModelProxy.PropertyModified += this.OnDataModelPropertyModified;
        }

        /// <summary>
        /// Disposes the instance.
        /// </summary>
        public override void Dispose()
        {
            DataModelProxy.PropertyModified -= this.OnDataModelPropertyModified;
            base.Dispose();
        }

        /// <summary>
        /// Update user data models.
        /// </summary>
        public void UpdateUserData()
        {
            var dataModel = this.DataModel as EmitterCustomShaderData;

            // Create user data models.
            IEnumerable<UserDataBase> userDataModels =
                UserDataManager.CreateUserDataForOwner(typeof(EmitterCustomShaderData));

            // Add these created user pages.
            var customShaderDataList = new ArrayCollection<EmitterCustomShaderSettingData>(8);

            int index = 0;
            foreach (UserDataBase dm in userDataModels)
            {
                var customShaderData = dm as EmitterCustomShaderSettingData;
                if (customShaderData != null)
                {
                    customShaderDataList[index] = customShaderData;
                    ++index;
                }
            }

            // Set the created user data models to our data model.
            dataModel.Settings = customShaderDataList;

            // Fire event.
            this.TriggerPropertyModifiedEvent("Settings");
        }

        /// <summary>
        /// Handle PropertyModified event from data model proxy.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDataModelPropertyModified(
            object sender,
            DataModelPropertyModifiedEventArgs e)
        {
            var myDataModel = this.DataModel as EmitterCustomShaderData;
            if (myDataModel == null)
            {
                return;
            }

            if (e.DataModel == myDataModel.SelectedSettingData &&
                e.PropertyName == "Flags")
            {
                this.TriggerPropertyModifiedEvent("SelectedSettingFlags");
            }

            if (e.DataModel == myDataModel.SelectedSettingData &&
                e.PropertyName.Contains("Switch"))
            {
                this.TriggerPropertyModifiedEvent("SelectedSettingSwitches");
            }
        }
    }
}
