﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.DataModelLogic.Utilities;

namespace EffectMaker.DataModelLogic.BinaryData
{
    /// <summary>
    /// Interface for all kinds of binary element instances.
    /// </summary>
    public interface IBinaryElementInstance : IDisposable
    {
        /// <summary>
        /// Get the parent binary element.
        /// </summary>
        IBinaryElementInstance Parent { get; }

        /// <summary>
        /// Get the tag of the binary element.
        /// </summary>
        string Tag { get; }

        /// <summary>
        /// Get the local offset of the binary data from the beginning of its parent.
        /// (including binary structure headers if any)
        /// </summary>
        int LocalOffset { get; set; }

        /// <summary>
        /// Get the offset of the binary data in the root binary structure.
        /// </summary>
        int Offset { get; set; }

        /// <summary>
        /// Get the size of the raw binary data.
        /// </summary>
        int Size { get; }

        /// <summary>
        /// Get or set the size of the binary data including the binary headers if any.
        /// </summary>
        int SizeIncludeHeader { get; }

        /// <summary>
        /// Called by children when a child element has been created.
        /// </summary>
        /// <param name="reportingElement">The child element that is created.</param>
        void ReportChildElementCreated(IBinaryElementInstance reportingElement);

        /// <summary>
        /// Called by children when a child element has been removed.
        /// </summary>
        /// <param name="reportingElement">The child element that is removed.</param>
        void ReportChildElementRemoved(IBinaryElementInstance reportingElement);

        /// <summary>
        /// Find the root binary element of this instance.
        /// </summary>
        /// <returns>The root binary element.</returns>
        IBinaryElementInstance FindRootElement();

        /// <summary>
        /// Find all the children of this instance that has the specified tag.
        /// </summary>
        /// <param name="tag">The tag.</param>
        /// <returns>The binary element instances that has the tag.</returns>
        IEnumerable<IBinaryElementInstance> FindChildrenWithTag(string tag);

        /// <summary>
        /// Find owner binary element instance of the source data model.
        /// </summary>
        /// <param name="dataModel">The source data model.</param>
        /// <returns>The binary element instance that holds the source data model.</returns>
        IBinaryElementInstance FindSourceDataModelOwner(DataModelBase dataModel);

        /// <summary>
        /// Called by a child element to report that it's size has changed
        /// and the offset needs to be recalculated.
        /// </summary>
        /// <param name="reportingElement">The reporting child element.</param>
        void ReportSizeChanged(IBinaryElementInstance reportingElement);

        /// <summary>
        /// Convert to raw binary data.
        /// </summary>
        /// <returns>True on success.</returns>
        bool ConvertBinary();

        /// <summary>
        /// Write the binary data for the given session.
        /// </summary>
        /// <param name="session">The WriteBinaryDataSession that is being processed now.</param>
        /// <returns>True on success.</returns>
        bool WriteBinaryData(WriteBinaryDataSession session);
    }
}
