﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.BinaryConverters;
using EffectMaker.DataModelLogic.Utilities;

using EffectMaker.Foundation.Collections;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.BinaryData
{
    /// <summary>
    /// Binary data field instance created with the binary field definition.
    /// </summary>
    public class BinaryFieldInstance : IBinaryElementInstance
    {
        /// <summary>The binary data field definition.</summary>
        private BinaryFieldDefinition definition;

        /// <summary>The binary data structure instances.</summary>
        private List<BinaryStructInstance> structInstances = new List<BinaryStructInstance>();

        /// <summary>The converted binary data.</summary>
        private byte[] binaryData = null;

        /// <summary>The binary size.</summary>
        private int binarySize = 0;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="parent">The parent binary data structure instance.</param>
        /// <param name="definition">The binary data field definition.</param>
        public BinaryFieldInstance(
            BinaryStructInstance parent,
            BinaryFieldDefinition definition)
        {
            this.Parent = parent;
            this.definition = definition;

            this.ReportChildElementCreated(this);

            // Register the handling for the data model properties.
            foreach (InputPropertyDefinition def in this.definition.InputProperties)
            {
                BinaryConversionInfoManager.RegisterBinaryFieldInstance(
                    this,
                    def.Guid);
            }
        }

        /// <summary>
        /// Get the parent binary element.
        /// </summary>
        public IBinaryElementInstance Parent { get; private set; }

        /// <summary>
        /// Get the binary field definition.
        /// </summary>
        public BinaryFieldDefinition Definition
        {
            get { return this.definition; }
        }

        /// <summary>
        /// Get the tag of the binary element.
        /// </summary>
        public string Tag
        {
            get
            {
                if (this.definition == null || this.definition.Tag == null)
                {
                    return string.Empty;
                }

                return this.definition.Tag;
            }
        }

        /// <summary>
        /// Get the local offset of the binary data from the beginning of its parent.
        /// (including binary structure headers if any)
        /// </summary>
        public int LocalOffset { get; set; }

        /// <summary>
        /// Get the offset of the binary data in the root binary structure.
        /// </summary>
        public int Offset { get; set; }

        /// <summary>
        /// Get the size of the binary data.
        /// </summary>
        public virtual int Size
        {
            get
            {
                return this.binarySize;
            }

            protected set
            {
                this.binarySize = value;
                this.SizeIncludeHeader = value;
            }
        }

        /// <summary>
        /// Get or set the size of the binary data including the binary headers if any.
        /// </summary>
        public virtual int SizeIncludeHeader { get; protected set; }

        /// <summary>
        /// Get the flag indicating whether this field is currently converting binary data.
        /// </summary>
        public bool IsConvertingBinary { get; protected set; }

        /// <summary>
        /// Enumerate the underlying binary structure instances.
        /// </summary>
        protected IEnumerable<BinaryStructInstance> UnderlyingBinaryStructures
        {
            get { return this.structInstances; }
        }

        /// <summary>
        /// Dispose.
        /// </summary>
        public virtual void Dispose()
        {
            // Unregister the handling for the data model properties.
            foreach (InputPropertyDefinition def in this.definition.InputProperties)
            {
                BinaryConversionInfoManager.UnregisterBinaryFieldInstance(
                    this,
                    def.Guid);
            }

            // Dispose the binary structure instances.
            this.ClearUnderlyingBinaryStructures();

            this.ReportChildElementRemoved(this);
        }

        /// <summary>
        /// Find the root binary element of this instance.
        /// </summary>
        /// <returns>The root binary element.</returns>
        public virtual IBinaryElementInstance FindRootElement()
        {
            if (this.Parent == null)
            {
                return this;
            }
            else
            {
                return this.Parent.FindRootElement();
            }
        }

        /// <summary>
        /// Find all the children of this instance that their definition's tag matches
        /// the specified string.
        /// </summary>
        /// <param name="tag">The tag.</param>
        /// <returns>The binary element instances that has the tag.</returns>
        public IEnumerable<IBinaryElementInstance> FindChildrenWithTag(string tag)
        {
            foreach (BinaryStructInstance instance in this.structInstances)
            {
                if (string.IsNullOrEmpty(instance.Tag) == false &&
                    instance.Tag == tag)
                {
                    yield return instance;
                }

                foreach (IBinaryElementInstance child in instance.FindChildrenWithTag(tag))
                {
                    yield return child;
                }
            }
        }

        /// <summary>
        /// Find owner binary element instance of the source data model.
        /// </summary>
        /// <param name="dataModel">The source data model.</param>
        /// <returns>The binary element instance that holds the source data model.</returns>
        public IBinaryElementInstance FindSourceDataModelOwner(DataModelBase dataModel)
        {
            if (this.Parent != null)
            {
                return this.Parent.FindSourceDataModelOwner(dataModel);
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Check if the the given property of the specified data model is one of the source
        /// property of this binary field instance.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <param name="propertyGuid">The Guid of the property.</param>
        /// <returns>True if the property is one of the source property.</returns>
        public virtual bool IsSourceProperty(DataModelBase dataModel, Guid propertyGuid)
        {
            // Get the data model the parent binary structure instance is handling.
            var parentStruct = this.Parent as BinaryStructInstance;
            if (parentStruct == null)
            {
                return false;
            }

            DataModelBase myDataModel = parentStruct.DataModel;

            // Check if the data model in the arguments is a child of our handling data model.
            bool shouldHandle = false;
            DataModelBase currDataModel = dataModel;
            while (currDataModel != null)
            {
                if (currDataModel.Guid == myDataModel.Guid)
                {
                    if (object.ReferenceEquals(currDataModel, myDataModel) == true)
                    {
                        shouldHandle = true;
                        break;
                    }
                    else
                    {
                        Logger.Log(LogLevels.Debug, "BinaryFieldInstance.IsSourceProperty : The data models has the same Guid but are different instances. There might be unnecessary clone of the data model.");
                    }
                }

                currDataModel = currDataModel.Parent;
            }

            return shouldHandle;
        }

        /// <summary>
        /// Called by children when a child element has been created.
        /// </summary>
        /// <param name="reportingElement">The child element that is created.</param>
        public virtual void ReportChildElementCreated(IBinaryElementInstance reportingElement)
        {
            if (this.Parent != null)
            {
                this.Parent.ReportChildElementCreated(reportingElement);
            }
        }

        /// <summary>
        /// Called by children when a child element has been removed.
        /// </summary>
        /// <param name="reportingElement">The child element that is removed.</param>
        public virtual void ReportChildElementRemoved(IBinaryElementInstance reportingElement)
        {
            if (this.Parent != null)
            {
                this.Parent.ReportChildElementRemoved(reportingElement);
            }
        }

        /// <summary>
        /// Called by a child element to report that it's size has changed
        /// and the offset needs to be recalculated.
        /// </summary>
        /// <param name="reportingElement">The reporting child element.</param>
        public virtual void ReportSizeChanged(IBinaryElementInstance reportingElement)
        {
            if (this.IsConvertingBinary == true)
            {
                return;
            }

            int originalSize = this.Size;

            // Update the binary offset of the array elements.
            int offset = 0;
            int offsetIncludeHeader = 0;
            foreach (BinaryStructInstance element in this.structInstances)
            {
                element.LocalOffset = offsetIncludeHeader;
                element.Offset = offset + this.Offset;

                // Increment.
                offsetIncludeHeader += element.SizeIncludeHeader;
                offset += element.Size;
            }

            this.Size = offset;
            this.SizeIncludeHeader = offsetIncludeHeader;

            if (this.Parent != null && originalSize != offset)
            {
                this.Parent.ReportSizeChanged(this);
            }
        }

        /// <summary>
        /// Convert binary data from the data model.
        /// </summary>
        /// <returns>True on success.</returns>
        public virtual bool ConvertBinary()
        {
            this.IsConvertingBinary = true;

            this.CheckFieldOffsetForDebugBreak();

            try
            {
                if ((this.definition.ConverterDefinition == null) ||
                    (this.definition.InputProperties == null) ||
                    (this.definition.InputProperties.Length <= 0))
                {
                    return false;
                }

                // Create the converter.
                var converter = this.definition.ConverterDefinition.CreateConverter();

                // The required input value count of the converter
                // should match the input property count.
                int inputValueCount = this.definition.InputProperties.Length;
                if (converter.MinInputValueCount > inputValueCount ||
                    converter.MaxInputValueCount < inputValueCount)
                {
                    Logger.Log(LogLevels.Warning, "BinaryFieldInstance.ConvertBinary : Input value count does not match the requested input count of the converter.");
                    return false;
                }

                // Feed input values to the converter.
                if (this.FeedInputValueToConverter(converter) == false)
                {
                    return false;
                }

                int originalBinarySize = this.Size;

                // Convert!!!
                byte[] bytes;
                if (converter.Convert(out bytes) == false)
                {
                    Logger.Log(LogLevels.Warning, "BinaryFieldInstance.ConvertBinary : The converter reported a failure when converting the data.");
                    return false;
                }

                object output = converter.GetOutputValue();

                // If this field contains a binary data structure,
                // assign the data model to it.
                if (this.definition.StructDefinition != null)
                {
                    if ((output is IEnumerable<DataModelBase>) == false)
                    {
                        Logger.Log(LogLevels.Warning, "BinaryFieldInstance.ConvertBinary : The output binary field is a binary data, however the assigned converter does not output binary data.");
                        return false;
                    }

                    // Clear the binary structures first.
                    this.ClearUnderlyingBinaryStructures();

                    var dataModels = output as IEnumerable<DataModelBase>;

                    // Convert all the binary data structures in the array.
                    int offset = 0;
                    int offsetIncludeHeader = 0;
                    foreach (DataModelBase dataModel in dataModels)
                    {
                        // Create the binary structure.
                        var element = this.Definition.StructDefinition.CreateInstance(this, dataModel);

                        this.AddBinaryStructureInstance(element);

                        // Setup the binary data offset of the binary structure.
                        element.LocalOffset = offsetIncludeHeader;
                        element.Offset = offset + this.Offset;

                        // Convert the element.
                        if (element.ConvertBinary() == false)
                        {
                            return false;
                        }

                        // Increment.
                        offsetIncludeHeader += element.SizeIncludeHeader;
                        offset += element.Size;
                    }

                    // Update the binary size.
                    this.Size = offset;
                    this.SizeIncludeHeader = offsetIncludeHeader;

                    if (this.Parent != null && offset != originalBinarySize)
                    {
                        // Report size change.
                        this.Parent.ReportSizeChanged(this);
                    }
                }
                else
                {
                    if (bytes == null)
                    {
                        Logger.Log(LogLevels.Warning, "BinaryFieldInstance.ConvertBinary : The converter failed generating binary data.");
                        return false;
                    }

                    // This field does not contain a binary structure,
                    // just save the converted binary data.
                    this.binaryData = bytes;
                    this.Size = this.binaryData.Length;

                    // If the binary data size has changed, report to the parent.
                    if (this.Size != originalBinarySize)
                    {
                        this.Parent.ReportSizeChanged(this);
                    }
                }

                return true;
            }
            finally
            {
                this.IsConvertingBinary = false;
            }
        }

        /// <summary>
        /// Write the binary data for the given session.
        /// </summary>
        /// <param name="session">The WriteBinaryDataSession that is being processed now.</param>
        /// <returns>True on success.</returns>
        public virtual bool WriteBinaryData(WriteBinaryDataSession session)
        {
            if (this.structInstances != null && this.structInstances.Count > 0)
            {
                using (new WriteBinaryDataBlock(session, this))
                {
                    foreach (BinaryStructInstance element in this.structInstances)
                    {
                        if (element.WriteBinaryData(session) == false)
                        {
                            return false;
                        }
                    }
                }
            }
            else if (this.binaryData != null)
            {
                using (new WriteBinaryDataBlock(session, this))
                {
                    session.Stream.Write(this.binaryData, 0, this.binaryData.Length);
                }
            }
            else if (this.structInstances == null)
            {
                Logger.Log(LogLevels.Warning, "BinaryFieldInstance.WriteBinaryData : Failed writing data to binary stream.");
                return false;
            }

            return true;
        }

        /// <summary>
        /// Add an underlying binary structure instance to the list.
        /// </summary>
        /// <param name="instance">The binary structure instance.</param>
        protected void AddBinaryStructureInstance(BinaryStructInstance instance)
        {
            this.structInstances.Add(instance);
        }

        /// <summary>
        /// Clear all the underlying binary structures.
        /// </summary>
        protected void ClearUnderlyingBinaryStructures()
        {
            // Dispose the binary structure instances.
            foreach (BinaryStructInstance structInstance in this.structInstances)
            {
                structInstance.Dispose();
            }

            // Clear the list.
            this.structInstances.Clear();
        }

        /// <summary>
        /// Check the offset of the field for debug breaking execution at the specified offset.
        /// </summary>
        protected void CheckFieldOffsetForDebugBreak()
        {
            #if DEBUG

            // Break the execution if the condition is met.
            if (BinaryFieldDefinition.EnableBinaryFieldDebug == true &&
                BinaryFieldDefinition.BreakOnFieldOffset == this.Offset)
            {
                System.Diagnostics.Debugger.Break();
            }

            #endif
        }

        /// <summary>
        /// Feed input values to the converter.
        /// </summary>
        /// <param name="converter">The converter.</param>
        /// <returns>True on success.</returns>
        protected bool FeedInputValueToConverter(ConverterBase converter)
        {
            BinaryStructInstance parentStruct = this.Parent as BinaryStructInstance;
            if (parentStruct == null)
            {
                Logger.Log(LogLevels.Warning, "BinaryFieldInstance.FeedInputValueToConverter : Failed converting binary data because the binary field does not have parent binary structure.");
                return false;
            }

            object value;

            int index = 0;
            foreach (InputPropertyDefinition def in this.Definition.InputProperties)
            {
                // Get the property value with the Guid.
                DataModelBase currDataModel = parentStruct.DataModel;
                while (currDataModel.GetPropertyValueWithPath(def.Path, out value) == false)
                {
                    // Try to get the property from parent data model.
                    currDataModel = currDataModel.Parent;
                    if (currDataModel == null)
                    {
                        Logger.Log(LogLevels.Warning, "BinaryFieldInstance.FeedInputValueToConverter : Unable to find the input data model property.");
                        return false;
                    }
                }

                #if DEBUG

                // Break the execution if the condition is met.
                if (BinaryFieldDefinition.EnableBinaryFieldDebug == true &&
                    BinaryFieldDefinition.BreakOnInputPropertyGuid == def.Guid)
                {
                    System.Diagnostics.Debugger.Break();
                }

                #endif

                // Set the property value to the converter as input.
                converter.SetInputValue(index, value);

                ++index;
            }

            return true;
        }
    }
}
