﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Base abstract class for data model value converter.
    /// </summary>
    public abstract class ConverterBase
    {
        /// <summary>The input values.</summary>
        private object[] inputValues = null;

        /// <summary>The flags indicating whether the input values are assigned.</summary>
        private bool[] inputValuesAssignedFlags = null;

        /// <summary>The output value.</summary>
        private object outputValue = null;

        /// <summary>The converter parameter holder.</summary>
        private ConverterParamHolder parameterHolder = null;

        /// <summary>
        /// 文字列として保持したいパラメータの名前リスト
        /// </summary>
        private List<string> stringParameterNames = new List<string>();

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="parameterHolder">The converter parameter holder.</param>
        public ConverterBase(ConverterParamHolder parameterHolder)
        {
            this.parameterHolder = parameterHolder;
            this.inputValues = new object[this.MaxInputValueCount];
            this.inputValuesAssignedFlags = new bool[this.MaxInputValueCount];
            for (int i = 0; i < this.inputValues.Length; ++i)
            {
                this.inputValues[i] = null;
                this.inputValuesAssignedFlags[i] = false;
            }
        }

        /// <summary>
        /// Get the minimum number of input values.
        /// This value is the same as the maximum input value count by default.
        /// </summary>
        public virtual int MinInputValueCount
        {
            get { return this.MaxInputValueCount; }
        }

        /// <summary>
        /// Get the number of input values.
        /// </summary>
        public abstract int MaxInputValueCount { get; }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public abstract string Description { get; }

        /// <summary>
        /// Enumerate the descriptions of the input values.
        /// </summary>
        public IEnumerable<string> InputValueDescriptions
        {
            get
            {
                for (int i = 0; i < this.MaxInputValueCount; ++i)
                {
                    yield return this.GetInputValueDescription(i);
                }
            }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public abstract string OutputValueDescription { get; }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public abstract string GetInputValueDescription(int index);

        /// <summary>
        /// 指定したパラメータ名が文字列か否かを返します。
        /// </summary>
        /// <param name="name">パラメータ名</param>
        /// <returns>文字列ならtrue,そうでなければfalse.</returns>
        public bool IsStringParameter(string name)
        {
            return this.stringParameterNames.Contains(name);
        }

        /// <summary>
        /// Enumerate the names of the parameters accepted by the converter.
        /// </summary>
        /// <returns>The parameter names.</returns>
        public IEnumerable<string> EnumerateParameterNames()
        {
            if (this.parameterHolder == null)
            {
                return Enumerable.Empty<string>();
            }

            return this.parameterHolder.EnumerateParameterNames();
        }

        /// <summary>
        /// Enumerate the name and description of the parameters accepted by the converter.
        /// </summary>
        /// <returns>The name and description of the parameters.</returns>
        public IEnumerable<KeyValuePair<string, string>> EnumerateParameterNameAndDesc()
        {
            if (this.parameterHolder == null)
            {
                return Enumerable.Empty<KeyValuePair<string, string>>();
            }

            return this.parameterHolder.EnumerateParameterNameAndDesc();
        }

        /// <summary>
        /// Check if the parameter is valid.
        /// </summary>
        /// <param name="name">The name of the parameter.</param>
        /// <param name="parameter">The value of the parameter.</param>
        /// <returns>False if the given parameter value is invalid, or the parameter name does not exist.</returns>
        public bool ValidateParameter(string name, object parameter)
        {
            if (this.parameterHolder == null)
            {
                return false;
            }

            return this.parameterHolder.ValidateParameter(name, parameter);
        }

        /// <summary>
        /// Set parameter for the converter.
        /// </summary>
        /// <param name="name">The name of the parameter.</param>
        /// <param name="parameter">The value of the parameter.</param>
        /// <returns>True on success.</returns>
        public bool SetParameter(string name, object parameter)
        {
            if (this.parameterHolder == null)
            {
                return false;
            }

            return this.parameterHolder.SetParameter(name, parameter);
        }

        /// <summary>
        /// Get the parameter value of the designated name.
        /// </summary>
        /// <param name="name">The parameter name.</param>
        /// <param name="value">The parameter value.</param>
        /// <returns>False if the parameter is not found or set.</returns>
        public bool GetParameter(string name, out object value)
        {
            value = null;
            if (this.parameterHolder == null)
            {
                return false;
            }

            return this.parameterHolder.GetParameter(name, out value);
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public abstract bool Convert();

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <param name="binaryOutput">The binary output data.</param>
        /// <returns>True on success.</returns>
        public virtual bool Convert(out byte[] binaryOutput)
        {
            if (this.Convert() == false)
            {
                binaryOutput = null;
                return false;
            }

            var output = this.GetOutputValue();

            if (output is AnimationTableData)
            {
                var tmpList = new List<Primitivef>();
                foreach (var keyFrame in output as AnimationTableData)
                {
                    tmpList.Add(keyFrame.Value);
                }

                binaryOutput = BinaryConversionUtility.ForResource.Convert(tmpList);
            }
            else if (output is KeyFrameData)
            {
                var kfd = output as KeyFrameData;
                binaryOutput = BinaryConversionUtility.ForResource.Convert(kfd.Value);
            }
            else
            {
                binaryOutput = BinaryConversionUtility.ForResource.Convert(output);
            }

            Debug.Assert(binaryOutput != null, "The output value cannot be converted to binary data.");

            return true;
        }

        /// <summary>
        /// Set input value for the conversion.
        /// </summary>
        /// <param name="index">The index of the value.</param>
        /// <param name="value">The value.</param>
        public void SetInputValue(int index, object value)
        {
            if ((index < 0) || (index >= this.MaxInputValueCount))
            {
                Debug.Assert(false, "The assigned index is out of range.");
                return;
            }

            this.inputValues[index] = value;
            this.inputValuesAssignedFlags[index] = true;
        }

        /// <summary>
        /// Get output value of the conversion.
        /// </summary>
        /// <returns>The output value.</returns>
        public object GetOutputValue()
        {
            return this.outputValue;
        }

        /// <summary>
        /// Reset this converter.
        /// </summary>
        public void Reset()
        {
            for (int i = 0; i < this.inputValues.Length; ++i)
            {
                this.inputValues[i] = null;
                this.inputValuesAssignedFlags[i] = false;
            }

            this.outputValue = null;
        }

        /// <summary>
        /// 文字列として扱うパラメータ名を追加します。
        /// </summary>
        /// <param name="name">パラメータ名</param>
        protected void AddStringParameterName(string name)
        {
            if (!this.stringParameterNames.Contains(name))
            {
                this.stringParameterNames.Add(name);
            }
        }

        /// <summary>
        /// 文字列として扱うパラメータ名のリストをクリアします。
        /// </summary>
        protected void ClearStringParameterNames()
        {
            this.stringParameterNames.Clear();
        }

        /// <summary>
        /// Get the count of assigned input values.
        /// </summary>
        /// <returns>The assigned input value count.</returns>
        protected int CountAssignedInputValues()
        {
            return this.inputValuesAssignedFlags.Count(item => item == true);
        }

        /// <summary>
        /// Get the flag indicating whether the input value at the given index is assigned or not.
        /// </summary>
        /// <param name="index">The index to the value.</param>
        /// <returns>True if the input value is assigned.</returns>
        protected bool IsInputValueAssigned(int index)
        {
            if ((index < 0) || (index >= this.MaxInputValueCount))
            {
                Debug.Assert(false, "The assigned index is out of range.");
                return false;
            }

            return this.inputValuesAssignedFlags[index];
        }

        /// <summary>
        /// Get input value.
        /// </summary>
        /// <param name="index">The index of the value.</param>
        /// <returns>The input value.</returns>
        protected object GetInputValue(int index)
        {
            if ((index < 0) || (index >= this.MaxInputValueCount))
            {
                Debug.Assert(false, "The assigned index is out of range.");
                return null;
            }

            return this.inputValues[index];
        }

        /// <summary>
        /// Set output value of the conversion.
        /// </summary>
        /// <param name="value">The value.</param>
        protected void SetOutputValue(object value)
        {
            this.outputValue = value;
        }
    }
}
