﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections;
using System.Collections.Generic;
using System.Linq;

using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.Attributes;
using EffectMaker.DataModel.RandomColor;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Convert animation tables into an array of their key frame count.
    /// </summary>
    public class ColorAnimationTableCountConverter : ConverterBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public ColorAnimationTableCountConverter()
            : base(null)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 3; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.AnimationTableCountConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.AnimationTableCountConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.AnimationTableCountConverterInputValueDesc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            var type = this.GetInputValue(0) as int?;
            if (type == null)
            {
                Logger.Log(LogLevels.Warning, "ColorAnimationTableConverter.Convert : The input value is invalid.");
                return false;
            }

            int outputCount = 0;
            switch (type.GetValueOrDefault())
            {
                case 2:
                    var animTable = this.GetInputValue(1) as AnimationTableData;
                    if (animTable == null)
                    {
                        return false;
                    }

                    outputCount = animTable.Count;
                    break;

                case 3:
                    var randTable = this.GetInputValue(2) as RandomColorTable;
                    if (randTable == null)
                    {
                        return false;
                    }

                    for (int i = 0; i < randTable.Count; ++i)
                    {
                        if (randTable[i].Enabled)
                        {
                            outputCount++;
                        }
                    }

                    break;

                default:
                    outputCount = 0;
                    break;
            }

            this.SetOutputValue(outputCount);

            return true;
        }
    }
}
