﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.Communicator;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.ViewerMessages
{
    /// <summary>
    /// The base class for all the viewer messages.
    /// </summary>
    public abstract class MessageBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public MessageBase()
        {
            this.ReceivedMessageSize = 0;
            this.Status = MessageStatus.ReadyToSend;
            this.MessageDataBuffer = null;
        }

        /// <summary>
        /// Get the flag indicating if should send message to the viewer.
        /// </summary>
        /// <remarks>
        /// This property provides a shortcut for determine the connection
        /// status, so that hides communication layer away from outside.
        /// Besides, if user need to override this flag for debug purpose,
        /// this property can also be easily modified.
        /// </remarks>
        public static bool ShouldSendMessage
        {
            get
            {
                return MessageManager.Instance.IsConnected;
            }
        }

        /// <summary>
        /// Get the type of the message.
        /// </summary>
        public abstract MessageTypes MessageType { get; }

        /// <summary>
        /// Get the size of the message.
        /// (used while sending the message.)
        /// </summary>
        public abstract int MessageSize { get; }

        /// <summary>
        /// Get or set the size of the message received from viewer.
        /// </summary>
        public virtual int ReceivedMessageSize { get; protected set; }

        /// <summary>
        /// Get or set the status of the message.
        /// </summary>
        public virtual MessageStatus Status { get; protected set; }

        /// <summary>
        /// MessageManagerにメッセージを追加するのを取り消すかどうか。
        /// このフラグをtrueにしてもメッセージを追加する直前までの処理は行われます。
        /// </summary>
        public virtual bool ShouldDropMessage { get;  set; }

        /// <summary>
        /// Get or set the buffer for the written data of the message.
        /// This buffer is only used when the message is enqueued in SendMessageQueue.
        /// </summary>
        protected byte[] MessageDataBuffer { get; set; }

        /// <summary>
        /// Set the status of the message to ready.
        /// Note that if Send() is called, the message will still be sent
        /// either the status is ready or not.
        /// The status only affect the message when it is queuing up in SendMessageQueue.
        /// </summary>
        public virtual void SetReady()
        {
            this.Status = MessageStatus.ReadyToSend;
            SendMessageQueue.NotifyMessageStatusChanged(this);
        }

        /// <summary>
        /// Cancel this message so the message will not be sent.
        /// Note that if Send() is called, the message will still be sent
        /// no matter what the status is.
        /// The status only affect the message when it is queuing up in SendMessageQueue.
        /// </summary>
        public virtual void CancelMessage()
        {
            this.Status = MessageStatus.Canceled;
            SendMessageQueue.NotifyMessageStatusChanged(this);
        }

        /// <summary>
        /// Write the data to binary, enqueue the message to the SendMessageQueue
        /// and wait to be sent.
        /// </summary>
        /// <returns>True on success.</returns>
        public virtual bool Send()
        {
            using (var stream = new MemoryStream())
            {
                using (var writer = new MessageWriter(stream))
                {
                    // Write message to the memory stream.
                    if (this.WriteMessage(writer) == false)
                    {
                        return false;
                    }

                    // Save the data to our buffer.
                    // The data will be sent when the message is ready and
                    // all previous message are sent in SendMessageQueue.
                    this.MessageDataBuffer = new byte[stream.Length];
                    Array.Copy(stream.GetBuffer(), this.MessageDataBuffer, (int)stream.Length);
                }
            }

            // The message will be push to the SendMessageQueue first,
            // wait until all previous messages are sent out, then if
            // this message is ready to be sent, send to the viewer.
            SendMessageQueue.Enqueue(this);

            return true;
        }

        /// <summary>
        /// Send the message which is previously enqueued in SendMessageQueue.
        /// </summary>
        public virtual void SendEnqueuedMessage()
        {
            if (this.MessageDataBuffer == null)
            {
                throw new InvalidOperationException("The message has no data to be sent.");
            }

            using (var stream = new MemoryStream())
            {
                stream.Write(this.MessageDataBuffer, 0, this.MessageDataBuffer.Length);
                using (var writer = new MessageWriter(stream))
                {
                    // Write message to the memory stream.
                    this.WriteMessageBeforeSent(writer);

                    // Create the message the message manager uses.
                    // (This message is different from MessageBase.)
                    var message = new Message()
                    {
                        Buffer = stream.GetBuffer(),
                        Size = (int)stream.Length,
                    };

                    // Push the message to the queue.
                    if (this.ShouldDropMessage == false)
                    {
                        MessageManager.Instance.PushMessage(message);
                    }
                }
            }
        }

        /// <summary>
        /// Read message data from binary data.
        /// </summary>
        /// <param name="stream">The stream contains the message's binary data.</param>
        /// <returns>True on success.</returns>
        public virtual bool Read(Stream stream)
        {
            var type = (MessageTypes)BinaryConversionUtility.ForProtocol.ReadStream<uint>(stream);
            if (type != this.MessageType)
            {
                return false;
            }

            this.ReceivedMessageSize = BinaryConversionUtility.ForProtocol.ReadStream<int>(stream);

            return true;
        }

        /// <summary>
        /// Write message data to a stream with the given writer.
        /// This happens before the message is enqueued.
        /// If data is not ready at this moment, WriteMessageBeforeSent() should
        /// be used to write the rest of the message data.
        /// </summary>
        /// <param name="writer">The message writer.</param>
        /// <returns>True on success.</returns>
        protected virtual bool WriteMessage(MessageWriter writer)
        {
            writer.Write(BinaryConversionUtility.ForProtocol.Convert((uint)this.MessageType));
            writer.Write(BinaryConversionUtility.ForProtocol.Convert(this.MessageSize));

            return true;
        }

        /// <summary>
        /// Write the rest of the message data before the message is actually sent.
        /// </summary>
        /// <param name="writer">The message writer.</param>
        /// <returns>True on success.</returns>
        protected virtual bool WriteMessageBeforeSent(MessageWriter writer)
        {
            return true;
        }
    }
}
