﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.Synchronization
{
    /// <summary>
    /// EffectMaker のプロセステーブルです。
    /// </summary>
    public class ProcessTable
    {
        /// <summary>
        /// 現在のプロセスのロックファイルストリームです。
        /// </summary>
        private static FileStream lockFileStream;

        /// <summary>
        /// TableDirectoryPath のバッキングフィールドです。
        /// </summary>
        private static string tableDirectoryPath;

        /// <summary>
        /// プロセステーブルのフォルダパスを取得します。
        /// </summary>
        public static string TableDirectoryPath
        {
            get
            {
                if (string.IsNullOrEmpty(tableDirectoryPath))
                {
                    tableDirectoryPath = Path.Combine(IOConstants.AppDataTempPath, "ProcessTable");
                }

                return tableDirectoryPath;
            }
        }

        /// <summary>
        /// LockFilePath のバッキングフィールドです。
        /// </summary>
        private static string lockFilePath;

        /// <summary>
        /// 現在のプロセスのロックファイルパスを取得します。
        /// </summary>
        public static string LockFilePath
        {
            get
            {
                if (string.IsNullOrEmpty(lockFilePath))
                {
                    string processId = Process.GetCurrentProcess().Id.ToString();
                    lockFilePath = Path.Combine(tableDirectoryPath, processId + ".lock");
                }

                return lockFilePath;
            }
        }

        /// <summary>
        /// 自身のプロセスをテーブルに追加します。
        /// </summary>
        public static void AddMyProcess()
        {
            using (new GlobalSyncSection())
            {
                IOUtility.SafeCreateDirectory(TableDirectoryPath);

                // ロックファイルを作成
                try
                {
                    File.WriteAllText(LockFilePath, IOConstants.ExecutablePath);
                    lockFileStream = new FileStream(LockFilePath, FileMode.Open, FileAccess.Read);
                }
                catch (Exception e)
                {
                    Logger.LogException("Console", LogLevels.Warning, e);
                }
            }
        }

        /// <summary>
        /// 自身のプロセスをテーブルから削除します。
        /// </summary>
        public static void DeleteMyProcess()
        {
            using (new GlobalSyncSection())
            {
                // ロックファイルを閉じる
                if (lockFileStream != null)
                {
                    lockFileStream.Close();
                    lockFileStream = null;
                }

                // ロックファイルを削除する
                // 削除し損ねたファイルがあればそれも削除する
                IOUtility.DeleteExpiredFiles(TableDirectoryPath, TimeSpan.Zero);

                // プロセステーブルフォルダが空になったら削除する
                try
                {
                    if (Directory.EnumerateFiles(TableDirectoryPath).Any() == false)
                    {
                        Directory.Delete(TableDirectoryPath);
                    }
                }
                catch
                {
                }
            }
        }

        /// <summary>
        /// プロセス一覧を取得します。
        /// </summary>
        /// <returns>プロセス一覧を返します。</returns>
        public static List<ProcessInfo> GetProcesses()
        {
            List<ProcessInfo> processes = new List<ProcessInfo>();

            using (new GlobalSyncSection())
            {
                // ロックファイルを列挙してプロセス一覧を取得する
                try
                {
                    foreach (string path in Directory.EnumerateFiles(TableDirectoryPath))
                    {
                        int processId;
                        string processFilePath;

                        // ファイル名からプロセス ID を取得
                        string fileName = Path.GetFileNameWithoutExtension(path);
                        int.TryParse(fileName, out processId);

                        // ファイルの内容からプロセスファイルのパスを取得
                        try
                        {
                            using (FileStream fileStream = new FileStream(path, FileMode.Open, FileAccess.Read))
                            using (StreamReader streamReader = new StreamReader(fileStream))
                            {
                                processFilePath = streamReader.ReadToEnd();
                            }
                        }
                        catch
                        {
                            processFilePath = path;
                        }

                        ProcessInfo processInfo = new ProcessInfo()
                        {
                            ProcessId = processId,
                            FilePath = processFilePath
                        };

                        processes.Add(processInfo);
                    }
                }
                catch
                {
                }
            }

            return processes;
        }

        /// <summary>
        /// プロセス情報です。
        /// </summary>
        public class ProcessInfo
        {
            /// <summary>
            /// プロセス ID を取得または設定します。
            /// </summary>
            public int ProcessId { get; set; }

            /// <summary>
            /// ファイルパスを取得または設定します。
            /// </summary>
            public string FilePath { get; set; }
        }
    }
}
