﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Communicator;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.SpecDefinitions
{
    /// <summary>
    /// 動作ターゲットの定義です。
    /// </summary>
    [Serializable]
    [TypeConverter(typeof(ExpandableObjectConverter))]
    public class SpecDefinition : IXmlDocSerializable
    {
        /// <summary>
        /// 動作ターゲット名です。
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// プラットフォームを指定する際のエイリアスです。
        /// </summary>
        public List<string> AliasList { get; set; }

        /// <summary>
        /// バイナリヘッダ名です。
        /// </summary>
        public string BinaryHeader { get; set; }

        /// <summary>
        /// CPUのエンディアンです(0:Little/1:Big)
        /// </summary>
        public int CpuEndian { get; set; }

        /// <summary>
        /// GPUのエンディアンです(0:Little/1:Big)
        /// </summary>
        public int GpuEndian { get; set; }

        /// <summary>
        /// ターゲットとするグラフィクスAPIです。
        /// </summary>
        public string GraphicsApi { get; set; }

        /// <summary>
        /// CPUエミッタを強制するか否かを取得または設定します。
        /// </summary>
        public bool ForceCpuCalc { get; set; }

        /// <summary>
        /// ターゲットのアドレスサイズです。32bitなら4、64bitなら8になります。
        /// </summary>
        public int AddressSize { get; set; }

        /// <summary>
        /// 接続情報です。
        /// </summary>
        public List<ConnectionContext> Connections { get; set; }

        /// <summary>
        /// PCビューアのパスです。
        /// </summary>
        public string PCViewerPath { get; set; }

        /// <summary>
        /// 実機ビューアのパスです。
        /// </summary>
        public string TargetViewerPath { get; set; }

        /// <summary>
        /// テクスチャパッキングオプションです。
        /// </summary>
        public TexturePackingOption TexturePackingOption { get; set; }

        /// <summary>
        /// 有効なネイティブフォーマットを示すタグ
        /// </summary>
        public List<string> ValidNativeTextureFormats { get; set; }

        /// <summary>
        /// テクスチャのネイティブフォーマットを強制するか否か
        /// </summary>
        public bool ForceNativeTexture { get; set; }

        /// <summary>
        /// シェーダコンバートオプションです。
        /// </summary>
        public ShaderConversionOption ShaderConversionOption { get; set; }

        /// <summary>
        /// ポリゴンの表裏を設定します。
        /// </summary>
        public string FrontFace { get; set; }

        /// <summary>
        /// スペック指定の推奨名(コマンドラインヘルプで表示する)を取得します。
        /// </summary>
        public string RecommendName
        {
            get
            {
                // 64ビット版のVFXB向けスペックをApiTypeでの指定名にする
                if (this.BinaryHeader == "VFXB" && this.AddressSize == 8)
                {
                    return this.GraphicsApi;
                }

                // TODO:CafeをGX2で拾い上げるための暫定対応
                if (this.BinaryHeader == "EFTB" && this.GraphicsApi == "GX2")
                {
                    return this.GraphicsApi;
                }

                if (this.AliasList != null && this.AliasList.Any())
                {
                    return this.AliasList.First();
                }

                return this.Name;
            }
        }

        /// <summary>
        /// 引数の文字列がこのスペック定義を示しているかを判定します。
        /// </summary>
        /// <param name="name">スペックを指定する文字列</param>
        /// <returns>名前かエイリアスで該当するものがあればtrue,該当がなければfalse.</returns>
        public bool IsSpecified(string name)
        {
            if (string.Equals(this.RecommendName, name, StringComparison.CurrentCultureIgnoreCase))
            {
                return true;
            }

            if (string.Equals(this.Name, name, StringComparison.CurrentCultureIgnoreCase))
            {
                return true;
            }

            return this.AliasList != null &&
                this.AliasList.Any(alias => string.Equals(alias, name, StringComparison.CurrentCultureIgnoreCase));
        }

        /// <summary>
        /// 有効なコードタイプかどうかを判定します。
        /// </summary>
        /// <param name="name">コードタイプ名</param>
        /// <returns>このスペックで有効なコードタイプならtrue,そうでなければfalse.</returns>
        public bool IsValidCodeType(string name)
        {
            return this.ShaderConversionOption.ValidCodeTypes != null &&
                this.ShaderConversionOption.ValidCodeTypes.Any(
                codeType => string.Equals(codeType, name, StringComparison.CurrentCultureIgnoreCase));
        }

        /// <summary>
        /// 有効なタイリングモードかどうかを判定します。
        /// </summary>
        /// <param name="name">タイリングモード名</param>
        /// <returns>このスペックで有効なタイリングモードならtrue,そうでなければfalse.</returns>
        public bool IsValidTileMode(string name)
        {
            return this.TexturePackingOption != null &&
                this.TexturePackingOption.ValidTileModes != null &&
                this.TexturePackingOption.ValidTileModes.Any(
                tileMode => string.Equals(tileMode, name, StringComparison.CurrentCultureIgnoreCase));
        }

        /// <summary>
        /// 有効なタイリング最適化モードかどうかを判定します。
        /// </summary>
        /// <param name="name">タイリング最適化モード名</param>
        /// <returns>このスペックで有効なタイリング最適化モードならtrue,そうでなければfalse</returns>
        public bool IsValidTileOptimize(string name)
        {
            return this.TexturePackingOption != null &&
                this.TexturePackingOption.ValidTileOptimizes != null &&
                this.TexturePackingOption.ValidTileOptimizes.Any(
                tileOptimize => string.Equals(tileOptimize, name, StringComparison.CurrentCultureIgnoreCase));
        }

        /// <summary>
        /// 有効なタイリングサイズ最適化のしきい値かどうかを判定します。
        /// </summary>
        /// <param name="threshold">タイリングサイズ最適化のしきい値</param>
        /// <returns>このスペックで有効なタイリングサイズ最適化のしきい値ならtrue,そうでなければfalse</returns>
        public bool IsValidTileSizeThreshold(int threshold)
        {
            if (string.Equals(this.TexturePackingOption.TileOptimize, "size_auto", StringComparison.CurrentCultureIgnoreCase) == false)
            {
                return false;
            }

            if (threshold < 0 || threshold > 100)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.Name = this.ReadElement(context, "Name", this.Name);
            this.AliasList = this.ReadListElement(context, "AliasList", this.AliasList);
            this.BinaryHeader = this.ReadElement(context, "BinaryHeader", this.BinaryHeader);
            this.CpuEndian = this.ReadElement(context, "CpuEndian", this.CpuEndian);
            this.GpuEndian = this.ReadElement(context, "GpuEndian", this.GpuEndian);
            this.GraphicsApi = this.ReadElement(context, "GraphicsApi", this.GraphicsApi);
            this.ForceCpuCalc = this.ReadElement(context, "ForceCpuCalc", this.ForceCpuCalc);
            this.AddressSize = this.ReadElement(context, "AddressSize", this.AddressSize);
            this.Connections = this.ReadListElement(context, "Connections", this.Connections);
            this.PCViewerPath = this.ReadElement(context, "PCViewerPath", this.PCViewerPath);
            this.TargetViewerPath = this.ReadElement(context, "TargetViewerPath", this.TargetViewerPath);
            this.TexturePackingOption = this.ReadElement(context, "TexturePackingOption", this.TexturePackingOption);
            this.ValidNativeTextureFormats = this.ReadListElement(context, "ValidNativeTextureFormats", this.ValidNativeTextureFormats);
            this.ForceNativeTexture = this.ReadElement(context, "ForceNativeTexture", this.ForceNativeTexture);
            this.ShaderConversionOption = this.ReadElement(context, "ShaderConversionOption", this.ShaderConversionOption);
            this.FrontFace = this.ReadElement(context, "FrontFace", this.FrontFace);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "Name", this.Name);
            this.WriteEnumerableElement(context, "AliasList", this.AliasList, "Alias");
            this.WriteElement(context, "BinaryHeader", this.BinaryHeader);
            this.WriteElement(context, "CpuEndian", this.CpuEndian);
            this.WriteElement(context, "GpuEndian", this.GpuEndian);
            this.WriteElement(context, "GraphicsApi", this.GraphicsApi);
            this.WriteElement(context, "ForceCpuCalc", this.ForceCpuCalc);
            this.WriteElement(context, "AddressSize", this.AddressSize);
            this.WriteEnumerableElement(context, "Connections", this.Connections, "Context");
            this.WriteElement(context, "PCViewerPath", this.PCViewerPath);
            this.WriteElement(context, "TargetViewerPath", this.TargetViewerPath);
            this.WriteElement(context, "TexturePackingOption", this.TexturePackingOption);
            this.WriteEnumerableElement(context, "ValidNativeTextureFormats", this.ValidNativeTextureFormats, "Format");
            this.WriteElement(context, "ForceNativeTexture", this.ForceNativeTexture);
            this.WriteElement(context, "ShaderConversionOption", this.ShaderConversionOption);
            this.WriteElement(context, "FrontFace", this.FrontFace);

            return true;
        }
    }

    /// <summary>
    /// シェーダコンバートオプションです。
    /// </summary>
    public class ShaderConversionOption : IXmlDocSerializable
    {
        /// <summary>
        /// シェーダーマネージャパス
        /// </summary>
        public string ManagerPath { get; set; }

        /// <summary>
        /// シェーダーマネージャクラス名
        /// </summary>
        public string ManagerTypeName { get; set; }

        /// <summary>
        /// このスペックで利用するシェーダコードのサブフォルダ名
        /// </summary>
        public string CodeFolderName { get; set; }

        /// <summary>
        /// スペック依存宣言シェーダのファイル名(PC用)
        /// </summary>
        public string SpecDeclarationGeneric { get; set; }

        /// <summary>
        /// スペック依存宣言シェーダのファイル名(実機用)
        /// </summary>
        public string SpecDeclarationTarget { get; set; }

        /// <summary>
        /// シェーダバイナリのアライメントサイズです。
        /// </summary>
        public int BinaryAlignment { get; set; }

        /// <summary>
        /// UBOのアライメントサイズです。
        /// </summary>
        public int UboAlignment { get; set; }

        /// <summary>
        /// シェーダーコード配列(SHCA)をバイナリに書き込むかどうか。
        /// </summary>
        public bool WriteCodeArray { get; set; }

        /// <summary>
        /// 実機シェーダがプリコンパイル可能かどうか
        /// </summary>
        public bool TargetIsPreCompile { get; set; }

        /// <summary>
        /// ShaderConverterを実行する際に指定するオプション
        /// </summary>
        public string ConverterOption { get; set; }

        /// <summary>
        /// ShaderConverterに指定するコードタイプ
        /// </summary>
        public string CodeType { get; set; }

        /// <summary>
        /// このスペックでShaderConverterに指定可能なコードタイプのリスト
        /// </summary>
        public List<string> ValidCodeTypes { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.ManagerPath = this.ReadElement(context, "ManagerPath", this.ManagerPath);
            this.ManagerTypeName = this.ReadElement(context, "ManagerTypeName", this.ManagerTypeName);
            this.CodeFolderName = this.ReadElement(context, "CodeFolderName", this.CodeFolderName);
            this.SpecDeclarationGeneric = this.ReadElement(context, "SpecDeclarationGeneric", this.SpecDeclarationGeneric);
            this.SpecDeclarationTarget = this.ReadElement(context, "SpecDeclarationTarget", this.SpecDeclarationTarget);

            this.BinaryAlignment = this.ReadElement(context, "BinaryAlignment", this.BinaryAlignment);
            this.UboAlignment = this.ReadElement(context, "UboAlignment", this.UboAlignment);
            this.WriteCodeArray = this.ReadElement(context, "WriteCodeArray", this.WriteCodeArray);
            this.TargetIsPreCompile = this.ReadElement(context, "TargetIsPreCompile", this.TargetIsPreCompile);

            this.ConverterOption = this.ReadElement(context, "ConverterOption", this.ConverterOption);
            this.CodeType = this.ReadElement(context, "CodeType", this.CodeType);
            this.ValidCodeTypes = this.ReadListElement(context, "ValidCodeTypes", this.ValidCodeTypes);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "ManagerPath", this.ManagerPath);
            this.WriteElement(context, "ManagerTypeName", this.ManagerTypeName);
            this.WriteElement(context, "CodeFolderName", this.CodeFolderName);
            this.WriteElement(context, "SpecDeclarationGeneric", this.SpecDeclarationGeneric);
            this.WriteElement(context, "SpecDeclarationTarget", this.SpecDeclarationTarget);

            this.WriteElement(context, "BinaryAlignment", this.BinaryAlignment);
            this.WriteElement(context, "UboAlignment", this.UboAlignment);
            this.WriteElement(context, "WriteCodeArray", this.WriteCodeArray);
            this.WriteElement(context, "TargetIsPreCompile", this.TargetIsPreCompile);

            this.WriteElement(context, "ConverterOption", this.ConverterOption);
            this.WriteElement(context, "CodeType", this.CodeType);
            this.WriteEnumerableElement(context, "ValidCodeTypes", this.ValidCodeTypes, "Type");

            return true;
        }
    }

    /// <summary>
    /// テクスチャパッキングオプションです。
    /// </summary>
    public class TexturePackingOption : IXmlDocSerializable
    {
        /// <summary>
        /// タイリングモードです。
        /// </summary>
        public string TileMode { get; set; }

        /// <summary>
        /// このスペックでサポートするタイリングモードのリストです。
        /// </summary>
        public List<string> ValidTileModes { get; set; }

        /// <summary>
        /// タイリング最適化モードです。
        /// </summary>
        public string TileOptimize { get; set; }

        /// <summary>
        /// このスペックでサポートするタイリング最適化モードのリストです。
        /// </summary>
        public List<string> ValidTileOptimizes { get; set; }

        /// <summary>
        /// タイリングサイズ最適化のしきい値です。
        /// </summary>
        public int? TileSizeThreshold { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.TileMode = this.ReadElement(context, "TileMode", this.TileMode);
            this.ValidTileModes = this.ReadListElement(context, "ValidTileModes", this.ValidTileModes);

            this.TileOptimize = this.ReadElement(context, "TileOptimize", this.TileOptimize);
            this.ValidTileOptimizes = this.ReadListElement(context, "ValidTileOptimizes", this.ValidTileOptimizes);

            this.TileSizeThreshold = this.ReadElement(context, "TileSizeThreshold", this.TileSizeThreshold);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "TileMode", this.TileMode);
            this.WriteEnumerableElement(context, "ValidTileModes", this.ValidTileModes, "Mode");

            this.WriteElement(context, "TileOptimize", this.TileOptimize);
            this.WriteEnumerableElement(context, "ValidTileOptimizes", this.ValidTileOptimizes);

            this.WriteElement(context, "TileSizeThreshold", this.TileSizeThreshold);

            return true;
        }
    }
}
