﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Linq;
using EffectMaker.BusinessLogic.Properties;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Collections.Generic;
using EffectMaker.Foundation.Log;

using BaseDeserializeResult = EffectMaker.BusinessLogic.Serializer.DataModelSerializer<EffectMaker.DataModel.Specific.DataModels.EmitterSetData>.DeserializeResult;

namespace EffectMaker.BusinessLogic.Serializer
{
    /// <summary>
    /// エミッタセットデータのシリアライザです。
    /// </summary>
    public class EmitterSetDataSerializer : DataModelSerializer<EmitterSetData>
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EmitterSetDataSerializer()
        {
        }

        /// <summary>
        /// デシリアライズを行います。
        /// </summary>
        /// <param name="value">デシリアライズする値</param>
        /// <returns>デシリアライズ結果を返します。</returns>
        public override DataModelSerializer.DeserializeResult Deserialize(Stream stream)
        {
            XDocument doc;

            // XMLデータをストリームから読み込む
            try
            {
                long origPos = stream.Position;

                doc = XDocument.Load(stream);

                stream.Position = origPos;
            }
            catch (Exception e)
            {
                DeserializeResult res = new DeserializeResult();
                res.Exception = e;

                return res;
            }

            bool modifiedXml = false;

            // XMLデータの手動書き換えを行う
            {
                Version version = GetDataModelVersion(doc);

                // 旧ストライプデータをエミッタプラグインに読み替え
                if (version < new Version("1.0.0.7"))
                {
                    modifiedXml |= this.UpdateStripeToEmitterPlugin(doc);
                }

                // カスタムアクションのコールバックなしをエミッタ拡張パラメータに読み替え
                if (version < new Version("1.0.0.7"))
                {
                    modifiedXml |= this.UpdateDefaultCustomActionToEmitterExtParams(doc);
                }

                // 旧USD/UDD形式のデータをUDD2形式に読み替え
                if (true)
                {
                    modifiedXml |= this.UpdateObsoluteUddToUdd2(doc);
                }

                // ストライプの付け方に応じて表示面の表裏を読み替え
                if (version < new Version("1.0.0.6"))
                {
                    modifiedXml |= this.FixReservedShaderDisplaySurface(doc);
                }

                // チャイルドエミッタのパーティクル追従タイプを「エミッタに追従しない」に書き換える
                if (version < new Version("1.0.0.7"))
                {
                    modifiedXml |= this.FixChildEmitterParticleFollowType(doc);
                }
            }

            BaseDeserializeResult resInterm;

            // デシリアライズ！
            if (modifiedXml)
            {
                using (Stream tmpStream = new MemoryStream())
                {
                    doc.Save(tmpStream);
                    tmpStream.Position = 0;

                    resInterm = (BaseDeserializeResult)base.Deserialize(tmpStream);
                }
            }
            else
            {
                resInterm = (BaseDeserializeResult)base.Deserialize(stream);
            }

            DeserializeResult result = new DeserializeResult(resInterm);

            // デシリアライズ失敗…
            if (result.IsSuccess == false)
            {
                return result;
            }

            EmitterSetData emitterSetData = (EmitterSetData)result.DataModel;

            // エミッタ数が上限を超えていたら警告を表示する
            {
                if (emitterSetData.EmitterList.Count > 16)
                {
                    Logger.Log("LogView,Console", LogLevels.Warning, Resources.WarningEmitterCountOverflow);
                }

                foreach (EmitterData emitter in emitterSetData.EmitterList)
                {
                    if (emitter.EmitterList.Count > 16)
                    {
                        Logger.Log("LogView,Console", LogLevels.Warning, Resources.WarningChildEmitterCountOverflow, emitter.Name);
                    }
                }
            }

            // カスタムアクションデータをユーザー定義に従って修復する
            {
                string[] customActionTypeNames = this.EnumerateCustomActionTypeNames(doc);

                bool replaced = false;

                foreach (EmitterData emitterData in emitterSetData.AllChildEmitters)
                {
                    replaced |= !this.RepairCustomActionData(emitterData, customActionTypeNames);
                }

                result.IsSelectedCustomActionDataReplaced = replaced;
            }

            // カスタムシェーダデータをユーザー定義に従って修復する
            {
                string[] customShaderTypeNames = this.EnumerateCustomShaderTypeNames(doc);

                bool replaced = false;

                foreach (EmitterData emitterData in emitterSetData.AllChildEmitters)
                {
                    replaced |= !this.RepairCustomShaderData(emitterData, customShaderTypeNames);
                }

                result.IsSelectedCustomShaderDataReplaced = replaced;
            }

            // エミッタ拡張パラメータをユーザー定義に従って修復する
            {
                string emitterExtParamsTypeName = this.GetEmitterExtParamsTypeName(doc);

                bool replaced = false;

                foreach (EmitterData emitterData in emitterSetData.AllChildEmitters)
                {
                    replaced |= !this.RepairEmitterExtParamsData(emitterData, emitterExtParamsTypeName);
                }

                result.IsEmitterExtParamsDataReplaced = replaced;
            }

            return result;
        }

        /// <summary>
        /// ファイルへのシリアライズを行います。
        /// </summary>
        /// <param name="dataModel">シリアライズする値</param>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>シリアライズ結果を返します。</returns>
        public SerializeResult SerializeToFile(EmitterSetData dataModel, string filePath)
        {
            SerializeResult result;

            try
            {
                using (FileStream stream = new FileStream(filePath, FileMode.Create, FileAccess.Write))
                {
                    result = (SerializeResult)this.Serialize(dataModel, stream);
                }
            }
            catch (Exception e)
            {
                result = new SerializeResult();

                result.Exception = e;
            }

            return result;
        }

        /// <summary>
        /// ファイルからのデシリアライズを行います。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>デシリアライズ結果を返します。</returns>
        public DeserializeResult DeserializeFromFile(string filePath)
        {
            DeserializeResult result;

            try
            {
                using (FileStream stream = new FileStream(filePath, FileMode.Open, FileAccess.Read))
                {
                    result = (DeserializeResult)this.Deserialize(stream);
                }
            }
            catch (Exception e)
            {
                result = new DeserializeResult();

                result.Exception = e;
            }

            return result;
        }

        #region Data Update Methods

        /// <summary>
        /// シリアライズされたデータモデルのバージョンを取得します。
        /// </summary>
        /// <param name="doc"処理するXMLデータ></param>
        /// <returns>データモデルのバージョンを返します。</returns>
        private Version GetDataModelVersion(XDocument doc)
        {
            XElement rootElement = doc.Root;

            Version version = new Version();

            // <EmitterSetData Version="*.*.*.*"> を取得
            if (rootElement != null)
            {
                XAttribute versionAttribute = rootElement.Attribute("Version");
                if (versionAttribute != null)
                {
                    Version.TryParse(versionAttribute.Value, out version);
                }
            }

            return version;
        }

        /// <summary>
        /// 旧StripeDataをReservedShaderとして読み込めるようにします。
        /// </summary>
        /// <param name="doc">処理するXMLデータ</param>
        /// <returns>XMLデータを更新したときはtrue、XMLデータに変更がないときはfalseを返します。</returns>
        private bool UpdateStripeToEmitterPlugin(XDocument doc)
        {
            const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            const string EmitterElementName               = "EmitterData";
            const string StripeElementName                = "StripeData";
            const string StripeUpdatedAttrName            = "UpdatedToEmitterPlugin";
            const string StripeUserDataElementName        = "UserPageData";
            const string EmitterPluginElementName         = "ReservedShader";
            const string PluginPageElementName            = "PageData";
            const string EmitterPluginUserDataElementName = "ContentsData";

            XName nilAttrName  = XName.Get("nil", XsiNamespaceUri);
            XName typeAttrName = XName.Get("type", XsiNamespaceUri);

            bool modified = false;

            // XMLデータを編集する
            foreach (XElement element in doc.Descendants(EmitterElementName))
            {
                //// We want to convert
                //// <StripeData xsi:type="StripeHistoryData">
                ////   <UserPageData xsi:type="StripeHistoryUserData">
                ////     <StripeType>0</StripeType>
                ////          ...
                ////     <DirInterpolation>1</DirInterpolation>
                ////   </UserPageData>
                //// </StripeData>
                //// into
                //// <ReservedShader>
                ////   <PageData>
                ////     <ContentsData xsi:type="RSStripeComplexUserData">
                ////       <StripeType>0</StripeType>
                ////            ...
                ////       <DirInterpolation>1</DirInterpolation>
                ////     </ContentsData>
                ////   </PageData>
                //// </ReservedShader>

                // First locate the <StripeData> element in the emitter.
                XElement stripeElement = element.Element(StripeElementName);
                if (stripeElement == null || stripeElement.IsEmpty == true)
                {
                    continue;
                }

                // Has the stripe converted to emitter plug-in?
                XAttribute convertedAttr = stripeElement.Attribute(StripeUpdatedAttrName);
                if (convertedAttr != null)
                {
                    continue;
                }

                // The <UserPageData> in the <StripeData> is the element we need,
                // and it will be renamed to <ContentsData> element then added to
                // <PageData> of <ReservedShader>.
                XElement stripeUserDataElement = stripeElement.Element(StripeUserDataElementName);
                if (stripeUserDataElement == null || stripeUserDataElement.IsEmpty == true)
                {
                    continue;
                }

                // Confirm the plug-in element is empty. If not, this file is probably
                // already converted.
                XElement pluginElement = element.Element(EmitterPluginElementName);
                if (pluginElement != null && pluginElement.IsEmpty == false)
                {
                    continue;
                }

                // Make a clone of the stripe user data element.
                var pluginUserDataElement = new XElement(stripeUserDataElement);

                // First find the type of the stripe data from the xsi:type attribute.
                var typeAttr = pluginUserDataElement.Attribute(typeAttrName);
                string originalType = typeAttr.Value;

                // Find the target type to convert to.
                UserDataInfo info =
                    ReservedShaderUserDataManager.FindUpdateTargetUserData(originalType);
                if (info == null)
                {
                    continue;
                }

                string targetType = info.DataModelType.Name;

                // Rename the element and set the type to the corresponding
                // emitter plug-in user data type.
                pluginUserDataElement.Name = EmitterPluginUserDataElementName;
                pluginUserDataElement.SetAttributeValue(typeAttrName, targetType);

                // If the emitter plug-in element already exists, remove it first.
                // We will create a new one afterward.
                if (pluginElement != null)
                {
                    pluginElement.Remove();
                }

                // Create the emitter plug-in element and add the renamed stripe data to it.
                pluginElement =
                    new XElement(
                        EmitterPluginElementName,
                        new XElement(
                            PluginPageElementName,
                            pluginUserDataElement));

                // Add the created element to the emitter.
                element.Add(pluginElement);

                modified = true;
            }

            return modified;
        }

        /// <summary>
        /// FE1でコールバックなしのカスタムアクションに保存されていたデータをエミッタ拡張パラメータに変換します。
        /// </summary>
        /// <param name="doc">処理するXMLデータ</param>
        /// <returns>XMLデータを更新したときはtrue、XMLデータに変更がないときはfalseを返します。</returns>
        private bool UpdateDefaultCustomActionToEmitterExtParams(XDocument doc)
        {
            const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            const string EmitterElementName                       = "EmitterData";
            const string CustomActionElementName                  = "CustomActionData";
            const string CustomActionDefaultSettingElementName    = "CustomActionDefaultSettingData";
            const string EmitterExtParamsElementName              = "EmitterExtParams";
            const string EmitterExtParamsSettingDataElementName   = "EmitterExtParamsSettingData";
            const string EmitterExtParamsSettingTypeAttributeName = "UDD2_EmitterExtParamsSetting";

            XName nilAttrName  = XName.Get("nil", XsiNamespaceUri);
            XName typeAttrName = XName.Get("type", XsiNamespaceUri);

            // コンバートテーブル
            var convertDictionary = new Dictionary<string, string>() { { "Flag0", "flags" } };
            for (int i = 0; i < 8; ++i)
            {
                convertDictionary.Add(string.Format("FloatParam{0}", i), String.Format("paramFloat{0}", i));
            }

            for (int i = 0; i < 6; ++i)
            {
                convertDictionary.Add(string.Format("IntParam{0}", i), String.Format("paramInt{0}", i));
            }

            bool modified = false;

            // XMLデータを編集する
            foreach (XElement element in doc.Descendants(EmitterElementName))
            {
                // <CustomAction> を探す
                XElement customActionElement = element.Element(CustomActionElementName);
                if (customActionElement == null || customActionElement.IsEmpty == true)
                {
                    continue;
                }

                // <CustomActionDefaultSetting> を探す
                XElement defaultSettingElement = customActionElement.Element(CustomActionDefaultSettingElementName);
                if (defaultSettingElement == null || defaultSettingElement.IsEmpty)
                {
                    continue;
                }

                // <EmitterExtParams> の中身が既に存在しているならコンバート済みなので何もしない
                XElement extParamsElement = element.Element(EmitterExtParamsElementName);
                if (extParamsElement != null && extParamsElement.IsEmpty == false)
                {
                    continue;
                }

                // コンバート先のエミッタ拡張パラメータのインスタンス
                var convertedExtParamsSettingElement = new XElement(EmitterExtParamsSettingDataElementName);
                convertedExtParamsSettingElement.SetAttributeValue(typeAttrName, EmitterExtParamsSettingTypeAttributeName);

                // コンバート！
                foreach (var pair in convertDictionary)
                {
                    var elem = defaultSettingElement.Element(pair.Key);
                    if (elem == null)
                    {
                        continue;
                    }

                    convertedExtParamsSettingElement.SetElementValue(pair.Value, elem.Value);
                }

                // エミッタ拡張パラメータノードをいったん削除
                if (extParamsElement != null)
                {
                    extParamsElement.Remove();
                }

                // エミッタ拡張パラメータノードを再生成
                extParamsElement = new XElement(EmitterExtParamsElementName, convertedExtParamsSettingElement);

                // エミッタに追加
                element.Add(extParamsElement);

                // 元のカスタムアクションは削除
                customActionElement.Remove();

                modified = true;
            }

            return modified;
        }

        /// <summary>
        /// 旧USD/UDD形式のデータをUDD2形式の定義で読み込めるようにします。
        /// </summary>
        /// <param name="doc">処理するXMLデータ</param>
        /// <returns>XMLデータを更新したときはtrue、XMLデータに変更がないときはfalseを返します。</returns>
        private bool UpdateObsoluteUddToUdd2(XDocument doc)
        {
            const string EmitterElementName             = "EmitterData";
            const string CustomActionElementName        = "CustomActionData";
            const string EmitterCustomShaderElementName = "EmitterCustomShaderData";
            const string SettingsElementName            = "Settings";

            bool modified = false;

            // XMLデータを編集する
            foreach (XElement element in doc.Descendants(EmitterElementName))
            {
                // <EmitterCustomShaderData> を探す
                XElement customShaderElement = element.Element(EmitterCustomShaderElementName);
                if (customShaderElement != null && customShaderElement.IsEmpty == false)
                {
                    var settingsElem = customShaderElement.Element(SettingsElementName);
                    if (settingsElem != null && settingsElem.IsEmpty == false)
                    {
                        // インデックスからUDD2対応のカスタムシェーダのデータモデル名を取得する関数
                        Func<int, string> getCustomShaderName = (index) =>
                        {
                            var def = CustomShaderUserDataManager.GetCustomShaderDefinition(index);
                            if (def == null || !def.UsingUdd2 || string.IsNullOrEmpty(def.Udd2FullFilePath))
                            {
                                return string.Empty;
                            }

                            return "UDD2_" + Path.GetFileNameWithoutExtension(def.Udd2FullFilePath);
                        };

                        modified |= this.UpdateUdd1To2Core(settingsElem, getCustomShaderName);
                    }
                }

                // <CustomActionData> を探す
                XElement customActionElement = element.Element(CustomActionElementName);
                if (customActionElement != null && customActionElement.IsEmpty == false)
                {
                    var settingsElem = customActionElement.Element(SettingsElementName);
                    if (settingsElem != null && settingsElem.IsEmpty == false)
                    {
                        // インデックスからUDD2対応のカスタムアクションのデータモデル名を取得する関数
                        Func<int, string> getCustomActionName = (index) =>
                        {
                            var def = CustomActionUserDataManager.GetCustomActionDefinition(index);
                            if (def != null)
                            {
                                if (def.UsingUdd2 && !string.IsNullOrEmpty(def.Udd2FileName))
                                {
                                    return "UDD2_" + Path.GetFileNameWithoutExtension(def.Udd2FileName);
                                }

                                // 例外的にデフォルトのカスタムアクション定義が読まれていたら、そこに流し込む形で対応する
                                string defName = Path.GetFileNameWithoutExtension(def.DataModelFileName);
                                if (defName == string.Format("CustomActionSetting{0}", index))
                                {
                                    return defName;
                                }
                            }

                            return string.Empty;
                        };

                        modified |= this.UpdateUdd1To2Core(settingsElem, getCustomActionName);
                    }
                }
            }

            return modified;
        }

        /// <summary>
        /// 旧USD/UDDからUDD2形式に変換する処理のコアです。
        /// </summary>
        /// <param name="settingsElem">XMLで記録されているセッティングのコレクションノード</param>
        /// <param name="getTargetName">インデックスからUDD2形式の定義名を得る関数</param>
        /// <returns>変換が発生したらtrue,しなければfalseを返します。</returns>
        private bool UpdateUdd1To2Core(XElement settingsElem, Func<int, string> getTargetName)
        {
            const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            XName typeAttrName = XName.Get("type", XsiNamespaceUri);

            // カスタムシェーダのフラグはそのまま読めるが、カスタムアクションのフラグはFlag0
            var convertDictionary = new Dictionary<string, string>() { { "Flag0", "flags" } };

            // 実数パラメータはカスタムシェーダでは32個、カスタムアクションでは8個
            for (int i = 0; i < 32; ++i)
            {
                convertDictionary.Add(string.Format("FloatParam{0}", i), String.Format("paramFloat{0}", i));
            }

            // 整数パラメータはカスタムシェーダにはなく、カスタムアクションでは6個
            for (int i = 0; i < 6; ++i)
            {
                convertDictionary.Add(string.Format("IntParam{0}", i), String.Format("paramInt{0}", i));
            }

            bool modified = false;
            int index = 0;

            foreach (var settingElem in settingsElem.Elements())
            {
                if (settingElem == null || settingElem.IsEmpty)
                {
                    ++index;
                    continue;
                }

                string targetName = getTargetName(index);

                if (string.IsNullOrEmpty(targetName) ||
                    settingElem.Name.ToString().StartsWith("UDD2_") ||
                    targetName == settingElem.Name)
                {
                    ++index;
                    continue;
                }

                settingElem.Name = targetName;
                settingElem.SetAttributeValue(typeAttrName, settingElem.Name);

                // コンバート！
                foreach (var pair in convertDictionary)
                {
                    var elem = settingElem.Element(pair.Key);

                    if (elem == null || elem.IsEmpty)
                    {
                        continue;
                    }

                    settingElem.SetElementValue(pair.Value, elem.Value);

                    modified = true;
                }

                ++index;
            }

            return modified;
        }

        /// <summary>
        /// ストライプの付け方に応じて表示面をフリップします。
        /// </summary>
        /// <param name="doc">処理するXMLデータ</param>
        /// <returns>XMLデータを更新したときはtrue、XMLデータに変更がないときはfalseを返します。</returns>
        private bool FixReservedShaderDisplaySurface(XDocument doc)
        {
            const string EmitterElementName             = "EmitterData";
            const string ReservedShaderElementName      = "ReservedShader";
            const string PageElementName                = "PageData";
            const string ContentsElementName            = "ContentsData";
            const string StripeComplexElementName       = "EPStripeComplexData";
            const string StripeHistoryTailElementName   = "EPStripeHistoryTailData";
            const string StripeTypeElementName          = "StripeType";
            const string EmitterBasicSettingElementName = "EmitterBasicSettingData";
            const string EmitterBasicRenderElementName  = "EmitterBasicRenderData";
            const string DisplaySurfaceElementName      = "DisplaySurface";

            const string xsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            XName typeAttrName = XName.Get("type", xsiNamespaceUri);

            bool modified = false;

            // XMLデータを編集する
            foreach (XElement element in doc.Descendants(EmitterElementName))
            {
                bool doFlip = false;

                // <ReservedShader> を探す
                XElement reservedShaderElement = element.Element(ReservedShaderElementName);
                if (reservedShaderElement != null && reservedShaderElement.IsEmpty == false)
                {
                    XElement pageElement = reservedShaderElement.Element(PageElementName);
                    if (pageElement != null && pageElement.IsEmpty == false)
                    {
                        XElement contentsElement = pageElement.Element(ContentsElementName);
                        if (contentsElement != null && contentsElement.IsEmpty == false)
                        {
                            XAttribute xsiTypeAttribute = contentsElement.Attribute(typeAttrName);
                            if (xsiTypeAttribute != null)
                            {
                                // 連結式ストライプ || (履歴式ストライプ2 && 計算方法 == ビルボードストライプ)
                                // の条件を満たすかチェックする
                                if (xsiTypeAttribute.Value == StripeComplexElementName)
                                {
                                    doFlip = true;
                                }
                                else if (xsiTypeAttribute.Value == StripeHistoryTailElementName)
                                {
                                    XElement stripeTypeElement = contentsElement.Element(StripeTypeElementName);
                                    if (stripeTypeElement != null && stripeTypeElement.IsEmpty == false)
                                    {
                                        if (stripeTypeElement.Value == "0")
                                        {
                                            doFlip = true;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                if (doFlip == false)
                {
                    continue;
                }

                // <EmitterBasicSettingData> を探す
                XElement emitterBasicSettingElement = element.Element(EmitterBasicSettingElementName);
                if (emitterBasicSettingElement != null && emitterBasicSettingElement.IsEmpty == false)
                {
                    XElement emitterBasicRenderElement = emitterBasicSettingElement.Element(EmitterBasicRenderElementName);
                    if (emitterBasicRenderElement != null && emitterBasicRenderElement.IsEmpty == false)
                    {
                        XElement displaySurfaceElement = emitterBasicRenderElement.Element(DisplaySurfaceElementName);
                        if (displaySurfaceElement != null && displaySurfaceElement.IsEmpty == false)
                        {
                            // 表示面の表裏をひっくり返す
                            if (displaySurfaceElement.Value == "1")
                            {
                                displaySurfaceElement.Value = "2";
                                modified = true;
                            }
                            else if (displaySurfaceElement.Value == "2")
                            {
                                displaySurfaceElement.Value = "1";
                                modified = true;
                            }
                        }
                    }
                }
            }

            return modified;
        }

        /// <summary>
        /// チャイルドエミッタのパーティクル追従タイプを「エミッタに追従しない」に書き換えます。
        /// </summary>
        /// <param name="doc">処理するXMLデータ</param>
        /// <returns>XMLデータを更新したときはtrue、XMLデータに変更がないときはfalseを返します。</returns>
        private bool FixChildEmitterParticleFollowType(XDocument doc)
        {
            const string EmitterListElementName             = "EmitterList";
            const string EmitterElementName                 = "EmitterData";
            const string EmitterBasicSettingElementName     = "EmitterBasicSettingData";
            const string EmitterBasicInheritanceElementName = "EmitterBasicInheritanceData";
            const string EnableEmitterParticleElementName   = "EnableEmitterParticle";
            const string EmitterBasicBasicElementName       = "EmitterBasicBasicData";
            const string ParticleFollowTypeElementName      = "ParticleFollowType";

            bool modified = false;

            // XMLデータを編集する
            XElement rootElement = doc.Root;
            if (rootElement != null && rootElement.IsEmpty == false)
            {
                XElement emitterListElement = rootElement.Element(EmitterListElementName);
                if (emitterListElement != null && emitterListElement.IsEmpty == false)
                {
                    foreach (XElement emitterElement in emitterListElement.Elements(EmitterElementName))
                    {
                        XElement childEmitterListElement = emitterElement.Element(EmitterListElementName);
                        if (childEmitterListElement != null && childEmitterListElement.IsEmpty == false)
                        {
                            foreach (XElement childEmitterElement in childEmitterListElement.Elements(EmitterElementName))
                            {
                                XElement basicSettingElement = childEmitterElement.Element(EmitterBasicSettingElementName);
                                if (basicSettingElement != null && basicSettingElement.IsEmpty == false)
                                {
                                    // パーティクルにエミッタを割り当てがOnのときはノータッチ
                                    XElement basicInheritanceElement = basicSettingElement.Element(EmitterBasicInheritanceElementName);
                                    if (basicInheritanceElement != null && basicInheritanceElement.IsEmpty == false)
                                    {
                                        XElement enableEmitterParticleElement = basicInheritanceElement.Element(EnableEmitterParticleElementName);
                                        if (enableEmitterParticleElement != null && enableEmitterParticleElement.IsEmpty == false)
                                        {
                                            if (enableEmitterParticleElement.Value == "True")
                                            {
                                                continue;
                                            }
                                        }
                                        else
                                        {
                                            // EnableEmitterParticleは途中で追加されたデータのため存在しない場合もありえる
                                            // チャイルドエミッタの場合のデフォルト値はTrueなので、このときもノータッチ
                                            continue;
                                        }
                                    }

                                    // チャイルドのエミッタパーティクル追従タイプを「エミッタに追従しない」に書き換える
                                    XElement basicBasicElement = basicSettingElement.Element(EmitterBasicBasicElementName);
                                    if (basicBasicElement != null && basicBasicElement.IsEmpty == false)
                                    {
                                        XElement particleFollowTypeElement = basicBasicElement.Element(ParticleFollowTypeElementName);
                                        if (particleFollowTypeElement != null && particleFollowTypeElement.IsEmpty == false)
                                        {
                                            particleFollowTypeElement.Value = "1";
                                            modified = true;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }

            return modified;
        }

        #endregion

        #region UDD Repair Methods

        /// <summary>
        /// シリアライズされたデータのカスタムアクションデータタイプを列挙します。
        /// カスタムアクションが未設定のときは名前にnullが入ります。
        /// </summary>
        /// <param name="doc">処理するXMLデータ</param>
        /// <returns>カスタムアクションデータタイプの名前を返します。</returns>
        private string[] EnumerateCustomActionTypeNames(XDocument doc)
        {
            const string EmitterElementName          = "EmitterData";
            const string CustomActionDataElementName = "CustomActionData";
            const string SettingsElementName         = "Settings";

            const string xsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            XName typeAttrName = XName.Get("type", xsiNamespaceUri);

            string[] customActionTypeNames = new string[8];

            bool enumerated = false;

            // カスタムアクションデータタイプの名前を列挙する
            foreach (XElement element in doc.Descendants(EmitterElementName))
            {
                // <CustomActionData> を探す
                XElement customActionElement = element.Element(CustomActionDataElementName);
                if (customActionElement != null && customActionElement.IsEmpty == false)
                {
                    var settingsElem = customActionElement.Element(SettingsElementName);
                    if (settingsElem != null && settingsElem.IsEmpty == false)
                    {
                        int index = 0;

                        foreach (XElement settingElem in settingsElem.Elements())
                        {
                            if (settingElem.IsEmpty == false)
                            {
                                XAttribute xsiTypeAttribute = settingElem.Attribute(typeAttrName);
                                if (xsiTypeAttribute != null)
                                {
                                    customActionTypeNames[index] = xsiTypeAttribute.Value;
                                }
                            }

                            ++index;
                        }

                        enumerated = true;
                    }
                }

                if (enumerated)
                {
                    break;
                }
            }

            return customActionTypeNames;
        }

        /// <summary>
        /// エミッタのカスタムアクションデータについて内容を検証し、ユーザーデータ定義との不整合が見つかれば修復を行います。
        /// </summary>
        /// <param name="emitter">エミッタ</param>
        /// <param name="customActionTypeNames">シリアライズされたカスタムアクションタイプのリスト</param>
        /// <returns>通常はtrue、選択されているカスタムアクションデータが置き換えられたときfalseを返します。</returns>
        private bool RepairCustomActionData(EmitterData emitter, string[] customActionTypeNames)
        {
            bool result = true;

            if (emitter.CustomActionData == null)
            {
                return true;
            }

            // 処理用の変数
            UserDataInfo[] infoArray = CustomActionUserDataManager.EnumerateCustomActionUserDataInfo().ToArray();
            Debug.Assert(infoArray.Length == 8, "カスタムアクションのUDD情報が不正");

            ArrayCollection<CustomActionSettingData> settings = emitter.CustomActionData.Settings;
            CustomActionSettingData selectedSettingData = emitter.CustomActionData.SelectedSettingData;
            int selectedSettingIndex = emitter.CustomActionData.SelectedSettingIndex;

            int newSelectedSettingIndex = selectedSettingIndex;
            CustomActionSettingData[] newSettings = new CustomActionSettingData[8];

            string[] addedDataNames = new string[8];
            string[] removedDataNames = new string[8];

            List<CustomActionSettingData> tmpSettings = settings.Where(x => x != null).ToList();

            // コールバックIDごとにユーザーデータ型に一致するデータを検索する
            for (int i = 0; i < infoArray.Length; ++i)
            {
                UserDataInfo info = infoArray[i];

                // ユーザーデータ型が未定義のデータはnullにする
                if (info == null)
                {
                    newSettings[i] = null;
                    continue;
                }

                // ユーザーデータ型が一致するデータを検索
                CustomActionSettingData setting = tmpSettings.FirstOrDefault(x => info.DataModelType.IsInstanceOfType(x));

                // データが見つからなければ新規で作る
                if (setting == null)
                {
                    newSettings[i] = (CustomActionSettingData)UserDataManager.CreateUserDataFromInfo(info);
                    addedDataNames[i] = info.DataModelType.Name;
                    continue;
                }

                newSettings[i] = setting;
                tmpSettings.Remove(setting);

                // データが選択されているとき、並べ替え後のインデックスを記録
                if (setting == selectedSettingData)
                {
                    newSelectedSettingIndex = i;
                }
            }

            // 対応する型がアセンブリから見つからないときは空のCustomActionSettingDataが作られるため、以下の処理は不要
            #if false
            // デシリアライズ時に削除されたデータの名前を取得する
            for (int i = 0; i < 8; ++i)
            {
                if (customActionTypeNames[i] != null && settings[i] == null)
                {
                    removedDataNames[i] = customActionTypeNames[i];
                }
            }
            #endif

            // データの並べ替え時に削除されたデータの名前を取得する
            foreach (CustomActionSettingData tmp in tmpSettings)
            {
                int i = settings.Select((value, index) => new { value, index }).First(x => x.value == tmp).index;
                removedDataNames[i] = customActionTypeNames[i];
            }

            // ログが増えすぎるため出力オフ
            #if false
            // データの削除または追加を行ったことをログに出力する
            for (int i = 0; i < 8; ++i)
            {
                if (removedDataNames[i] != null)
                {
                    Logger.Log("LogView,Console", LogLevels.Warning, "CustomActionRemoved", emitter.Name, i, removedDataNames[i]);
                }

                if (addedDataNames[i] != null)
                {
                    Logger.Log("LogView,Console", LogLevels.Warning, "CustomActionAdded", emitter.Name, i, addedDataNames[i]);
                }
            }
            #endif

            // カスタムアクションデータを更新する（親子関係は順番に変更がないときでも更新する必要あり）
            // CustomActionData.Children は CustomShaderData.Settings のセッターで更新されるが、
            // シリアライズ時は CustomActionData.Settings の内容だけが書き換えられてしまう
            // そのため、CustomShaderData.PostConstruct() で作成されたインスタンスが Children に残ってしまう
            // if (newSettings.SequenceEqual(settings) == false)
            {
                emitter.CustomActionData.Children.Clear();

                for (int i = 0; i < 8; ++i)
                {
                    emitter.CustomActionData.Settings[i] = newSettings[i];

                    if (newSettings[i] != null)
                    {
                        emitter.CustomActionData.AddChild(newSettings[i]);
                    }
                }
            }

            // 選択インデックスを更新する
            if (newSelectedSettingIndex != selectedSettingIndex)
            {
                emitter.CustomActionData.SelectedSettingIndex = newSelectedSettingIndex;

                Logger.Log("LogView,Console", LogLevels.Warning, Resources.WarningSelectingCustomActionIndexModified, emitter.Name, selectedSettingIndex, newSelectedSettingIndex);
            }

            // 置き換えられた選択データをログに出力する
            if (removedDataNames[selectedSettingIndex] != null)
            {
                result = false;

                CustomActionSettingData newSelectedSettingData = newSettings[newSelectedSettingIndex];
                string newSelectedSettingDataName = (newSelectedSettingData == null ? "null" : newSelectedSettingData.GetType().Name);

                Logger.Log("LogView,Console", LogLevels.Warning, Resources.WarningSelectingCustomActionDataReplaced, emitter.Name, customActionTypeNames[selectedSettingIndex], newSelectedSettingDataName);
            }

            // アセンブリデータ型とデシリアライズした型を表示する
            if (result == false)
            {
                StringBuilder assemblyTypeNames = new StringBuilder("CustomAction AssemblyTypes: ");

                for (int i = 0; i < infoArray.Length; ++i)
                {
                    if (infoArray[i] == null || infoArray[i].DataModelType == null)
                    {
                        assemblyTypeNames.Append(i + ": null");
                    }
                    else
                    {
                        assemblyTypeNames.Append(i + ": " + infoArray[i].DataModelType.Name);
                    }

                    if (i < infoArray.Length - 1)
                    {
                        assemblyTypeNames.Append(", ");
                    }
                }

                StringBuilder emitterTypeNames = new StringBuilder("CustomAction EsetTypes: ");

                for (int i = 0; i < customActionTypeNames.Length; ++i)
                {
                    if (customActionTypeNames[i] == null)
                    {
                        emitterTypeNames.Append(i + ": null");
                    }
                    else
                    {
                        emitterTypeNames.Append(i + ": " + customActionTypeNames[i]);
                    }

                    if (i < customActionTypeNames.Length - 1)
                    {
                        emitterTypeNames.Append(", ");
                    }
                }

                Logger.Log("LogView", LogLevels.Warning, assemblyTypeNames.ToString());
                Logger.Log("LogView", LogLevels.Warning, emitterTypeNames.ToString());
            }

            return result;
        }

        /// <summary>
        /// シリアライズされたデータのカスタムシェーダデータタイプを列挙します。
        /// カスタムシェーダが未設定のときは名前にnullが入ります。
        /// </summary>
        /// <param name="doc">処理するXMLデータ</param>
        /// <returns>カスタムシェーダデータタイプの名前を返します。</returns>
        private string[] EnumerateCustomShaderTypeNames(XDocument doc)
        {
            const string EmitterElementName          = "EmitterData";
            const string CustomShaderDataElementName = "EmitterCustomShaderData";
            const string SettingsElementName         = "Settings";

            const string xsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            XName typeAttrName = XName.Get("type", xsiNamespaceUri);

            string[] customShaderTypeNames = new string[8];

            bool enumerated = false;

            // カスタムシェーダデータタイプの名前を列挙する
            foreach (XElement element in doc.Descendants(EmitterElementName))
            {
                // <EmitterCustomShaderData> を探す
                XElement customShaderElement = element.Element(CustomShaderDataElementName);
                if (customShaderElement != null && customShaderElement.IsEmpty == false)
                {
                    var settingsElem = customShaderElement.Element(SettingsElementName);
                    if (settingsElem != null && settingsElem.IsEmpty == false)
                    {
                        int index = 0;

                        foreach (XElement settingElem in settingsElem.Elements())
                        {
                            if (settingElem.IsEmpty == false)
                            {
                                XAttribute xsiTypeAttribute = settingElem.Attribute(typeAttrName);
                                if (xsiTypeAttribute != null)
                                {
                                    customShaderTypeNames[index] = xsiTypeAttribute.Value;
                                }
                            }

                            ++index;
                        }

                        enumerated = true;
                    }
                }

                if (enumerated)
                {
                    break;
                }
            }

            return customShaderTypeNames;
        }

        /// <summary>
        /// エミッタのカスタムシェーダデータについて内容を検証し、ユーザーデータ定義との不整合が見つかれば修復を行います。
        /// </summary>
        /// <param name="emitter">エミッタ</param>
        /// <param name="customShaderTypeNames">シリアライズされたカスタムシェーダタイプのリスト</param>
        /// <returns>通常はtrue、選択されているカスタムシェーダデータが置き換えられたときfalseを返します。</returns>
        private bool RepairCustomShaderData(EmitterData emitter, string[] customShaderTypeNames)
        {
            bool result = true;

            // 処理用の変数
            UserDataInfo[] infoArray = CustomShaderUserDataManager.EnumerateCustomShaderUserDataInfo().ToArray();
            Debug.Assert(infoArray.Length == 8, "カスタムシェーダのUDD情報が不正");

            ArrayCollection<EmitterCustomShaderSettingData> settings = emitter.EmitterCustomShaderData.Settings;
            EmitterCustomShaderSettingData selectedSettingData = emitter.EmitterCustomShaderData.SelectedSettingData;
            int selectedSettingIndex = emitter.EmitterCustomShaderData.SelectedSettingIndex;

            int newSelectedSettingIndex = selectedSettingIndex;
            EmitterCustomShaderSettingData[] newSettings = new EmitterCustomShaderSettingData[8];

            string[] addedDataNames = new string[8];
            string[] removedDataNames = new string[8];

            List<EmitterCustomShaderSettingData> tmpSettings = settings.Where(x => x != null).ToList();

            // シェーダIDごとにユーザーデータ型に一致するデータを検索する
            for (int i = 0; i < infoArray.Length; ++i)
            {
                UserDataInfo info = infoArray[i];

                // ユーザーデータ型が未定義のデータはnullにする
                if (info == null)
                {
                    newSettings[i] = null;
                    continue;
                }

                // ユーザーデータ型が一致するデータを検索
                EmitterCustomShaderSettingData setting = tmpSettings.FirstOrDefault(x => info.DataModelType.IsInstanceOfType(x));

                // データが見つからなければ新規で作る
                if (setting == null)
                {
                    newSettings[i] = (EmitterCustomShaderSettingData)UserDataManager.CreateUserDataFromInfo(info);
                    addedDataNames[i] = info.DataModelType.Name;
                    continue;
                }

                newSettings[i] = setting;
                tmpSettings.Remove(setting);

                // データが選択されているとき、並べ替え後のインデックスを記録
                if (setting == selectedSettingData)
                {
                    newSelectedSettingIndex = i;
                }
            }

            // 対応する型がアセンブリから見つからないときは空のEmitterCustomShaderSettingDataが作られるため、以下の処理は不要
            #if false
            // デシリアライズ時に削除されたデータの名前を取得する
            for (int i = 0; i < 8; ++i)
            {
                if (customShaderTypeNames[i] != null && settings[i] == null)
                {
                    removedDataNames[i] = customShaderTypeNames[i];
                }
            }
            #endif

            // データの並べ替え時に削除されたデータの名前を取得する
            foreach (EmitterCustomShaderSettingData tmp in tmpSettings)
            {
                int i = settings.Select((value, index) => new { value, index }).First(x => x.value == tmp).index;
                removedDataNames[i] = customShaderTypeNames[i];
            }

            // ログが増えすぎるため出力オフ
            #if false
            // データの削除または追加を行ったことをログに出力する
            for (int i = 0; i < 8; ++i)
            {
                if (removedDataNames[i] != null)
                {
                    Logger.Log("LogView,Console", LogLevels.Warning, "CustomShaderRemoved", emitter.Name, i, removedDataNames[i]);
                }

                if (addedDataNames[i] != null)
                {
                    Logger.Log("LogView,Console", LogLevels.Warning, "CustomShaderAdded", emitter.Name, i, addedDataNames[i]);
                }
            }
            #endif

            // カスタムシェーダデータを更新する（親子関係は順番に変更がないときでも更新する必要あり）
            // EmitterCustomShaderData.Children は EmitterCustomShaderData.Settings のセッターで更新されるが、
            // シリアライズ時は EmitterCustomShaderData.Settings の内容だけが書き換えられてしまう
            // そのため、EmitterCustomShaderData.PostConstruct() で作成されたインスタンスが Children に残ってしまう
            // if (newSettings.SequenceEqual(settings) == false)
            {
                emitter.EmitterCustomShaderData.Children.Clear();

                for (int i = 0; i < 8; ++i)
                {
                    emitter.EmitterCustomShaderData.Settings[i] = newSettings[i];

                    if (newSettings[i] != null)
                    {
                        emitter.EmitterCustomShaderData.AddChild(newSettings[i]);
                    }
                }
            }

            // 選択インデックスを更新する
            if (newSelectedSettingIndex != selectedSettingIndex)
            {
                emitter.EmitterCustomShaderData.SelectedSettingIndex = newSelectedSettingIndex;

                Logger.Log("LogView,Console", LogLevels.Warning, Resources.WarningSelectingCustomShaderIndexModified, emitter.Name, selectedSettingIndex, newSelectedSettingIndex);
            }

            // 置き換えられた選択データをログに出力する
            if (selectedSettingIndex != -1 && removedDataNames[selectedSettingIndex] != null)
            {
                result = false;

                EmitterCustomShaderSettingData newSelectedSettingData = newSettings[newSelectedSettingIndex];
                string newSelectedSettingDataName = (newSelectedSettingData == null ? "null" : newSelectedSettingData.GetType().Name);

                Logger.Log("LogView,Console", LogLevels.Warning, Resources.WarningSelectingCustomShaderDataReplaced, emitter.Name, customShaderTypeNames[selectedSettingIndex], newSelectedSettingDataName);
            }

            // アセンブリデータ型とデシリアライズした型を表示する
            if (result == false)
            {
                StringBuilder assemblyTypeNames = new StringBuilder("CustomShader AssemblyTypes: ");

                for (int i = 0; i < infoArray.Length; ++i)
                {
                    if (infoArray[i] == null || infoArray[i].DataModelType == null)
                    {
                        assemblyTypeNames.Append(i + ": null");
                    }
                    else
                    {
                        assemblyTypeNames.Append(i + ": " + infoArray[i].DataModelType.Name);
                    }

                    if (i < infoArray.Length - 1)
                    {
                        assemblyTypeNames.Append(", ");
                    }
                }

                StringBuilder emitterTypeNames = new StringBuilder("CustomShader EsetTypes: ");

                for (int i = 0; i < customShaderTypeNames.Length; ++i)
                {
                    if (customShaderTypeNames[i] == null)
                    {
                        emitterTypeNames.Append(i + ": null");
                    }
                    else
                    {
                        emitterTypeNames.Append(i + ": " + customShaderTypeNames[i]);
                    }

                    if (i < customShaderTypeNames.Length - 1)
                    {
                        emitterTypeNames.Append(", ");
                    }
                }

                Logger.Log("LogView", LogLevels.Warning, assemblyTypeNames.ToString());
                Logger.Log("LogView", LogLevels.Warning, emitterTypeNames.ToString());
            }

            return result;
        }

        /// <summary>
        /// シリアライズされたデータのエミッタ拡張パラメータデータタイプを列挙します。
        /// エミッタ拡張パラメータが未設定のときは名前にnullが入ります。
        /// </summary>
        /// <param name="doc">処理するXMLデータ</param>
        /// <returns>エミッタ拡張パラメータデータタイプの名前を返します。</returns>
        private string GetEmitterExtParamsTypeName(XDocument doc)
        {
            const string EmitterElementName          = "EmitterData";
            const string EmitterExtParamsElementName = "EmitterExtParams";
            const string SettingDataElementName      = "EmitterExtParamsSettingData";

            const string xsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            XName typeAttrName = XName.Get("type", xsiNamespaceUri);

            string emitterExtParamsTypeName = null;

            bool enumerated = false;

            // エミッタ拡張パラメータデータタイプの名前を取得する
            foreach (XElement element in doc.Descendants(EmitterElementName))
            {
                // <EmitterExtParams> を探す
                XElement emitterExtParamsElement = element.Element(EmitterExtParamsElementName);
                if (emitterExtParamsElement != null && emitterExtParamsElement.IsEmpty == false)
                {
                    var settingDataElem = emitterExtParamsElement.Element(SettingDataElementName);
                    if (settingDataElem != null && settingDataElem.IsEmpty == false)
                    {
                        XAttribute xsiTypeAttribute = settingDataElem.Attribute(typeAttrName);
                        if (xsiTypeAttribute != null)
                        {
                            emitterExtParamsTypeName = xsiTypeAttribute.Value;
                        }

                        enumerated = true;
                    }
                }

                if (enumerated)
                {
                    break;
                }
            }

            return emitterExtParamsTypeName;
        }

        /// <summary>
        /// エミッタのエミッタ拡張パラメータデータについて内容を検証し、ユーザーデータ定義との不整合が見つかれば修復を行います。
        /// </summary>
        /// <param name="emitter">エミッタ</param>
        /// <param name="emitterExtParamsTypeName">シリアライズされたエミッタ拡張パラメータデータタイプの名前</param>
        /// <returns>通常はtrue、使用されているエミッタ拡張パラメータデータが置き換えられたときfalseを返します。</returns>
        private bool RepairEmitterExtParamsData(EmitterData emitter, string emitterExtParamsTypeName)
        {
            bool result = true;

            if (emitter.EmitterExtParams == null)
            {
                return true;
            }

            // 処理用の変数
            UserDataInfo info = EmitterExtParamsUserDataManager.FindEmitterExtParamsUserDataInfo();

            EmitterExtParamsSettingData settingData = emitter.EmitterExtParams.EmitterExtParamsSettingData;
            EmitterExtParamsSettingData newSettingData = null;

            string addedDataName = null;
            string removedDataName = null;

            do
            {
                // ユーザーデータ型が未定義のときはnullにする
                if (info == null)
                {
                    newSettingData = null;
                    break;
                }

                // ユーザーデータ型が一致すればそれを使う
                if (info.DataModelType.IsInstanceOfType(settingData))
                {
                    newSettingData = (EmitterExtParamsSettingData)settingData;
                }

                // ユーザーデータ型が一致しなければ新規で作る
                if (newSettingData == null)
                {
                    newSettingData = (EmitterExtParamsSettingData)UserDataManager.CreateUserDataFromInfo(info);
                    addedDataName = info.DataModelType.Name;
                }
            } while (false);

            // 削除されたデータの名前を取得する
            if (emitterExtParamsTypeName != null)
            {
                if (newSettingData == null || settingData != newSettingData)
                {
                    removedDataName = emitterExtParamsTypeName;
                }
            }

            // ログが増えすぎるため出力オフ
            #if false
            // データの削除または追加を行ったことをログに出力する
            {
                if (removedDataName != null)
                {
                    Logger.Log("LogView,Console", LogLevels.Warning, "EmitterExtParamsDataRemoved", emitter.Name, removedDataName);
                }

                if (addedDataName != null)
                {
                    Logger.Log("LogView,Console", LogLevels.Warning, "EmitterExtParamsDataAdded", emitter.Name, addedDataName);
                }
            }
            #endif

            // エミッタ拡張パラメータデータを更新する
            // 親子関係は値に変更がないときに更新する必要はないが、他のカスタムデータに合わせて更新しておく
            // if (settingData != newSettingData)
            {
                emitter.EmitterExtParams.Children.Clear();

                emitter.EmitterExtParams.EmitterExtParamsSettingData = newSettingData;

                if (newSettingData != null)
                {
                    emitter.EmitterExtParams.AddChild(newSettingData);
                }
            }

            // 置き換えられたエミッタ拡張パラメータデータをログに出力する
            if (removedDataName != null)
            {
                result = false;

                string newSettingDataName = (newSettingData == null ? "null" : newSettingData.GetType().Name);

                Logger.Log("LogView,Console", LogLevels.Warning, Resources.WarningEmitterExtParamsDataReplaced, emitter.Name, removedDataName, newSettingDataName);
            }

            // アセンブリデータ型とデシリアライズした型を表示する
            if (result == false)
            {
                StringBuilder assemblyTypeName = new StringBuilder("EmitterExtParams AssemblyType: ");

                if (info == null)
                {
                    assemblyTypeName.Append("null");
                }
                else
                {
                    assemblyTypeName.Append(info.DataModelType.Name);
                }

                StringBuilder emitterTypeName = new StringBuilder("EmitterExtParams EsetType: ");

                if (emitterExtParamsTypeName == null)
                {
                    emitterTypeName.Append("null");
                }
                else
                {
                    emitterTypeName.Append(emitterExtParamsTypeName);
                }

                Logger.Log("LogView", LogLevels.Warning, assemblyTypeName.ToString());
                Logger.Log("LogView", LogLevels.Warning, emitterTypeName.ToString());
            }

            return result;
        }

        #endregion

        /// <summary>
        /// デシリアライズ結果です。
        /// </summary>
        public new class DeserializeResult : BaseDeserializeResult
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            public DeserializeResult()
            {
            }

            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="obj">初期値</param>
            public DeserializeResult(BaseDeserializeResult obj)
                : base(obj)
            {
            }

            /// <summary>
            /// 選択されているカスタムアクションデータが置き換えられたかどうか取得または設定します。
            /// </summary>
            public bool IsSelectedCustomActionDataReplaced { get; set; }

            /// <summary>
            /// 選択されているカスタムシェーダデータが置き換えられたかどうか取得または設定します。
            /// </summary>
            public bool IsSelectedCustomShaderDataReplaced { get; set; }

            /// <summary>
            /// 使用されているエミッタ拡張パラメータが置き換えられたかどうか取得または設定します。
            /// </summary>
            public bool IsEmitterExtParamsDataReplaced { get; set; }

            /// <summary>
            /// ユーザー定義データが置き換えられたかどうか取得します。
            /// </summary>
            public bool AnyUserDefinitionDataReplaced
            {
                get
                {
                    bool uddReplaced = false;

                    uddReplaced |= this.IsSelectedCustomActionDataReplaced;
                    uddReplaced |= this.IsSelectedCustomShaderDataReplaced;
                    uddReplaced |= this.IsEmitterExtParamsDataReplaced;

                    return uddReplaced;
                }
            }
        }
    }
}
