﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.Protocol
{
    /// <summary>
    /// Reads message from binary data.
    /// </summary>
    public class PacketReader
    {
        /// <summary>A dictionary that maps message type enum to their real type.</summary>
        private static readonly Dictionary<MessageTypes, Type> MessageMap =
            new Dictionary<MessageTypes, Type>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static PacketReader()
        {
            // Register your messages here.
            // E.g. RegisterMessage(MessageTypes.MyMessage, typeof(MyMessage));
            RegisterMessage(MessageTypes.ModelInfo, typeof(ModelInfoMessage));
            RegisterMessage(MessageTypes.SendBinary, typeof(ViewerMessage));
            RegisterMessage(MessageTypes.BinaryData, typeof(CameraMessage));
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public PacketReader()
        {
        }

        /// <summary>
        /// Read message from the given binary buffer.
        /// </summary>
        /// <param name="buffer">The byte array that contains the message data.</param>
        /// <returns>The message read from the binary buffer.</returns>
        public MessageBase[] Read(byte[] buffer)
        {
            var messages = new List<MessageBase>();

            int offset = 0;
            while (offset < buffer.Length)
            {
                // Read the message type and size.
                var type = (MessageTypes)BinaryConversionUtility.ForProtocol.ConvertBack<uint>(buffer, offset);

                var dataSize = BinaryConversionUtility.ForProtocol.ConvertBack<int>(buffer, offset + 4);
                int messageSize = dataSize + 8; // Plus type and size, 4 bytes each.

                try
                {
                    MessageBase message = null;
                    using (var stream = new MemoryStream(buffer, offset, messageSize))
                    {
                        // Find the message type from our map.
                        Type classType;
                        if (MessageMap.TryGetValue(type, out classType) == false)
                        {
                            Logger.Log(LogLevels.Error, "PacketReader.Read : The message type {0} is not unknown.", type.ToString());
                            continue;
                        }

                        // Create a instance of the message type.
                        message = (MessageBase)Activator.CreateInstance(classType);

                        // Read the message.
                        message.Read(stream);
                        messages.Add(message);
                    }
                }
                finally
                {
                    offset += messageSize;
                }
            }

            return messages.ToArray();
        }

        /// <summary>
        /// Register message.
        /// </summary>
        /// <param name="messageType">The message type enum.</param>
        /// <param name="classType">The type of the message class.</param>
        private static void RegisterMessage(MessageTypes messageType, Type classType)
        {
            if (typeof(MessageBase).IsAssignableFrom(classType) == false)
            {
                throw new ArgumentException("The message class must inherit from MessageBase.");
            }

            if (MessageMap.ContainsKey(messageType) == true)
            {
                throw new ArgumentException("The message type {0} has already been registered.", messageType.ToString());
            }

            MessageMap[messageType] = classType;
        }
    }
}
