﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.BinaryHeaders.Helpers;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;

using MyTextureManager = EffectMaker.BusinessLogic.Manager.TextureManager;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Texture
{
    /// <summary>
    /// Write texture resource to binary stream.
    /// </summary>
    public class TextureBinaryWriter : IBinaryResourceWriter
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="data">The texture binary data to write.</param>
        public TextureBinaryWriter(TextureBinaryData data)
        {
            if (data == null)
            {
                throw new ArgumentException("The texture binary data must not be null.");
            }

            this.DataModel = data;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var textureBin = this.DataModel as TextureBinaryData;
            if (textureBin == null)
            {
                return false;
            }

            // Save the start position in the stream.
            this.Position = stream.Position;

            // Try to get the texture.
            LoadTextureResult result = MyTextureManager.Instance.LoadTexture(textureBin.TexturePath, true);

            // Failed to find the texture, bail out.
            if (result.ResultCode != LoadTextureResultCode.Success)
            {
                return false;
            }

            // Only FTX textures are supported now.
            var ftxTexData = result.TextureData as FtxTextureData;
            if (ftxTexData == null)
            {
                return false;
            }

            // Write an empty binary header for place holder,
            // we will come back and fill in the correct values later.
            BinaryStructHeader.Empty.Write(stream);

            // Compute alignment(バイナリ先頭からのアラインメント).
            this.Offset = stream.Position % (long)ftxTexData.Alignment;
            if (this.Offset > 0)
            {
                this.Offset = (long)ftxTexData.Alignment - this.Offset;
            }

            context.UpdateAlignmentSize(ftxTexData.Alignment);

            // Write alignment with zeroed byte array.
            stream.Write(
                Enumerable.Repeat((byte)0, (int)this.Offset).ToArray(),
                0,
                (int)this.Offset);

            if (SpecManager.CurrentSpec.ValidNativeTextureFormats != null &&
                SpecManager.CurrentSpec.ValidNativeTextureFormats.Contains(ftxTexData.TextureTypeTag) &&
                ftxTexData.TextureTypeTag != "ORGB")
            {
                // 現在のスペックで対応しているネイティブフォーマットならストリームを付加するだけ
                stream.Write(ftxTexData.NativeImageStream, 0, ftxTexData.NativeImageStream.Length);
            }
            else if (!SpecManager.CurrentSpec.ForceNativeTexture)
            {
                // 未対応のネイティブフォーマットだったらオリジナルビットマップを書き込む
                ftxTexData.WriteOriginalBitmapStream(stream);
            }
            else
            {
                // オリジナルイメージが禁止されているスペックだったらエラーを出す
                Logger.Log(
                    Options.OptionStore.RuntimeOptions.IsCommandLineMode ? "Console" : "LogView",
                    LogLevels.Error,
                    "Invalid texture format {0}",
                    ftxTexData.TextureTypeTag);
                return false;
            }

            // Set the size.
            this.Size = stream.Position - this.Position;

            // Add this writer to the context, the context will write the binary header
            // for the added binary writers.
            context.AddBinaryWriter(this);

            return true;
        }
    }
}
