﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Utility;
using EffectMaker.BusinessLogic.Options;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters
{
    /// <summary>
    /// エミッタセットのファイルパスライタです。
    /// </summary>
    public class EmitterSetFilePathArrayWriter : IBinaryResourceWriter
    {
        /// <summary>The emitter sets to output file paths.</summary>
        private EmitterSetData[] emitterSets;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="emitterSets"></param>
        public EmitterSetFilePathArrayWriter(IEnumerable<EmitterSetData> emitterSets)
        {
            this.emitterSets = emitterSets.ToArray();

            var emitterSetPathTable = new EmitterSetFilePathTableData();

            // First collect all the emitter set paths
            foreach (var emitterSet in this.emitterSets)
            {
                emitterSetPathTable.AddEmitterSetPath(emitterSet.FilePath.ToLower());
            }

            this.DataModel = emitterSetPathTable;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var emitterSetPathTable = this.DataModel as EmitterSetFilePathTableData;
            if (emitterSetPathTable == null)
            {
                return false;
            }

            // Save the start position in the stream.
            this.Position = stream.Position;

            // Write an empty binary header for place holder,
            // we will come back and fill in the correct values later.
            BinaryStructHeader.Empty.Write(stream);

            if (emitterSetPathTable.EmitterSetPaths.Any())
            {
                // char    guid[16];       //!< エミッタセットのguid (常に0)
                // u32     nextOffset;     //!< 次のレコードへのオフセット
                // u32     filePathLength; //!< ファイルパスの長さ
                // char    filePath[8];    //!< ファイルパス

                int emitterSetsCount = this.emitterSets.Count();
                int counter = 0;

                foreach (var emitterSet in this.emitterSets)
                {
                    string relativeFilePath = PathUtility.GetRelativePath(OptionStore.ProjectConfig.ConfigFilePath, emitterSet.FilePath) + @"\" + emitterSet.Name + ".eset";

                    relativeFilePath =  relativeFilePath.ToLower().Replace('/', '\\');

                    int lengthWithNull = relativeFilePath.Length + 1;        // NULL終端を含むファイルパスの長さ
                    int paddingSize    = 8 - (lengthWithNull % 8);           // ファイルパスのサイズを8バイトアラインに揃えるためのパディング
                    int recordSize     = 24 + lengthWithNull + paddingSize;  // 1レコードあたりのサイズ。24はguid, nextOffset, fileLength変数のサイズのバイト和
                    int nextOffset     = (counter == emitterSetsCount - 1 ? 0 : recordSize);  // レコードの先頭から、次のレコード先頭へのオフセット

                    BinaryConversionUtility.ForResource.WriteStream(stream, Enumerable.Repeat((byte)0, 16));
                    BinaryConversionUtility.ForResource.WriteStream(stream, nextOffset);
                    BinaryConversionUtility.ForResource.WriteStream(stream, lengthWithNull);
                    BinaryConversionUtility.ForResource.WriteStream(stream, relativeFilePath.ToArray());
                    BinaryConversionUtility.ForResource.WriteStream(stream, Enumerable.Repeat((byte)0, 1));

                    if (paddingSize != 0)
                    {
                        BinaryConversionUtility.ForResource.WriteStream(stream, Enumerable.Repeat((byte)0, paddingSize));
                    }

                    counter++;
                }
            }

            // Save the size of the binary data.
            this.Size = stream.Position - this.Position;

            // Add this writer to the context, the context will write the binary header
            // for the added binary writers.
            context.AddBinaryWriter(this);

            return true;
        }
    }
}
