﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.BinaryHeaders
{
    /// <summary>
    /// The top level header in a binary file/stream.
    /// </summary>
    public class BinaryFileHeader
    {
        /// <summary>The binary file version.</summary>
        private const uint version = 0x00000016;

        /// <summary>
        /// バイナリファイルバージョンを取得します。
        /// </summary>
        public static uint Version
        {
            get
            {
                if (OverwriteVersion != null)
                {
                    return OverwriteVersion.Value;
                }
                else
                {
                    return version;
                }
            }
        }

        /// <summary>
        /// 上書きバージョンを取得または設定します。
        /// </summary>
        public static uint? OverwriteVersion { get; set; }

        /// <summary>バイナリサイズ</summary>
        public long BinarySize { get; private set; } = 0;

        /// <summary>An empty binary header.</summary>
        private static readonly BinaryFileHeader EmptyHeader = new BinaryFileHeader();

        /// <summary>
        /// グラフィクスAPI文字列と定数値の辞書です。
        /// </summary>
        private static readonly Dictionary<string, int> graphicsApiDictionary = new Dictionary<string, int>();

        /// <summary>
        /// 現在のアライメントサイズです。
        /// エフェクトバイナリ本体は4096(2^12)なので、それを初期値とします。
        /// </summary>
        private byte currentAlignmentSize = 12;

        /// <summary>
        /// スタティックコンストラクタです。
        /// </summary>
        static BinaryFileHeader()
        {
            graphicsApiDictionary["GLES2"] = 0; // GFXではサポートなしなので0にしておく
            graphicsApiDictionary["GL"] = 1;
            graphicsApiDictionary["GX2"] = 2;
            graphicsApiDictionary["NVN"] = 4;
            graphicsApiDictionary["VK"] = 5;
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public BinaryFileHeader()
        {
            this.BinaryName = string.Empty;
        }

        /// <summary>
        /// Get an empty binary header.
        /// </summary>
        public static BinaryFileHeader Empty
        {
            get { return EmptyHeader; }
        }

        /// <summary>
        /// Get the size of the binary header.
        /// </summary>
        public static uint Size
        {
            get { return SpecManager.CurrentSpec.BinaryHeader == "VFXB" ? (uint)64 : (uint)48; }
        }

        /// <summary>
        /// Get or set the binary name.
        /// </summary>
        public string BinaryName { get; set; }

        /// <summary>
        /// アライメントサイズを書き込むアクションを取得します。
        /// </summary>
        public Action<byte> WriteAlignmentSize { get; private set; }

        /// <summary>
        /// ファイルサイズを書き込むアクションを取得します。
        /// </summary>
        public Action WriteFileSize { get; private set; }

        /// <summary>
        /// Write the header to the given stream.
        /// </summary>
        /// <param name="stream">The stream to write to.</param>
        public void Write(Stream stream)
        {
            long current = stream.Position;
            string header = SpecManager.CurrentSpec.BinaryHeader;

            if (header == "VFXB")
            {
                this.WriteVfxbHeader(stream);
            }
            else if (header == "EFTB")
            {
                this.WriteEftbHeader(stream);
            }

            if (stream.Position - current != Size)
            {
                throw new InvalidProgramException("BinaryFileHeader.Write is wrong implemented.");
            }
        }

        /// <summary>
        /// ファイルのヘッダを読み取り、現在のスペックに適合しているかチェックします。
        /// </summary>
        /// <param name="filePath">バイナリファイルパス</param>
        /// <returns>0:合ってる,1:ヘッダが違う,2:バージョンが違う,3:スペックが違う</returns>
        public static int IsSuitableHeader(string filePath)
        {
            using (var stream = new FileStream(filePath, FileMode.Open, FileAccess.Read))
            {
                string header = SpecManager.CurrentSpec.BinaryHeader;

                if (header == "VFXB")
                {
                    return CheckVfxbHeader(stream);
                }

                if (header == "EFTB")
                {
                    return CheckEftbHeader(stream);
                }
            }

            return 3;
        }

        /// <summary>
        /// Vfxbファイルのヘッダを読み取り、現在のスペックに適合しているかチェックします。
        /// </summary>
        /// <param name="stream">バイナリファイルストリームパス</param>
        /// <returns>0:合ってる,1:ヘッダが違う,2:バージョンが違う,3:スペックが違う</returns>
        private static int CheckVfxbHeader(Stream stream)
        {
            byte[] byteData = new byte[32];
            stream.Read(byteData, 0, 32);

            // 現在指定されているspecから、バイナリの先頭にあるべき文字列を参照し、バイト列の最初と一致するかチェックする.
            if (System.Text.Encoding.ASCII.GetString(byteData, 0, 4) != SpecManager.CurrentSpec.BinaryHeader)
            {
                return 1;
            }

            // 設定されているエンディアンを元に、バイナリバージョンをチェック
            int binaryVersion = 0;
            var byteArray = Enumerable.Range(0, 2);

            // リトルエンディアンの場合は、逆順にデコードする
            if (byteData[12] == 0xFF)
            {
                byteArray = byteArray.Reverse();
            }

            // エンディアンにしたがって、上位ビットからシフトしつつデコードしていく
            foreach (var i in byteArray)
            {
                binaryVersion <<= 8;
                binaryVersion += byteData[10 + i];
            }

            // バイナリバージョンが一致しなかったらアウト
            if (binaryVersion != BinaryFileHeader.Version)
            {
                return 2;
            }

            // ApiTypeが違ったらアウト
            if (GetCurrentApiConst() != byteData[9])
            {
                return 3;
            }

            return 0;
        }

        /// <summary>
        /// Eftbファイルのヘッダを読み取り、現在のスペックに適合しているかチェックします。
        /// </summary>
        /// <param name="stream">バイナリファイルストリームパス</param>
        /// <returns>0:合ってる,1:ヘッダが違う,2:バージョンが違う,3:スペックが違う</returns>
        private static int CheckEftbHeader(Stream stream)
        {
            byte[] byteData = new byte[48];
            stream.Read(byteData, 0, 48);

            // 現在指定されているspecから、バイナリの先頭にあるべき文字列を参照し、バイト列の最初と一致するかチェックする.
            if (System.Text.Encoding.ASCII.GetString(byteData, 0, 4) != SpecManager.CurrentSpec.BinaryHeader)
            {
                return 1;
            }

            // 設定されているエンディアンを元に、バイナリバージョンをチェック
            int binaryVersion = 0;
            var byteArray = Enumerable.Range(0, 4);

            // リトルエンディアンの場合は、逆順にデコードする
            if (byteData[40] == 0)
            {
                byteArray = byteArray.Reverse();
            }

            // エンディアンにしたがって、上位ビットからシフトしつつデコードしていく
            foreach (var i in byteArray)
            {
                binaryVersion <<= 8;
                binaryVersion += byteData[4 + i];
            }

            // バイナリバージョンが一致しなかったらアウト
            if (binaryVersion != BinaryFileHeader.Version)
            {
                return 2;
            }

            // ApiTypeが違ったらアウト
            if (GetCurrentApiConst() != byteData[42])
            {
                return 3;
            }

            return 0;
        }

        /// <summary>
        /// 現在のスペックが持つApiTypeを示すバイト値を取得します。
        /// </summary>
        /// <returns>現在のスペックが持つApiTypeを示すバイト値</returns>
        private static byte GetCurrentApiConst()
        {
            int apiConst;
            if (!graphicsApiDictionary.TryGetValue(SpecManager.CurrentSpec.GraphicsApi, out apiConst))
            {
                apiConst = 0;
            }

            return (byte)apiConst;
        }

        /// <summary>
        /// VFXBバイナリ用のヘッダを書き出します。
        /// nn::util::BinaryFileHeader準拠です。
        /// </summary>
        /// <param name="stream">書き込むストリーム</param>
        private void WriteVfxbHeader(Stream stream)
        {
            // シグネチャ
            BinaryConversionUtility.ForResource.WriteStream(stream, "VFXB    ".ToCharArray());

            // マイクロバージョンは未使用
            BinaryConversionUtility.ForResource.WriteStream(stream, (byte)0);

            // マイナーバージョンにApiTypeを入れる
            BinaryConversionUtility.ForResource.WriteStream(stream, GetCurrentApiConst());

            // メジャーバージョン（ランタイムバイナリバージョン）
            BinaryConversionUtility.ForResource.WriteStream(stream, (ushort)Version);

            // バイトオーダーマーク（エンディアン）
            BinaryConversionUtility.ForResource.WriteStream(stream, (ushort)0xFEFF);

            // アライメントは後から調べて入れる
            long alignmentPosition = stream.Position;
            this.currentAlignmentSize = Math.Max(
                (byte)Math.Log(SpecManager.CurrentSpec.ShaderConversionOption.BinaryAlignment, 2.0),
                this.currentAlignmentSize);
            BinaryConversionUtility.ForResource.WriteStream(stream, currentAlignmentSize);
            this.WriteAlignmentSize = (size) =>
            {
                long current = stream.Position;
                this.currentAlignmentSize = Math.Max(size, this.currentAlignmentSize);
                stream.Seek(alignmentPosition, SeekOrigin.Begin);
                BinaryConversionUtility.ForResource.WriteStream(stream, this.currentAlignmentSize);
                stream.Seek(current, SeekOrigin.Begin);
            };

            // アドレスサイズ
            BinaryConversionUtility.ForResource.WriteStream(stream, (byte)(SpecManager.CurrentSpec.AddressSize * 8));

            // ファイル名へのオフセット（バイナリ名）
            // ヘッダの直後にバイナリ名を書き出すので先頭から32バイト目からがバイナリ名
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)32);

            // 未使用項目（フラグ）
            BinaryConversionUtility.ForResource.WriteStream(stream, (ushort)0);

            // 最初のブロックへのオフセット
            // 関数は使えないが、セットアップ時の処理で使う
            BinaryConversionUtility.ForResource.WriteStream(stream, (ushort)64);

            // 未使用項目（リロケーションテーブルへのオフセット）
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)0);

            // ファイルサイズも後から調べて入れる
            long fileSizePosition = stream.Position;
            BinaryConversionUtility.ForResource.WriteStream(stream, (uint)0);
            this.WriteFileSize = () =>
            {
                long current = stream.Position;
                stream.Seek(fileSizePosition, SeekOrigin.Begin);
                BinaryConversionUtility.ForResource.WriteStream(stream, (uint)stream.Length);
                stream.Seek(current, SeekOrigin.Begin);
                this.BinarySize = stream.Length;
            };

            // ヘッダの後32バイトにバイナリ名
            var output = Enumerable.Repeat('\0', 32).ToArray();
            this.BinaryName.ToCharArray(0, Math.Min(this.BinaryName.Length, 31)).CopyTo(output, 0);
            BinaryConversionUtility.ForResource.WriteStream(stream, output);
        }

        /// <summary>
        /// EFTBバイナリ用のヘッダを書き出します。
        /// </summary>
        /// <param name="stream">書き込むストリーム</param>
        private void WriteEftbHeader(Stream stream)
        {
            // 選択中のスペックからバイナリヘッダの文字列を取得する
            BinaryConversionUtility.ForResource.WriteStream(stream, "EFTB".ToCharArray());

            // バイナリバージョン
            BinaryConversionUtility.ForResource.WriteStream(stream, Version);

            // First initialize the array with zeros.
            var output = Enumerable.Repeat('\0', 32).ToArray();

            // Output value as a character array with a maximum length of 32 characters.
            this.BinaryName.ToCharArray(0, Math.Min(this.BinaryName.Length, 31)).CopyTo(output, 0);

            BinaryConversionUtility.ForResource.WriteStream(stream, output);

            // バイナリの情報を詰め込みます。
            var binaryInfo = new byte[]
            {
                (byte)SpecManager.CurrentSpec.CpuEndian,
                (byte)SpecManager.CurrentSpec.GpuEndian,
                GetCurrentApiConst(),
                (byte)SpecManager.CurrentSpec.AddressSize,
                0,
                0,
                0,
                0,
            };

            BinaryConversionUtility.ForResource.WriteStream(stream, binaryInfo);
        }
    }
}
