﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Communicator
{
    /// <summary>
    /// メッセージマネージャです.
    /// </summary>
    public sealed class MessageManager : IDisposable
    {
        #region Private Fields

        /// <summary>
        /// メッセージマネージャの唯一のインスタンスです.
        /// </summary>
        private static readonly MessageManager MsgInstance = new MessageManager();

        /// <summary>
        /// 自動接続チェックの時間間隔(msec単位)です.
        /// </summary>
        private readonly int autoConnectTimeMsec = 1000;

        /// <summary>
        /// タイムアウトの時間(msec単位)です.
        /// </summary>
        private readonly int timeOutMsec = 500;

        /// <summary>
        /// スリープさせる時間(mesc単位)です.
        /// </summary>
        private readonly int sleepTimeMsec = 200;

        /// <summary>
        /// PINGする時間間隔(msec単位)です.
        /// </summary>
        private readonly int pingMsec = 1000;

        /// <summary>
        /// 接続してから待つ時間間隔(msec単位)です.
        /// </summary>
        private readonly int connectWaitMsec = 3000;

        /// <summary>
        /// 送信用メッセージキューです.
        /// </summary>
        private MessageQueue sendQueue = new MessageQueue();

        /// <summary>
        /// アクティブなコネクターです.
        /// </summary>
        private IConnecter activeConnecter = null;

        /// <summary>
        /// アクティブなコネクションコンテストです.
        /// </summary>
        private ConnectionContext activeContext = null;

        /// <summary>
        /// 通信用スレッドです.
        /// </summary>
        private Thread thread = null;

        /// <summary>
        /// 持越しメッセージです.
        /// </summary>
        private Message leftOverMessage = null;

        /// <summary>
        /// 以前の接続状態です.
        /// </summary>
        private ConnectionStates prevState = ConnectionStates.Disconnected;

        /// <summary>
        /// 以前のコネクターです.
        /// </summary>
        private IConnecter prevActiveConnecter = null;

        /// <summary>
        /// コネクタインタフェースのリストです.
        /// </summary>
        private Dictionary<uint, IConnecter> connecters = new Dictionary<uint, IConnecter>();

        /// <summary>
        /// メッセージ通知インタフェースの連想コンテナです.
        /// </summary>
        private Dictionary<Guid, IMessageListener> messageListeners = new Dictionary<Guid, IMessageListener>();

        /// <summary>
        /// 接続状態通知インタフェースの連想コンテナです.
        /// </summary>
        private Dictionary<uint, IConnectionStateListener> connStateListeners = new Dictionary<uint, IConnectionStateListener>();

        /// <summary>
        /// スレッドを終了させるかどうかのフラグです.
        /// </summary>
        private bool isTermThread = false;

        /// <summary>
        /// 内部接続状態です.
        /// </summary>
        private bool stateConnected = false;

        /// <summary>
        /// 最後にPINGした時間です.
        /// </summary>
        private int lastPingTime = 0;

        #endregion

        #region Constructor

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        private MessageManager()
        {
            this.ProcessCount = 0;
            this.stateConnected = false;
            this.Initialize();
        }

        #endregion

        #region Public Properties

        /// <summary>
        /// インスタンスを取得します.
        /// </summary>
        public static MessageManager Instance
        {
            get { return MsgInstance; }
        }

        /// <summary>
        /// 接続されているか? 接続されていればtrue, そうでなければfalse
        /// </summary>
        public bool IsConnected
        {
            get
            {
                // アクティブなコネクタが存在するかチェック.
                if (this.activeConnecter == null)
                {
                    // 非接続状態.
                    return false;
                }

                // コネクタの接続状態を返却.
                return this.activeConnecter.IsConnected;
            }
        }

        /// <summary>
        /// 自動切断が有効かどうか?
        /// </summary>
        public bool IsAutoConnectEnabled { get; set; }

        /// <summary>
        /// エンコーディングのタイプ
        /// </summary>
        public Encoding EncodingType
        {
            get
            {
                // アクティブなコンテキストが存在するかチェック.
                if (this.activeContext == null)
                {
                    // デフォルト値を返却.
                    return Encoding.UTF8;
                }

                // コンテキストのエンコーディングを返却.
                return this.activeContext.Encoding;
            }
        }

        /// <summary>
        /// プラットフォームタイプ.
        /// </summary>
        public int Platform
        {
            get
            {
                // アクティブなコンテキストが存在するかチェック.
                if (this.activeContext == null)
                {
                    // デフォルト値を返却.
                    return -1;
                }

                // コンテキストのプラットフォームタイプを返却.
                return this.activeContext.Platform;
            }
        }

        /// <summary>
        /// ビッグエンディアンかどうかを取得します.
        /// </summary>
        public bool IsBigEndian
        {
            get
            {
                // アクティブなコンテキストが存在するかチェック.
                if (this.activeContext == null)
                {
                    // デフォルト値を返却.
                    return false;
                }

                // コンテキストのエンディアンを返却.
                return this.activeContext.IsBigEndian;
            }
        }

        /// <summary>
        /// 送信メッセージがあるかどうかを取得します.
        /// </summary>
        public bool HasSendMessage
        {
            get { return this.sendQueue.HasMessage; }
        }

        /// <summary>
        /// 送信メッセージキューに溜まっているメッセージ数を取得します.
        /// </summary>
        public int SendMessageCount
        {
            get { return this.sendQueue.Count; }
        }

        #endregion

        #region Private Properties

        /// <summary>
        /// 処理カウンターです.
        /// </summary>
        private ushort ProcessCount { get; set; }

        #endregion

        #region Public Methods

        /// <summary>
        /// 破棄処理を行います.
        /// </summary>
        public void Dispose()
        {
            // スレッド終了フラグを立てる.
            this.isTermThread = true;

            // スレッドが存在する場合.
            if (this.thread != null)
            {
                // スレッドの終了を待つ.
                this.thread.Join(this.timeOutMsec);

                // スレッドが生きてるかチェック.
                if (this.thread.IsAlive == true)
                {
                    // スレッドを終了させる ---> ThreadAbortExceptionが飛びます.
                    this.thread.Abort();
                }

                // スレッドをnullに.
                this.thread = null;
            }

            // 各コネクターをループ.
            foreach (KeyValuePair<uint, IConnecter> connecter in this.connecters)
            {
                // コネクタがなければ処理をスキップ.
                if (connecter.Value == null)
                {
                    continue;
                }

                // 切断処理.
                connecter.Value.Disconnect();

                // 破棄処理.
                connecter.Value.Dispose();
            }

            // リストをクリアします.
            this.connecters.Clear();
            this.messageListeners.Clear();
        }

        /// <summary>
        /// コネクタインタフェースを追加します.
        /// </summary>
        /// <param name="id">登録IDです.</param>
        /// <param name="conneter">登録するコネクタインタフェースです</param>
        public void AddConnecter(uint id, IConnecter conneter)
        {
            lock (this.connecters)
            {
                // 連想コンテナに追加.
                this.connecters.Add(id, conneter);
            }
        }

        /// <summary>
        /// コネクタインタフェースを削除します.
        /// </summary>
        /// <param name="id">登録IDです</param>
        public void RemoveConnecter(uint id)
        {
            lock (this.connecters)
            {
                // 連想コンテナから削除.
                this.connecters.Remove(id);
            }
        }

        /// <summary>
        /// コネクタインタフェースをクリアします.
        /// </summary>
        public void ClearConnecters()
        {
            lock (this.connecters)
            {
                this.connecters.Clear();
            }
        }

        /// <summary>
        /// メッセージ通知インタフェースを追加します.
        /// </summary>
        /// <param name="notifier">登録するメッセージ通知インタフェースです</param>
        /// <returns>登録IDを返します.</returns>
        public Guid AddMessageNotifier(IMessageListener notifier)
        {
            Guid guid = Guid.NewGuid();

            // 連想コンテナに追加.
            this.messageListeners.Add(guid, notifier);

            return guid;
        }

        /// <summary>
        /// メッセージ通知インタフェースを削除します.
        /// </summary>
        /// <param name="guid">登録IDです</param>
        public void RemoveMessageNotifier(Guid guid)
        {
            // 連想コンテナから削除.
            this.messageListeners.Remove(guid);
        }

        /// <summary>
        /// 接続状態通知インタフェースを追加します.
        /// </summary>
        /// <param name="id">登録IDです</param>
        /// <param name="notifier">登録する接続状態通知インタフェースです</param>
        public void AddStateNotifier(uint id, IConnectionStateListener notifier)
        {
            // 連想コンテナに追加.
            this.connStateListeners.Add(id, notifier);
        }

        /// <summary>
        /// 接続状態通知インタフェースを削除します.
        /// </summary>
        /// <param name="id">登録IDです.</param>
        public void RemoveNotifier(uint id)
        {
            // 連想コンテナから削除.
            this.connStateListeners.Remove(id);
        }

        /// <summary>
        /// 接続ターゲットが存在するかチェックします.
        /// </summary>
        /// <returns>接続ターゲットが存在する場合はtrue, 存在しない場合はfalseを返却</returns>
        public bool CheckTargetExistence()
        {
            bool ret = false;

            lock (this.activeConnecter)
            {
                ConnectionContext context = this.activeContext;

                // コンテキスがない，または準備できていない場合
                if (context == null || context.IsReady == false)
                {
                    // 接続できないのでfalseを返却.
                    return false;
                }

                // コネクターを使って接続ターゲットをチェック.
                ret = this.activeConnecter.CheckTargetExistence(context);
            }

            return ret;
        }

        /// <summary>
        /// 識別名からコネクターを取得します.
        /// </summary>
        /// <param name="name">識別名</param>
        /// <returns>コネクターを返却します</returns>
        public IConnecter GetConnecterByName(string name)
        {
            // 全てのコネクター分ループする.
            foreach (KeyValuePair<uint, IConnecter> connecter in this.connecters)
            {
                // 識別名が一致するかチェック.
                if (connecter.Value.Name == name)
                {
                    // 該当コネクタを返却.
                    return connecter.Value;
                }
            }

            // 無かったのでnullを返却.
            return null;
        }

        /// <summary>
        /// 登録IDからコネクターを取得します.
        /// </summary>
        /// <param name="id">登録ID</param>
        /// <returns>コネクターを返却します.</returns>
        public IConnecter GetConnecterById(uint id)
        {
            // 全てのコネクター分ループする.
            foreach (KeyValuePair<uint, IConnecter> connecter in this.connecters)
            {
                // 識別名が一致するかチェック.
                if (connecter.Key == id)
                {
                    // 該当コネクタを返却.
                    return connecter.Value;
                }
            }

            return null;
        }

        /// <summary>
        /// 識別名から現在のアクティブなコネクターを設定します.
        /// </summary>
        /// <param name="name">識別名</param>
        public void SetActiveConnecterByName(string name)
        {
            // 全てのコネクター分ループする.
            foreach (KeyValuePair<uint, IConnecter> connecter in this.connecters)
            {
                // 識別名が一致するかチェック.
                if (connecter.Value.Name == name)
                {
                    // 該当コネクタを返却.
                    this.activeConnecter = connecter.Value;

                    // アクティブコンテキストを設定.
                    this.activeContext.Copy(this.activeConnecter.ConnectionContext);

                    return;
                }
            }
        }

        /// <summary>
        /// 登録IDから現在のアクティブなコネクターを設定します.
        /// </summary>
        /// <param name="id">登録ID</param>
        public void SetActiveConnecterById(uint id)
        {
            // 全てのコネクター分ループする.
            foreach (KeyValuePair<uint, IConnecter> connecter in this.connecters)
            {
                // 識別名が一致するかチェック.
                if (connecter.Key == id)
                {
                    // 該当コネクタを設定.
                    this.activeConnecter = connecter.Value;

                    // アクティブコンテキストを設定.
                    this.activeContext.Copy(this.activeConnecter.ConnectionContext);

                    return;
                }
            }
        }

        /// <summary>
        /// ターゲットに接続します.
        /// </summary>
        /// <returns>接続に成功したらtrue, 接続に失敗したらfalseを返却します.</returns>
        public bool Connect()
        {
            bool result = false;

            lock (this.activeConnecter)
            {
                // アクティブコネクターがあるかチェック.
                if (this.activeConnecter != null && this.activeContext != null)
                {
                    // コネクターを使って接続.
                    result = this.activeConnecter.Connect(this.activeContext);
                    if (result)
                    {
                        // 内部フラグをONにする.
                        this.stateConnected = true;
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// ターゲットとの接続を切断します
        /// </summary>
        public void Disconnect()
        {
            lock (this.activeConnecter)
            {
                if (this.SendMessageCount > 0)
                {
                    // 溜まっているメッセージを切断前に送りつける.
                    this.ProcessSend();
                }

                // アクティブコネクターがあるかチェック.
                if (this.activeConnecter != null)
                {
                    // コネクターを使って切断.
                    this.activeConnecter.Disconnect();
                }
            }

            // 内部フラグをOFFにする.
            this.stateConnected = false;
        }

        /// <summary>
        /// メッセージキューにメッセージを追加します.
        /// </summary>
        /// <param name="message">メッセージです</param>
        public void PushMessage(Message message)
        {
            // 現在接続されていない，またはスレッドが無い場合.
            if (this.IsConnected == false || this.thread == null)
            {
                // 即終了.
                return;
            }

            // 排他制御
            lock (this.sendQueue)
            {
                // 送信キューに追加.
                this.sendQueue.Push(message);
            }
        }

        /// <summary>
        /// スレッドを起動する
        /// </summary>
        /// <returns>スレッドの起動に成功したらtrue, 失敗したらfalse</returns>
        public bool StartThread()
        {
            // 初期化処理がなされていない場合は、スレッドを起動しない.
            if (this.thread == null || this.activeContext == null)
            {
                Logger.Log(LogLevels.Warning, "MessageManager.StartThread() : MessageManager is not initialized.");
                return false;
            }

            // スレッドを起動.
            this.thread.Start();

            return true;
        }

        #endregion

        #region Private Methods

        /// <summary>
        /// 初期化処理を行います.
        /// </summary>
        private void Initialize()
        {
            // コンテキスト生成.
            this.activeContext = new ConnectionContext();

            // スレッド生成.
            this.thread = new Thread(new ThreadStart(this.ThreadProcess));

            // スレッドの設定.
            this.thread.IsBackground   = true;
            this.thread.Priority       = ThreadPriority.AboveNormal;
            this.thread.CurrentCulture = Thread.CurrentThread.CurrentUICulture;
            this.thread.Name           = "EffectMaker.CommunicatorThread";

            // スレッドを起動.
//            this.thread.Start();
        }

        /// <summary>
        /// メッセージキューのメッセージを取り出します.
        /// </summary>
        /// <returns>メッセージを返却します</returns>
        private Message PopMessage()
        {
            // 排他制御.
            lock (this.sendQueue)
            {
                // 送信キューから取り出し.
                return this.sendQueue.Pop();
            }
        }

        /// <summary>
        /// メッセージキューのメッセージをすべて取り出します.
        /// </summary>
        /// <returns>メッセージ配列を返却します</returns>
        private Message[] PopMessages()
        {
            // 排他制御.
            lock (this.sendQueue)
            {
                // 送信キューから一気に取り出し.
                return this.sendQueue.PopMessages();
            }
        }

        /// <summary>
        /// 接続状態を通知します.
        /// </summary>
        /// <param name="prevState">以前の接続状態</param>
        /// <param name="currState">現在の接続上</param>
        private void NotifyState(ConnectionStates prevState, ConnectionStates currState)
        {
            foreach (KeyValuePair<uint, IConnectionStateListener> key in this.connStateListeners)
            {
                // 状態を通知.
                key.Value.OnConnectionStateChanged(prevState, currState);
            }
        }

        /// <summary>
        /// メッセージを通知します.
        /// </summary>
        /// <param name="message">通知するメッセージ</param>
        private void NotifyMessage(Message message)
        {
            foreach (KeyValuePair<Guid, IMessageListener> notifiler in this.messageListeners)
            {
                notifiler.Value.OnMessageReceived(message);
            }
        }

        /// <summary>
        /// 接続状態をチェックします.
        /// </summary>
        /// <returns>接続可能であればtrueを返却します.</returns>
        private bool CheckState()
        {
            bool result = false;

            // PINGを送る.
            this.PingViewer();

            // 接続中かどうかチェック.
            if (this.IsConnected == true)
            {
                // 現在の状態.
                ConnectionStates curState = ConnectionStates.Connected;

                // 状態をチェック.
                switch (this.prevState)
                {
                    // 接続 ---> 接続.
                    case ConnectionStates.Connected:
                        if (this.activeConnecter != this.prevActiveConnecter)
                        {
                            this.NotifyState(this.prevState, curState);
                        }

                        break;

                    // 非接続 ---> 接続.
                    case ConnectionStates.Disconnected:
                        Thread.Sleep(this.connectWaitMsec);     // 少し寝かす.
                        this.NotifyState(this.prevState, curState);
                        break;

                    // エラー ---> 接続.
                    case ConnectionStates.Error:
                        this.NotifyState(this.prevState, curState);
                        break;
                }

                // 状態を更新.
                this.prevState = curState;

                this.prevActiveConnecter = this.activeConnecter;

                // 接続可能.
                result = true;
            }
            else
            {
                // コネクションをロストした場合.
                if (this.stateConnected == true)
                {
                    // 切断処理を実行.
                    this.Disconnect();
                }

                // 現在の状態.
                ConnectionStates curState = ConnectionStates.Disconnected;

                // 状態をチェック.
                switch (this.prevState)
                {
                    // 接続 ---> 非接続.
                    case ConnectionStates.Connected:
                        this.NotifyState(this.prevState, curState);
                        break;

                    // 非接続 ---> 非接続.
                    case ConnectionStates.Disconnected:
                        /* DO_NOTHING */
                        break;

                    // エラー ---> 非接続.
                    case ConnectionStates.Error:
                        this.NotifyState(this.prevState, curState);
                        break;
                }

                // 状態を更新.
                this.prevState = curState;

                this.prevActiveConnecter = this.activeConnecter;

                // 非接続可能.
                result = false;
            }

            // フラグを返却.
            return result;
        }

        /// <summary>
        /// 受信処理を行います.
        /// </summary>
        private void ProcessReceive()
        {
            // 非接続状態ならば即終了.
            if (this.IsConnected == false)
            {
                return;
            }

            try
            {
                // データを受信.
                byte[] packetBuffer;
                int packetSize = this.activeConnecter.Receive(out packetBuffer);

                // 受信データをチェック.
                if (packetSize <= 0 || packetBuffer == null || packetBuffer.Length <= 0)
                {
                    // サイズをゼロに.
                    packetSize = 0;
                }

                // 連結バッファ.
                byte[] combinedBuffer;

                // 持越し分のデータサイズ.
                int leftOverSize = 0;

                if (this.leftOverMessage != null)
                {
                    leftOverSize = this.leftOverMessage.Size;
                }

                if (leftOverSize <= 0 && packetSize > 0)
                {
                    // 受信したデータをそのまま入れる.
                    combinedBuffer = packetBuffer;
                }
                else if (leftOverSize > 0 && packetSize > 0)
                {
                    // バッファを確保.
                    combinedBuffer = new byte[leftOverSize + packetSize];

                    int idx = 0;

                    // 持越しバッファを入れる.
                    foreach (byte value in this.leftOverMessage.Buffer)
                    {
                        combinedBuffer[idx++] = value;
                    }

                    // 受信バッファを入れる.
                    foreach (byte value in packetBuffer)
                    {
                        combinedBuffer[idx++] = value;
                    }
                }
                else if (leftOverSize > 0 && packetSize <= 0)
                {
                    // 持越しバッファをそのままいれる.
                    combinedBuffer = this.leftOverMessage.Buffer;
                }
                else
                {
                    /* DO_NOTHING */
                    return;
                }

                if (combinedBuffer == null)
                {
                    return;
                }

                // バイトデータを読み取ってMessageに変換.
                {
                    Message msg = new Message();
                    int offset = 0;

                    // パケットが完全であると想定して実装.
                    // TCP/IPでは問題ないが, HIOでは問題あるかも...
                    do
                    {
                        //// =====================================
                        //// Protocol依存 ----->
                        int bufferSize = BinaryConversionUtility.ForProtocol.ConvertBack<int>(combinedBuffer, offset + 4);
                        bufferSize += 8;
                        //// Protocol依存 <-----
                        ////======================================

                        // メモリ確保.
                        msg.Buffer = new byte[bufferSize];

                        // データをコピる.
                        for (int i = 0; i < bufferSize; ++i)
                        {
                            msg.Buffer[i] = combinedBuffer[i + offset];
                        }

                        // サイズ設定.
                        msg.Size = msg.Buffer.Length;

                        // メッセージを通知.
                        this.NotifyMessage(msg);

                        // メッセージサイズ分だけオフセットを移動.
                        offset += msg.Size;

                        // メッセージをクリア.
                        msg.Clear();
                    }
                    while (offset < packetSize);

                    // メッセージが中途半端な場合.
                    if (msg.Size > 0 && msg.Buffer != null)
                    {
                        // 持越しメッセージを設定.
                        this.leftOverMessage = msg;
                    }
                }
            }
            catch (Exception exception)
            {
                // デバッグログを出力
                string message = exception.ToString();

                Logger.Log(LogLevels.Warning, "MessageManager.ProcessReceive() : Exception : {0}", message);
            }
        }

        /// <summary>
        /// 送信処理を行います.
        /// </summary>
        private void ProcessSend()
        {
            // 非接続状態ならば即終了.
            if (this.IsConnected == false)
            {
                return;
            }

            // メッセージが無い場合は即終了.
            if (this.HasSendMessage == false)
            {
                return;
            }

            // メッセージを取り出し.
            Message[] messages = this.PopMessages();

#if false
            // =====================================
            // Protocol依存 ----->
            if (messages.Length > 0)
            {
                int packetSize = 0;
                foreach (Message msg in messages)
                {
                    packetSize += msg.Size;
                }

                using (MemoryStream stream = new MemoryStream())
                {
                    using (BinaryWriter writer = new BinaryWriter(stream))
                    {
                        // 最大数を超えそうになったら，ゼロに戻す.
                        if (this.ProcessCount <= ushort.MaxValue)
                        {
                            this.ProcessCount = 0;
                        }

                        uint messageType = 1;     // Control Type
                        uint bufferSize  = 8;     // 8 = int:4byte : ushort:2byte + ushort:2byte
                        ushort numPackets = (ushort)messages.Length;
                        ushort processID = this.ProcessCount;

                        writer.Write(BinaryConversionUtility.Convert(messageType));              // MessageType.
                        writer.Write(BinaryConversionUtility.Convert(bufferSize));               // Buffer Size.
                        writer.Write(BinaryConversionUtility.Convert(packetSize));               // Total Packet Size.
                        writer.Write(BinaryConversionUtility.Convert(numPackets));               // Number of Packets.
                        writer.Write(BinaryConversionUtility.Convert(processID));                // Serial ID.

                        // 処理数をカウントアップ.
                        this.ProcessCount++;

                        // メッセージデータを取得.
                        byte[] buffer = stream.ToArray();

                        Message msg = new Message
                        {
                            Size = buffer.Length,
                            Buffer = buffer
                        };

                        // パケットを送信.
                        this.InnerSend(msg);
                    }
                }
            }
#endif

            //// Protocol依存 <-----
            //// =====================================

            // 各メッセージごとに処理.
            foreach (Message msg in messages)
            {
                this.InnerSend(msg);
            }
        }

        /// <summary>
        /// スレッド処理.
        /// </summary>
        private void ThreadProcess()
        {
            try
            {
                while (this.isTermThread == false)
                {
                    // アクティブなコネクターがないとき
                    if (this.activeConnecter == null)
                    {
                        // 待機
                        Thread.Sleep(this.sleepTimeMsec);
                        continue;
                    }

                    // 接続状態をチェックします.
                    if (this.CheckState())
                    {
                        // 受信処理.
                        this.ProcessReceive();

                        // 送信処理.
                        this.ProcessSend();
                    }
                    else
                    {
                        if (this.IsAutoConnectEnabled == true)
                        {
                            // 自動接続処理.
                            this.Connect();
                        }
                    }

                    // 接続されておらず, 自動接続が有効な場合.
                    if (this.IsConnected == false && this.IsAutoConnectEnabled == true)
                    {
                        // 少し寝かす.
                        Thread.Sleep(this.autoConnectTimeMsec);
                    }
                    else
                    {
                        // アクティブなコネクターがない，またはバッファがいっぱいの場合.
                        if (this.activeConnecter == null || this.activeConnecter.IsReceivedBufferFull == false)
                        {
                            // 少し寝かす.
                            Thread.Sleep(this.sleepTimeMsec);
                        }
                    }
                }
            }
            catch (ThreadAbortException)
            {
                /* DO_NOTHING */
                return;
            }
            catch (Exception exception)
            {
                if (this.isTermThread == false)
                {
                    // デバッグログを出力.
                    string output = exception.Message;

                    Logger.Log(LogLevels.Error, "MessageManager.ThreadProcess() : Exception: {0}", output);
                }

                // 切断処理.
                this.Disconnect();
            }
        }

        /// <summary>
        /// PINGメッセージをビューアに送る.
        /// </summary>
        private void PingViewer()
        {
            if (this.activeConnecter == null)
            {
                return;
            }

            ConnectionContext context = this.activeConnecter.ConnectionContext;
            if (context == null)
            {
                return;
            }

            int tickCount = Environment.TickCount;
            if ((tickCount - this.lastPingTime) < this.pingMsec)
            {
                return;
            }

            this.lastPingTime = tickCount;

            byte[] pingMsg = new byte[]
            {
                0x00, 0x00, 0x00, 0x00,
                0x00, 0x00, 0x00, 0x00,
            };

            bool result = this.activeConnecter.Send(pingMsg, pingMsg.Length);
        }

        /// <summary>
        /// パケット送信処理です
        /// </summary>
        /// <param name="msg">送信パケットです.</param>
        private void InnerSend(Message msg)
        {
            if (this.activeConnecter.Send(msg.Buffer, msg.Size) == false)
            {
                Logger.Log(LogLevels.Error, "MessageManager.InnerSend() : Send Error.");
            }
        }

        #endregion
    }
}
