﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

namespace EffectMaker.Communicator
{
    /// <summary>
    /// コネクターインタフェースです.
    /// </summary>
    public interface IConnecter : IDisposable
    {
        #region Public Properties

        /// <summary>
        /// 接続しているかどうか? 接続していればtrue, 非接続ならfalseです.
        /// </summary>
        bool IsConnected { get; }

        /// <summary>
        /// コネクター名
        /// </summary>
        string Name { get; }

        /// <summary>
        /// 受信データが受信バッファのサイズを超えるかどうか? 超える場合はtrue, 超えない場合はfalseです.
        /// </summary>
        bool IsReceivedBufferFull { get; }

        /// <summary>
        /// コネクションコンテキストを取得します.
        /// </summary>
        ConnectionContext ConnectionContext { get; }

        #endregion

        #region Public Methods

        /// <summary>
        /// 接続ターゲットが存在するかチェックします.
        /// </summary>
        /// <param name="context">コネクションコンテキストです</param>
        /// <returns>接続ターゲットが存在する場合はtrue, 存在しない場合はfalseを返却します.</returns>
        bool CheckTargetExistence(ConnectionContext context);

        /// <summary>
        /// ターゲットと接続します.
        /// </summary>
        /// <param name="context">コネクションコンテキストです</param>
        /// <returns>接続に成功したらtrue, 失敗したらfalseを返却します.</returns>
        bool Connect(ConnectionContext context);

        /// <summary>
        /// ターゲットと切断します.
        /// </summary>
        void Disconnect();

        /// <summary>
        /// ターゲットにパケットを送信します.
        /// </summary>
        /// <param name="buffer">送信パケット</param>
        /// <param name="bufferSize">送信パケットサイズ</param>
        /// <returns>送信に成功したらtrue, 失敗したらfalseを返却します</returns>
        bool Send(byte[] buffer, int bufferSize);

        /// <summary>
        /// ターゲットからパケットを受信します.
        /// </summary>
        /// <param name="buffer">受信パケット</param>
        /// <returns>受信したパケットサイズを返却します.</returns>
        int Receive(out byte[] buffer);

        #endregion
    }
}
