﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#ifdef NN_PCGX
#include "./pcgx_hio.h"
#endif

using namespace System;
using namespace System::IO;

namespace NintendoWare {
namespace HostIODriver {

    /*!--------------------------------------------------------------------------*
      @brief    シングルトンのオブジェクトを保持するクラスです。
     *---------------------------------------------------------------------------*/
    public ref class SingletonHolder
    {
    public:
        /*!
         *    @brief インスタンスを取得します。
         */
        static SingletonHolder^    GetInstance()
        {
            if (s_instance == nullptr){
                s_instance = gcnew SingletonHolder();
            }
            return s_instance;
        }

    public:
        /*!
         *    @brief CCTRHostIO オブジェクトを取得します。
         */
#ifdef NN_PCGX
        PCGX_HIODevice* GetDEVHIO() { return m_pDevhio; }
#else
        CCTRHostIO*     GetDEVHIO() { return m_pDevhio; }
#endif

        /*!
         *    @brief 接続状態のフラッグを取得します。
         */
        bool IsConnected() { return m_IsConnected; };

        /*!
         *    @brief ターゲットと接続をします。
         */
        bool ConnectHIO();

        /*!
         *    @brief    ターゲットと切断します。
         */
        bool DisconnectHIO();

        /*!
         * @brief 現在接続中のデバイスのIDを取得します。
         */
        unsigned long GetConnectID();

        /*!
         *    @brief ターゲットとシリアル通信回線を開きます。
         */
        bool OpenSerialIO( const int channel );

        /*!
         *    @brief    ターゲットとのシリアル通信回線を閉じます。
         */
        bool CloseSerialIO( const int channel );

        /*!
         *    @brief ターゲットとシリアル通信回線を接続します。
         */
        bool ConnectSerialIO( const int channel );

        /*!
         *    @brief    ターゲットとのシリアル通信回線を切断します。
         */
        bool DisconnectSerialIO( const int channel );

        /*!
         *    @brief    パケットをターゲットに送信します。
         */
        bool SendPacket( void* packet, const int packetSize, const int channel );

        /*!
         *    @brief    メモリーストリームを送信します。
         */
        bool Send(MemoryStream ^memoryStream, const int channel);

        /*!
         *    @brief    メモリーストリームを送信します。
         */
        bool Send( array<Byte> ^buffer, const int iSize, const int channel);

        /*!
         *    @brief    バッファをターゲットに送信します。
         */
        bool WriteSerialIO( array<Byte> ^buffer, const int size, const int channel );

        /*!
         * @brief    メモリーストリームを受信します。
         */
        bool Read(MemoryStream ^memoryStream, const int channel);

        /*!
         * @brief    バッファに受信します。
         */
        bool ReadSerialIO( array<Byte> ^buffer, const int size, const int channel );

        /*!
         * @brief    Check if there is any incoming data. The data size
         *          is returned if there is any, zero is returned if
         *          no incoming message in the driver buffer.
         */
        int GetIncomingDataSize(const int channel);

        /*!
         * @brief
         */
        int GetWritableDataSize(const int channel);

        /*!
         *    @brief シリアル通信をオープンします。
         */
        bool Open( const int channel )
        {
            if ( m_IsConnected==false )
            {
                if ( ConnectHIO()==false )
                    return false;
            }

            if ( OpenSerialIO(channel)==false )
                return false;

            if ( ConnectSerialIO(channel)==false )
                return false;

            return true;
        }

        /*!
         *    @brief シリアル通信をクローズします。
         */
        void Close( const int channel )
        {
            DisconnectSerialIO( channel );
            CloseSerialIO( channel );
        }

    public:
        /*!
         *    @brief デストラクタです。
         */
        ~SingletonHolder(){
            Relase();
        }

        /*!
         *    @brief ファイナライズです。
         */
        !SingletonHolder(){
            Relase();
        }

    private:
        /*!
         *    @brief コンストラクタです。シングルトンのため private に置きます。
         */
        SingletonHolder(){
            Create();
        }

        /*!
         *    @brief オブジェクトを生成します。
         */
        void Create(){
#ifdef NN_PCGX
            m_pDevhio = new PCGX_HIODevice;
#else
            m_pDevhio = new CCTRHostIO();
#endif
            m_IsConnected = false;
            ConnectHIO();
        }

        /*!
         *    @brief オブジェクトを破棄します。
         */
        void Relase(){
            if (m_pDevhio != NULL){
                DisconnectHIO();
                delete m_pDevhio;
                m_pDevhio = NULL;
            }
        }

    private:
        static SingletonHolder^    s_instance = nullptr;
        bool            m_IsConnected;
#ifdef NN_PCGX
        PCGX_HIODevice* m_pDevhio;
#else
        CCTRHostIO*     m_pDevhio;
#endif
    };

} // namesapce HostIODriver
} // namesapce NintendoWare
