﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;

namespace EffectMaker.Foundation.Serialization
{
    /// <summary>
    /// Helper class for operations needed for XmlDocSerialization.
    /// </summary>
    public static class XmlDocSerializationHelper
    {
        /// <summary>
        /// Enumerate all the elements with the specified tag name in the parent node.
        /// </summary>
        /// <param name="parentNode">The parent node.</param>
        /// <param name="tagName">The tag name of the elements to enumerate.</param>
        /// <returns>The elements.</returns>
        public static IEnumerable<XmlElement> EnumerateElementsByTagName(
            XmlNode parentNode,
            string tagName)
        {
            return EnumerateElementsByTagNames(parentNode, new string[] { tagName });
        }

        /// <summary>
        /// Enumerate all the elements with the specified tag name in the parent node.
        /// </summary>
        /// <param name="parentNode">The parent node.</param>
        /// <param name="tagNames">The tag names of the elements to enumerate.</param>
        /// <returns>The elements.</returns>
        public static IEnumerable<XmlElement> EnumerateElementsByTagNames(
            XmlNode parentNode,
            IEnumerable<string> tagNames)
        {
            var parentElement = parentNode as XmlElement;
            if (parentElement == null)
            {
                throw new ArgumentException("The parent node must to be a XML element.");
            }

            foreach (var child in parentElement.ChildNodes)
            {
                var element = child as XmlElement;
                if (element == null)
                {
                    continue;
                }

                if (tagNames.Any(n => n == element.Name) == true)
                {
                    yield return element;
                }
            }
        }

        /// <summary>
        /// Check if the given XML element contains an attribute "nil" and
        /// the value of the attribute is true.
        /// </summary>
        /// <param name="element">The XML element to check.</param>
        /// <returns>True if the element is a null element.</returns>
        public static bool IsNilElement(XmlElement element)
        {
            const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            string isNullStr = element.GetAttribute("nil", XsiNamespaceUri);

            bool isNull;
            if (string.IsNullOrEmpty(isNullStr) == false &&
                bool.TryParse(isNullStr, out isNull) == true &&
                isNull == true)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// Set the given XML element as empty. (add xsi:nil="true" attribute to it)
        /// </summary>
        /// <param name="element">The XML element.</param>
        public static void WriteNilElement(XmlElement element)
        {
            const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            var xsiNilAttribute = element.OwnerDocument.CreateAttribute("xsi", "nil", XsiNamespaceUri);
            xsiNilAttribute.Value = bool.TrueString;

            element.SetAttributeNode(xsiNilAttribute);
        }

        /// <summary>
        /// Check if the element has the xsi:type attribute, if it does, return the type.
        /// </summary>
        /// <param name="context">The data context.</param>
        /// <param name="element">The XML element to check.</param>
        /// <returns>The type if defined, otherwise null is returned.</returns>
        public static Type ReadElementType(
            XmlDocSerializationContext context,
            XmlElement element)
        {
            const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            string typeStr = element.GetAttribute("type", XsiNamespaceUri);
            if (string.IsNullOrEmpty(typeStr) == true)
            {
                return null;
            }

            Type type = context.FindExtraType(typeStr);

            return type;
        }

        /// <summary>
        /// Write xsi:type attribute to the given element so the serializer can
        /// deserialize the element with the correct type.
        /// </summary>
        /// <param name="element">The XML element.</param>
        /// <param name="type">The type.</param>
        public static void WriteElementType(XmlElement element, Type type)
        {
            const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

            var xsiTypeAttribute = element.OwnerDocument.CreateAttribute("xsi", "type", XsiNamespaceUri);
            xsiTypeAttribute.Value = type.Name;

            element.SetAttributeNode(xsiTypeAttribute);
        }

        /// <summary>
        /// Write xsi:type attribute to the given element so the serializer can
        /// deserialize the element with the correct type.
        /// </summary>
        /// <param name="writer">The XML writer at the element.</param>
        /// <param name="type">The type.</param>
        public static void WriteElementType(XmlWriter writer, Type type)
        {
            const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";
            writer.WriteAttributeString("xsi", "type", XsiNamespaceUri, type.Name);
        }

        /// <summary>
        /// Write a simple XML element that contains a single string value.
        /// </summary>
        /// <param name="context">The data context.</param>
        /// <param name="tagName">The XML element tag name.</param>
        /// <param name="value">The value.</param>
        /// <returns>True on success.</returns>
        public static bool WriteSimpleElement(
            XmlDocSerializationContext context,
            string tagName,
            string value)
        {
            var parentElement = context.CurrentNode as XmlElement;
            if (parentElement == null)
            {
                // Can only write to a parent element.
                return false;
            }

            var element = context.XmlDocument.CreateElement(tagName);
            element.InnerText = value;

            parentElement.AppendChild(element);

            return true;
        }
    }
}
