﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Xml.Serialization;

namespace EffectMaker.Foundation.Serialization
{
    /// <summary>
    /// A serializable version wrapper.
    /// </summary>
    [Serializable]
    public class VersionXml
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        public VersionXml()
        {
            this.Version = new Version();
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="version">The version to initialize from.</param>
        public VersionXml(Version version)
        {
            this.Version = (Version)version.Clone();
        }

        /// <summary>
        /// Constructor to initialize a new instance using the
        /// specified string.
        /// </summary>
        /// <param name="version">The version in string format.</param>
        public VersionXml(string version)
        {
            this.Version = new Version(version);
        }

        /// <summary>
        /// Constructor to initialize a new instance using the
        /// specified major and minor values.
        /// </summary>
        /// <param name="major">The major version.</param>
        /// <param name="minor">The minor version.</param>
        public VersionXml(int major, int minor)
        {
            this.Version = new Version(major, minor);
        }

        /// <summary>
        /// Constructor to initialize a new instance using the
        /// specified major, minor and build values.
        /// </summary>
        /// <param name="major">The major version.</param>
        /// <param name="minor">The minor version.</param>
        /// <param name="build">The build.</param>
        public VersionXml(int major, int minor, int build)
        {
            this.Version = new Version(major, minor, build);
        }

        /// <summary>
        /// Constructor to initialize a new instance using the
        /// specified major, minor, build and revision values.
        /// </summary>
        /// <param name="major">The major version.</param>
        /// <param name="minor">The minor version.</param>
        /// <param name="build">The build.</param>
        /// <param name="revision">The revision.</param>
        public VersionXml(int major, int minor, int build, int revision)
        {
            this.Version = new Version(major, minor, build, revision);
        }

        /// <summary>
        /// Get the version object.
        /// </summary>
        [XmlIgnore]
        public Version Version { get; private set; }

        /// <summary>
        /// Get or set the string representation of the version.
        /// </summary>
        [XmlText(typeof(string))]
        [EditorBrowsable(EditorBrowsableState.Never), Browsable(false)]
        public string Value
        {
            get
            {
                return this.Version.ToString();
            }

            set
            {
                Version temp;
                if (Version.TryParse(value, out temp) == false)
                {
                    throw new ArgumentException(value + " is of invalid version format.");
                }

                this.Version = temp;
            }
        }

        /// <summary>
        /// Implicitly convert VersionXml to Version.
        /// </summary>
        /// <param name="versionXml">The VersionXml instance to convert from.</param>
        /// <returns>The Version instance.</returns>
        public static implicit operator Version(VersionXml versionXml)
        {
            return versionXml.Version;
        }

        /// <summary>
        /// Implicitly convert Version to VersionXml.
        /// </summary>
        /// <param name="version">The Version instance to convert from.</param>
        /// <returns>The VersionXml instance.</returns>
        public static implicit operator VersionXml(Version version)
        {
            return new VersionXml(version);
        }
    }
}
