﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;

using EffectMaker.Foundation.Render.ObjectPicking;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Render.Renderable
{
    /// <summary>
    /// Class that represents the view port. (the rendering area)
    /// </summary>
    public class Viewport : RenderableBase
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="renderingControl">The control to render to.</param>
        public Viewport(Control renderingControl)
        {
            this.RenderingControl = renderingControl;
            this.RenderContext = new RenderContext(this);
            this.Bounds = Rectangle.Empty;
            this.BorderColor = Color.Transparent;
            this.BorderThickness = 0.0f;
            this.FillColor = Color.Transparent;
        }

        /// <summary>
        /// Get the control the viewport is rendering to.
        /// </summary>
        public Control RenderingControl { get; private set; }

        /// <summary>
        /// Get or set the render context.
        /// </summary>
        public RenderContext RenderContext { get; protected set; }

        /// <summary>
        /// Transform the given point to the viewport's world space.
        /// </summary>
        /// <param name="p">The point to transform.</param>
        /// <returns>The transformed point.</returns>
        public Point TransformPoint(Point p)
        {
            p.X = (int)(((float)p.X / this.Scale.X) - this.Translation.X - this.Bounds.Location.X);
            p.Y = (int)(((float)p.Y / this.Scale.Y) - this.Translation.Y - this.Bounds.Location.Y);

            return p;
        }

        /// <summary>
        /// Transform the given point to the viewport's world space.
        /// </summary>
        /// <param name="p">The point to transform.</param>
        /// <returns>The transformed point.</returns>
        public PointF TransformPoint(PointF p)
        {
            p.X = (p.X / this.Scale.X) - this.Translation.X - this.Bounds.Location.X;
            p.Y = (p.Y / this.Scale.Y) - this.Translation.Y - this.Bounds.Location.Y;

            return p;
        }

        /// <summary>
        /// Transform the given point to the viewport's world space.
        /// </summary>
        /// <param name="p">The point to transform.</param>
        /// <returns>The transformed point.</returns>
        public Point InverseTransformPoint(Point p)
        {
            p.X = (int)(((float)p.X + this.Bounds.Location.X + this.Translation.X) * this.Scale.X);
            p.Y = (int)(((float)p.Y + this.Bounds.Location.Y + this.Translation.Y) * this.Scale.Y);

            return p;
        }

        /// <summary>
        /// Transform the given point to the viewport's world space.
        /// </summary>
        /// <param name="p">The point to transform.</param>
        /// <returns>The transformed point.</returns>
        public PointF InverseTransformPoint(PointF p)
        {
            p.X = (p.X + this.Bounds.Location.X + this.Translation.X) * this.Scale.X;
            p.Y = (p.Y + this.Bounds.Location.Y + this.Translation.Y) * this.Scale.Y;

            return p;
        }

        /// <summary>
        /// Request redraw.
        /// </summary>
        public override void RequestRedraw()
        {
            this.IsRenderRequestedWhileSuspending = true;

            if (this.RenderingControl != null &&
                this.IsUpdateAndRenderingSuspended == false)
            {
                this.RenderingControl.Invalidate();
            }
        }

        /// <summary>
        /// Do picking on the renderables in the viewport.
        /// </summary>
        /// <param name="p">The point to pick with.</param>
        /// <returns>The pick context.</returns>
        public virtual ObjectPickContext DoPicking(PointF p)
        {
            var context = new ObjectPickContext(p);

            this.PickChildren(context);

            return context;
        }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        public virtual void Update(Graphics g)
        {
            this.RenderContext.Graphics = g;
            this.RenderContext.ViewRectangle = new Rectangle(
                (int)this.X,
                (int)this.Y,
                (int)this.Width,
                (int)this.Height);

            using (new PushPopRenderContext(this.RenderContext))
            {
                this.Update(this.RenderContext);
            }
        }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateSelf(RenderContext context)
        {
            this.TransformedBounds = new RectangleF(PointF.Empty, this.Size);
            this.ClippedBounds = this.TransformedBounds;

            // Update the children.
            foreach (RenderableBase child in this.Renderables)
            {
                child.Update(context);
            }
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            // Apply transformation to the render context.
            context.ApplyTransformation(this.Bounds.Location.X, this.Bounds.Location.Y);

            context.ApplyTransformation(this.Transformation);

            // Update the children.
            foreach (RenderableBase child in this.Renderables)
            {
                child.Update(context);
            }
        }

        /// <summary>
        /// Render the item's background.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBackground(Graphics g)
        {
            // Render the background.
            if (this.FillColor.Equals(Color.Transparent) == false)
            {
                using (var brush = new SolidBrush(this.FillColor))
                {
                    g.FillRectangle(brush, this.Bounds);
                }
            }
        }

        /// <summary>
        /// Render the item's border.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBorder(Graphics g)
        {
            // Render the border.
            if (this.BorderThickness > 0.0f && this.BorderColor.Equals(Color.Transparent) == false)
            {
                using (var pen = new Pen(this.BorderColor, this.BorderThickness))
                {
                    g.DrawRectangle(
                        pen,
                        this.X,
                        this.Y,
                        this.Width,
                        this.Height);
                }
            }
        }
    }
}
