﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;

using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Render.Renderable
{
    /// <summary>
    /// Class that renders a rounded rectangle.
    /// </summary>
    public class RoundedRectangleShape : RectangleShape
    {
        /// <summary>The path for rendering.</summary>
        private GraphicsPath renderPath = new GraphicsPath();

        /// <summary>
        /// Constructor.
        /// </summary>
        public RoundedRectangleShape() :
            base()
        {
            this.SetCornerRadius(0.0f);
        }

        /// <summary>
        /// Get or set the radius of the top-left corner.
        /// </summary>
        public float TopLeftCornerRadius { get; set; }

        /// <summary>
        /// Get or set the radius of the top-right corner.
        /// </summary>
        public float TopRightCornerRadius { get; set; }

        /// <summary>
        /// Get or set the radius of the bottom-left corner.
        /// </summary>
        public float BottomLeftCornerRadius { get; set; }

        /// <summary>
        /// Get or set the radius of the bottom-right corner.
        /// </summary>
        public float BottomRightCornerRadius { get; set; }

        /// <summary>
        /// Set corner radius.
        /// </summary>
        /// <param name="topLeft">The radius on the top-left corner.</param>
        /// <param name="topRight">The radius on the top-right corner.</param>
        /// <param name="bottomLeft">The radius on the bottom-left corner.</param>
        /// <param name="bottomRight">The radius on the bottom-right corner.</param>
        public void SetCornerRadius(
            float topLeft,
            float topRight,
            float bottomLeft,
            float bottomRight)
        {
            this.TopLeftCornerRadius = topLeft;
            this.TopRightCornerRadius = topRight;
            this.BottomLeftCornerRadius = bottomLeft;
            this.BottomRightCornerRadius = bottomRight;
        }

        /// <summary>
        /// Set a corner radius to all the corners.
        /// </summary>
        /// <param name="radius">The radius to set.</param>
        public void SetCornerRadius(float radius)
        {
            this.TopLeftCornerRadius = radius;
            this.TopRightCornerRadius = radius;
            this.BottomLeftCornerRadius = radius;
            this.BottomRightCornerRadius = radius;
        }

        /// <summary>
        /// Test if the given point lies on this object.
        /// </summary>
        /// <param name="p">The point.</param>
        /// <returns>True if the point picks this object.</returns>
        public override bool Pick(PointF p)
        {
            if (this.CanPick == false || this.Visible == false)
            {
                return false;
            }

            if (this.ClippedBounds.Contains(p) == false)
            {
                return false;
            }

            return this.renderPath.IsVisible(p);
        }

        /// <summary>
        /// Release the resources the renderable uses.
        /// </summary>
        /// <param name="isDisposing">True if the renderable is being disposed.</param>
        protected override void Dispose(bool isDisposing)
        {
            if (isDisposing == true && this.renderPath != null)
            {
                this.renderPath.Dispose();
                this.renderPath = null;
            }

            base.Dispose(isDisposing);
        }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateSelf(RenderContext context)
        {
            base.UpdateSelf(context);

            // First clear the path.
            this.renderPath.Reset();

            // Is there any rounded corner?
            if (this.TopLeftCornerRadius == 0.0f &&
                this.TopRightCornerRadius == 0.0f &&
                this.BottomLeftCornerRadius == 0.0f &&
                this.BottomRightCornerRadius == 0.0f)
            {
                return;
            }

            float top = Math.Min(this.Bounds.Top, this.Bounds.Bottom);
            float bottom = Math.Max(this.Bounds.Top, this.Bounds.Bottom);
            float left = Math.Min(this.Bounds.Left, this.Bounds.Right);
            float right = Math.Max(this.Bounds.Left, this.Bounds.Right);

            PointF[] points =
            {
                new PointF(left, top + this.TopLeftCornerRadius),
                new PointF(left + this.TopLeftCornerRadius, top),
                new PointF(right - this.TopRightCornerRadius, top),
                new PointF(right, top + this.TopRightCornerRadius),
                new PointF(right, bottom - this.BottomRightCornerRadius),
                new PointF(right - this.BottomRightCornerRadius, bottom),
                new PointF(left + this.BottomLeftCornerRadius, bottom),
                new PointF(left, bottom - this.BottomLeftCornerRadius)
            };

            // Transform the vertices with the transformation matrix.
            context.TransformationMatrix.TransformPoints(points);

            // Top-left corner.
            if (this.TopLeftCornerRadius > 0.0f)
            {
                this.renderPath.AddArc(
                    points[0].X,
                    points[1].Y,
                    (points[1].X - points[0].X) * 2.0f,
                    (points[0].Y - points[1].Y) * 2.0f,
                    180.0f,
                    90.0f);
            }
            else
            {
                this.renderPath.AddLine(points[1], points[2]);
            }

            // Top-right corner.
            if (this.TopRightCornerRadius > 0.0f)
            {
                this.renderPath.AddArc(
                    points[2].X - (points[3].X - points[2].X),
                    points[2].Y,
                    (points[3].X - points[2].X) * 2.0f,
                    (points[3].Y - points[2].Y) * 2.0f,
                    270.0f,
                    90.0f);
            }
            else
            {
                this.renderPath.AddLine(points[3], points[4]);
            }

            // Bottom-right corner.
            if (this.BottomRightCornerRadius > 0.0f)
            {
                this.renderPath.AddArc(
                    points[5].X - (points[4].X - points[5].X),
                    points[4].Y - (points[5].Y - points[4].Y),
                    (points[4].X - points[5].X) * 2.0f,
                    (points[5].Y - points[4].Y) * 2.0f,
                    0.0f,
                    90.0f);
            }
            else
            {
                this.renderPath.AddLine(points[5], points[6]);
            }

            // Bottom-left corner.
            if (this.BottomLeftCornerRadius > 0.0f)
            {
                this.renderPath.AddArc(
                    points[7].X,
                    points[7].Y - (points[6].Y - points[7].Y),
                    (points[6].X - points[7].X) * 2.0f,
                    (points[6].Y - points[7].Y) * 2.0f,
                    90.0f,
                    90.0f);
            }
            else
            {
                this.renderPath.AddLine(points[7], points[0]);
            }

            // Close the path.
            this.renderPath.CloseAllFigures();
        }

        /// <summary>
        /// Render the item's background.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBackground(Graphics g)
        {
            if (this.TopLeftCornerRadius == 0.0f &&
                this.TopRightCornerRadius == 0.0f &&
                this.BottomLeftCornerRadius == 0.0f &&
                this.BottomRightCornerRadius == 0.0f)
            {
                base.DrawBackground(g);
                return;
            }

            // Fill the rectangle.
            if (this.FillColor.Equals(Color.Transparent) == false)
            {
                using (var brush = new SolidBrush(this.FillColor))
                {
                    // TODO : View rectangle clipping!
                    g.FillPath(brush, this.renderPath);
                }
            }
        }

        /// <summary>
        /// Render the item's border.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBorder(Graphics g)
        {
            if (this.TopLeftCornerRadius == 0.0f &&
                this.TopRightCornerRadius == 0.0f &&
                this.BottomLeftCornerRadius == 0.0f &&
                this.BottomRightCornerRadius == 0.0f)
            {
                base.DrawBorder(g);
                return;
            }

            // Draw the border.
            if (this.BorderThickness > 0.0f &&
                this.BorderColor.Equals(Color.Transparent) == false)
            {
                using (var pen = new Pen(this.BorderColor, this.BorderThickness))
                {
                    // TODO : View rectangle clipping!
                    g.DrawPath(pen, this.renderPath);
                }
            }
        }
    }
}
