﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;

namespace EffectMaker.Foundation.FileMonitor
{
    /// <summary>
    /// モニターマネージャー
    /// </summary>
    public class FileMonitorManager
    {
        /// <summary>
        /// ファイル変更モニター辞書
        /// </summary>
        private readonly Dictionary<Key, FileChangedCallbackData> fileChangeCallbackData =
            new Dictionary<Key, FileChangedCallbackData>(new KeyCompare());

        /// <summary>
        /// パスごとにモニタ登録回数をカウントする辞書
        /// </summary>
        private readonly Dictionary<string, int> monitoringCount = null;

        /// <summary>
        /// アクセス不可パス
        /// </summary>
        private readonly HashSet<string> inaccessiblePaths = new HashSet<string>();

        /// <summary>
        /// イベント通知同期オブジェクト
        /// </summary>
        private ISynchronizeInvoke synchronizingObject;

        /// <summary>
        /// イベント通知同期オブジェクト
        /// </summary>
        private object fileChangeCallbackDataSyncObj = new object();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="synchronizingObject">イベント通知同期オブジェクト</param>
        /// <param name="countMode">イベント通知を1回だけに抑えるモードのON/OFF</param>
        public FileMonitorManager(ISynchronizeInvoke synchronizingObject, bool countMode = false)
        {
            this.synchronizingObject = synchronizingObject;

            if (countMode)
            {
                this.monitoringCount = new Dictionary<string, int>();
            }
        }

        /// <summary>
        /// ファイル変更時コールバック
        /// </summary>
        /// <param name="path">変更されたパス</param>
        /// <param name="userData">ユーザーデータ</param>
        /// <param name="changeType">変更の種類</param>
        public delegate void OnFileChangedCallback(string path, object userData, WatcherChangeTypes changeType);

        /// <summary>
        /// ファイルモニターの登録
        /// </summary>
        /// <param name="path">監視したいパス</param>
        /// <param name="userData">コールバック時に渡されるデータ</param>
        /// <param name="onFileChangedCallback">コールバックデリゲート</param>
        /// <param name="isDirectoryChildren">ディレクトリ内の子も監視する</param>
        /// <returns>登録が成功したかどうか</returns>
        public bool RegisterMonitor(string path, object userData, OnFileChangedCallback onFileChangedCallback, bool isDirectoryChildren = false)
        {
            lock (this.fileChangeCallbackDataSyncObj)
            {
                if (this.monitoringCount != null)
                {
                    if (this.monitoringCount.ContainsKey(path))
                    {
                        this.monitoringCount[path]++;
                        return true;
                    }

                    this.monitoringCount[path] = 1;
                }

                return this.RegisterMonitorCore(path, userData, onFileChangedCallback, isDirectoryChildren, false);
            }
        }

        /// <summary>
        /// ファイルモニターの登録(存在しないパスをターゲットにするアセット用)
        /// </summary>
        /// <param name="path">監視したいパス</param>
        /// <param name="userData">コールバック時に渡されるデータ</param>
        /// <param name="onFileChangedCallback">コールバックデリゲート</param>
        /// <param name="isDirectoryChildren">ディレクトリ内の子も監視する</param>
        /// <returns>登録が成功したかどうか</returns>
        public bool RegisterMonitorPreCopyPath(string path, object userData, OnFileChangedCallback onFileChangedCallback, bool isDirectoryChildren = false)
        {
            lock (this.fileChangeCallbackDataSyncObj)
            {
                if (this.monitoringCount != null)
                {
                    if (this.monitoringCount.ContainsKey(path))
                    {
                        this.monitoringCount[path]++;
                        return true;
                    }

                    this.monitoringCount[path] = 1;
                }

                return this.RegisterMonitorCore(path, userData, onFileChangedCallback, isDirectoryChildren, true);
            }
        }

        /// <summary>
        /// ファイルモニターの登録解除
        /// </summary>
        /// <param name="path">解除したいパス</param>
        /// <param name="isDirectoryChildren">ディレクトリ内の子も監視する</param>
        /// <returns>登録解除が成功したかどうか</returns>
        public bool UnregisterMonitor(string path, bool isDirectoryChildren = false)
        {
            Debug.Assert(string.IsNullOrEmpty(path) == false, "パスの指定が不正");
            //// 存在しなくなったファイルやパスの監視を解除することがありうるので、アサートを無効化
            ////Debug.Assert(File.Exists(path) || Directory.Exists(Path.GetDirectoryName(path)), "指定パスが見つからない");

            lock (this.fileChangeCallbackDataSyncObj)
            {
                if (this.monitoringCount != null)
                {
                    if (this.monitoringCount.ContainsKey(path))
                    {
                        this.monitoringCount[path]--;
                        if (this.monitoringCount[path] > 0)
                        {
                            return true;
                        }

                        this.monitoringCount.Remove(path);
                    }
                }

                var key = new Key(path, isDirectoryChildren);

                FileChangedCallbackData fileChangedCallbackData;
                if (this.fileChangeCallbackData.TryGetValue(key, out fileChangedCallbackData) == false)
                {
                    // 未登録あればなにもしない

                    ////Logger.Log("FileMonitorManager.UnregisterMonitor() : {0}, {1}", this.fileChangeCallbackData.Count(), path);
                    return false;
                }

                fileChangedCallbackData.Dispose();
                this.fileChangeCallbackData.Remove(key);

                ////Logger.Log("FileMonitorManager.UnregisterMonitor() : {0}, {1}", this.fileChangeCallbackData.Count(), path);
                return true;
            }
        }

        /// <summary>
        /// ファイルモニターの登録をすべて解除します。
        /// </summary>
        public void AllClearMonitor()
        {
            lock (this.fileChangeCallbackDataSyncObj)
            {
                // モニター登録回数をすべてクリア
                if (this.monitoringCount != null)
                {
                    this.monitoringCount.Clear();
                }

                // ファイル変更モニターをすべてクリア
                foreach (var item in this.fileChangeCallbackData)
                {
                    FileChangedCallbackData fileChangedCallbackData = item.Value;
                    fileChangedCallbackData.Dispose();
                }

                fileChangeCallbackData.Clear();
            }
        }

        /// <summary>
        /// ファイルモニターの登録
        /// </summary>
        /// <param name="path">監視したいパス</param>
        /// <param name="userData">コールバック時に渡されるデータ</param>
        /// <param name="onFileChangedCallback">コールバックデリゲート</param>
        /// <param name="isDirectoryChildren">ディレクトリ内の子も監視する</param>
        /// <param name="isNotYetExsitngPath">まだ存在しないパスをターゲットにするか否か</param>
        /// <returns>登録が成功したかどうか</returns>
        private bool RegisterMonitorCore(string path, object userData, OnFileChangedCallback onFileChangedCallback, bool isDirectoryChildren, bool isNotYetExsitngPath)
        {
            Debug.Assert(string.IsNullOrEmpty(path) == false, "パスの指定が不正");
            Debug.Assert(onFileChangedCallback != null, "コールバックデリゲートが不正");

            if (!isNotYetExsitngPath && this.IsAccessible(path) == false)
            {
                ////Logger.Log("FileMonitorManager.RegisterMonitor() : {0}, {1}", this.fileChangeCallbackData.Count(), path);
                return false;
            }

            var key = new Key(path, isDirectoryChildren);

            // 既に登録済みであれば登録しない
            if (this.fileChangeCallbackData.ContainsKey(key))
            {
                ////Logger.Log("FileMonitorManager.RegisterMonitor() : {0}, {1}", this.fileChangeCallbackData.Count(), path);
                return false;
            }

            // 登録する
            var fileMonitor = new FileMonitor(path, this.synchronizingObject, isDirectoryChildren);
            fileMonitor.FileChanged += (sender, args) =>
            {
                var currentMonitor = sender as FileMonitor;
                FileChangedCallbackData callbackData;

                if (currentMonitor != null && this.fileChangeCallbackData.TryGetValue(new Key(currentMonitor.Path, isDirectoryChildren), out callbackData))
                {
                    Debug.Assert(currentMonitor == callbackData.FileMonitor, "不正な状態。要調査");

                    callbackData.OnFileChangedCallback(currentMonitor.Path, callbackData.UserData, args.ChangeType);
                }
            };

            this.fileChangeCallbackData[key] = new FileChangedCallbackData
            {
                UserData = userData,
                OnFileChangedCallback = onFileChangedCallback,
                FileMonitor = fileMonitor
            };

            ////Logger.Log("FileMonitorManager.RegisterMonitor() : {0}, {1}", this.fileChangeCallbackData.Count(), path);
            return true;
        }

        /// <summary>
        /// アクセス可能か
        /// </summary>
        /// <param name="filepath">ファイルパス</param>
        /// <returns>アクセス可能なときtrueを返します。</returns>
        private bool IsAccessible(string filepath)
        {
            if (this.inaccessiblePaths.Contains(filepath))
            {
                return false;
            }

            try
            {
                if (File.Exists(filepath))
                {
                    using (var stream = File.Open(filepath, FileMode.Open, FileAccess.Read))
                    {
                    }
                }
                else
                {
                    Directory.GetAccessControl(filepath);
                }

                return true;
            }
            catch (Exception)
            {
                this.inaccessiblePaths.Add(filepath);

                return false;
            }
        }

        /// <summary>
        /// キー
        /// </summary>
        private class Key
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="path">ファイルパス</param>
            /// <param name="isDirectoryChildren">ディレクトリ内の子も監視する</param>
            public Key(string path, bool isDirectoryChildren)
            {
                this.Path                = path;
                this.IsDirectoryChildren = isDirectoryChildren;
            }

            /// <summary>
            /// ファイルパス
            /// </summary>
            public string Path { get; private set; }

            /// <summary>
            /// ディレクトリ内の子も監視する
            /// </summary>
            public bool IsDirectoryChildren { get; private set; }
        }

        /// <summary>
        /// キーコンペア
        /// </summary>
        private class KeyCompare : IEqualityComparer<Key>
        {
            /// <summary>
            /// キーの比較
            /// </summary>
            /// <param name="x">キー1</param>
            /// <param name="y">キー2</param>
            /// <returns>同じときtrueを返します。</returns>
            public bool Equals(Key x, Key y)
            {
                return
                    (x.Path                == y.Path) &&
                    (x.IsDirectoryChildren == y.IsDirectoryChildren);
            }

            /// <summary>
            /// ハッシュコードの取得
            /// </summary>
            /// <param name="obj">キー</param>
            /// <returns>ハッシュコードを返します。</returns>
            public int GetHashCode(Key obj)
            {
                return obj.Path.GetHashCode() ^ obj.IsDirectoryChildren.GetHashCode();
            }
        }

        /// <summary>
        /// ファイル変更時データ
        /// </summary>
        private class FileChangedCallbackData : IDisposable
        {
            /// <summary>
            /// ユーザーデータ
            /// </summary>
            public object UserData { get; set; }

            /// <summary>
            /// ファイル変更時コールバック
            /// </summary>
            public OnFileChangedCallback OnFileChangedCallback { get; set; }

            /// <summary>
            /// ファイルモニター
            /// </summary>
            public FileMonitor FileMonitor { get; set; }

            /// <summary>
            /// アンマネージ リソースの解放およびリセットに関連付けられているアプリケーション定義のタスクを実行します。
            /// </summary>
            public void Dispose()
            {
                if (FileMonitor != null)
                {
                    FileMonitor.Dispose();
                    FileMonitor = null;
                }
            }
        }
    }
}
