﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace EffectMaker.Foundation.Extensions
{
    /// <summary>
    /// Class containing extension methods related with the Control class.
    /// </summary>
    public static class ControlExtensions
    {
        /// <summary>
        /// Safely set a NumericUpDown value by clamping it if necessary.
        /// </summary>
        /// <param name="numericUpDown">The NumericUpDown control to set the value.</param>
        /// <param name="value">The value to set.</param>
        public static void SetValueClamped(this NumericUpDown numericUpDown, float value)
        {
            var v = new decimal(value);

            decimal min = numericUpDown.Minimum;
            decimal max = numericUpDown.Maximum;

            numericUpDown.Value = Math.Max(min, Math.Min(v, max));
        }

        /// <summary>
        /// Safely set a NumericUpDown value by clamping it if necessary.
        /// </summary>
        /// <param name="numericUpDown">The NumericUpDown control to set the value.</param>
        /// <param name="value">The value to set.</param>
        public static void SetValueClamped(this NumericUpDown numericUpDown, double value)
        {
            var v = new decimal(value);

            decimal min = numericUpDown.Minimum;
            decimal max = numericUpDown.Maximum;

            numericUpDown.Value = Math.Max(min, Math.Min(v, max));
        }

        /// <summary>
        /// Safely set a NumericUpDown value by clamping it if necessary.
        /// </summary>
        /// <param name="numericUpDown">The NumericUpDown control to set the value.</param>
        /// <param name="value">The value to set.</param>
        public static void SetValueClamped(this NumericUpDown numericUpDown, int value)
        {
            var v = new decimal(value);

            decimal min = numericUpDown.Minimum;
            decimal max = numericUpDown.Maximum;

            numericUpDown.Value = Math.Max(min, Math.Min(v, max));
        }

        /// <summary>
        /// Safely set a NumericUpDown value by clamping it if necessary.
        /// </summary>
        /// <param name="numericUpDown">The NumericUpDown control to set the value.</param>
        /// <param name="value">The value to set.</param>
        public static void SetValueClamped(this NumericUpDown numericUpDown, uint value)
        {
            var v = new decimal(value);

            decimal min = numericUpDown.Minimum;
            decimal max = numericUpDown.Maximum;

            numericUpDown.Value = Math.Max(min, Math.Min(v, max));
        }

        /// <summary>
        /// Safely set a NumericUpDown value by clamping it if necessary.
        /// </summary>
        /// <param name="numericUpDown">The NumericUpDown control to set the value.</param>
        /// <param name="value">The value to set.</param>
        public static void SetValueClamped(this NumericUpDown numericUpDown, long value)
        {
            var v = new decimal(value);

            decimal min = numericUpDown.Minimum;
            decimal max = numericUpDown.Maximum;

            numericUpDown.Value = Math.Max(min, Math.Min(v, max));
        }

        /// <summary>
        /// Safely set a NumericUpDown value by clamping it if necessary.
        /// </summary>
        /// <param name="numericUpDown">The NumericUpDown control to set the value.</param>
        /// <param name="value">The value to set.</param>
        public static void SetValueClamped(this NumericUpDown numericUpDown, ulong value)
        {
            var v = new decimal(value);

            decimal min = numericUpDown.Minimum;
            decimal max = numericUpDown.Maximum;

            numericUpDown.Value = Math.Max(min, Math.Min(v, max));
        }

        /// <summary>
        /// Displays the window on a different screen than the root control, if possible.
        /// </summary>
        /// <param name="form">The form to open on a different screen.</param>
        /// <param name="mainControl">The main control that defines the primary screen.</param>
        public static void ShowOnDifferentScreen(this Form form, Control mainControl)
        {
            if (form == null)
            {
                throw new ArgumentNullException("form");
            }

            form.ShowOnDifferentScreen(mainControl, null);
        }

        /// <summary>
        /// Displays the window on a different screen than the root control, if possible.
        /// </summary>
        /// <param name="form">The form to open on a different screen.</param>
        /// <param name="mainControl">The main control that defines the primary screen.</param>
        /// <param name="owner">The owner window.</param>
        public static void ShowOnDifferentScreen(this Form form, Control mainControl, IWin32Window owner)
        {
            if (Screen.AllScreens.Length <= 1)
            {
                form.Show(owner);
            }
            else
            {
                var mainScreen = Screen.FromControl(mainControl);
                Screen otherScreen;

                if (mainScreen.Primary)
                {
                    otherScreen = Screen.AllScreens[1];
                }
                else
                {
                    otherScreen = Screen.PrimaryScreen;
                }

                form.Show(owner);

                form.Left += otherScreen.Bounds.X - mainScreen.Bounds.X;
                form.Top += otherScreen.Bounds.Y - mainScreen.Bounds.Y;
            }
        }

        /// <summary>
        /// Adjust the bounds so it can fit to the visible screen area.
        /// </summary>
        /// <param name="form">The form to show.</param>
        /// <param name="bounds">The location and size of the form to show.</param>
        /// <returns>The adjusted location and size.</returns>
        public static Rectangle AdjustBoundsToVisibleScreenArea(
            this Form form,
            Rectangle bounds)
        {
            Rectangle result = bounds;

            // Compute the complete visible screen area.
            Rectangle visibleScreenArea = new Rectangle();
            foreach (Screen screen in Screen.AllScreens)
            {
                if (visibleScreenArea.IsEmpty == true)
                {
                    visibleScreenArea.X = screen.Bounds.X;
                    visibleScreenArea.Y = screen.Bounds.Y;
                    visibleScreenArea.Width = screen.Bounds.Width;
                    visibleScreenArea.Height = screen.Bounds.Height;
                }
                else
                {
                    visibleScreenArea = Rectangle.Union(visibleScreenArea, screen.Bounds);
                }
            }

            // Does the bounds go out of the working area ( horizontally )?
            if (result.Right > visibleScreenArea.Right)
            {
                result.X =
                    visibleScreenArea.Right - Math.Min(visibleScreenArea.Width, result.Width);
            }
            else if (result.Left < visibleScreenArea.Left)
            {
                result.X = visibleScreenArea.Left;
            }

            // Does the bounds go out of the working area ( vertically )?
            if (result.Bottom > visibleScreenArea.Bottom)
            {
                result.Y =
                    visibleScreenArea.Bottom - Math.Min(visibleScreenArea.Height, form.Height);
            }
            else if (result.Top < visibleScreenArea.Top)
            {
                result.Y = visibleScreenArea.Top;
            }

            // Check the width of the bounds
            if (result.Width > visibleScreenArea.Width)
            {
                result.Width = visibleScreenArea.Width;
            }

            // Check the height of the bounds
            if (result.Height > visibleScreenArea.Height)
            {
                result.Height = visibleScreenArea.Height;
            }

            return result;
        }

        /// <summary>
        /// Displays the window with the specified location and size.
        /// </summary>
        /// <param name="form">The form to show.</param>
        /// <param name="owner">The owner window.</param>
        /// <param name="bounds">The location and size of the form to show.</param>
        public static void ShowWithBounds(
            this Form form,
            IWin32Window owner,
            Rectangle bounds)
        {
            form.Bounds = AdjustBoundsToVisibleScreenArea(form, bounds);
            form.Show(owner);
        }
    }
}
