﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace EffectMaker.Foundation.Core
{
    /// <summary>
    /// Represent information containing path lookup.
    /// </summary>
    public struct PathLookupInfo : IEquatable<PathLookupInfo>
    {
        /// <summary>
        /// Initializes the PathLookupInfo instance.
        /// Does not include sub directories and search pattern is *.*
        /// </summary>
        /// <param name="path">Absolute base path.</param>
        public PathLookupInfo(string path)
            : this(path, false)
        {
        }

        /// <summary>
        /// Initializes the PathLookupInfo instance.
        /// Does not include sub directories.
        /// </summary>
        /// <param name="path">Absolute base path.</param>
        /// <param name="searchPatterns">Search patterns to lookup.</param>
        public PathLookupInfo(string path, params string[] searchPatterns)
            : this(path, false, searchPatterns)
        {
        }

        /// <summary>
        /// Initializes the PathLookupInfo instance.
        /// Search pattern is *.*
        /// </summary>
        /// <param name="path">Absolute base path.</param>
        /// <param name="includeSubDirectories">Tells whether to include sub directories
        /// in the lookup or not.</param>
        public PathLookupInfo(string path, bool includeSubDirectories)
            : this(path, includeSubDirectories, new string[0])
        {
        }

        /// <summary>
        /// Initializes the PathLookupInfo instance.
        /// </summary>
        /// <param name="path">Absolute base path.</param>
        /// <param name="includeSubDirectories">Tells whether to include sub directories
        /// in the lookup or not.</param>
        /// <param name="searchPatterns">Search patterns to lookup.</param>
        public PathLookupInfo(string path, bool includeSubDirectories, params string[] searchPatterns)
            : this()
        {
            if (string.IsNullOrWhiteSpace(path))
            {
                throw new ArgumentException("Invalid 'path' argument.", "path");
            }

            this.Path = path;
            this.SearchPatterns = searchPatterns.DefaultIfEmpty("*.*").ToArray();
            this.IncludeSubDirectories = includeSubDirectories;
        }

        /// <summary>
        /// Gets the absolute base path.
        /// </summary>
        public string Path { get; private set; }

        /// <summary>
        /// Gets the search patterns.
        /// </summary>
        public string[] SearchPatterns { get; private set; }

        /// <summary>
        /// Gets whether sub directories are included in the lookup.
        /// </summary>
        public bool IncludeSubDirectories { get; private set; }

        /// <summary>
        /// Construct a PathLookupInfo instance from a clipboard form text.
        /// </summary>
        /// <param name="clipboardString">Clipboard form text to parse.</param>
        /// <param name="pathLookupInfo">Retrieves a PathLookupInfo instance.</param>
        /// <returns>Returns bool if the text could be parsed, false otherwise.</returns>
        public static bool FromClipboardString(
            string clipboardString,
            out PathLookupInfo pathLookupInfo)
        {
            pathLookupInfo = new PathLookupInfo();

            if (string.IsNullOrWhiteSpace(clipboardString))
            {
                return false;
            }

            var elements = clipboardString.Split(new[] { '|' }, StringSplitOptions.None);
            if (elements.Length != 3)
            {
                return false;
            }

            bool subdirs;
            if (bool.TryParse(elements[1].Trim(), out subdirs) == false)
            {
                return false;
            }

            pathLookupInfo = new PathLookupInfo(
                elements[0].Trim(),
                subdirs,
                elements[2].Trim().Split(',', ';', ' '));

            return true;
        }

        /// <summary>
        /// Check for equality of two instances.
        /// </summary>
        /// <param name="t1">Left hand side instance.</param>
        /// <param name="t2">Right hand side instance.</param>
        /// <returns>Returns true if equals, false otherwise.</returns>
        public static bool operator ==(PathLookupInfo t1, PathLookupInfo t2)
        {
            return t1.Equals(t2);
        }

        /// <summary>
        /// Check for inequality of two instances.
        /// </summary>
        /// <param name="t1">Left hand side instance.</param>
        /// <param name="t2">Right hand side instance.</param>
        /// <returns>Returns true if different, false otherwise.</returns>
        public static bool operator !=(PathLookupInfo t1, PathLookupInfo t2)
        {
            return !(t1 == t2);
        }

        /// <summary>
        /// Provide a textual representation of the current instance.
        /// </summary>
        /// <returns>Returns a textual representation.</returns>
        public override string ToString()
        {
            return string.Format(
                "{0} : {1}{2}",
                this.Path,
                string.Join(", ", this.SearchPatterns),
                this.IncludeSubDirectories ? " ; recursive" : string.Empty);
        }

        /// <summary>
        /// Provides a textual representation for clipboard purpose.
        /// </summary>
        /// <returns>Returns a textual representation for clipboard.</returns>
        public string ToClipboardString()
        {
            return string.Format(
                "{0}|{1}|{2}",
                this.Path,
                this.IncludeSubDirectories,
                string.Join(";", this.SearchPatterns));
        }

        /// <summary>
        /// Check for equality with another PathLookupInfo instance.
        /// </summary>
        /// <param name="other">The other PathLookupInfo to check the equality upon.</param>
        /// <returns>Returns true if equals, false otherwise.</returns>
        public bool Equals(PathLookupInfo other)
        {
            try
            {
                return string.Compare(
                    System.IO.Path.GetFullPath(this.Path).TrimEnd('\\'),
                    System.IO.Path.GetFullPath(other.Path).TrimEnd('\\'),
                    StringComparison.InvariantCultureIgnoreCase) == 0 &&
                    this.SearchPatterns.SequenceEqual(other.SearchPatterns) &&
                    this.IncludeSubDirectories == other.IncludeSubDirectories;
            }
            catch (ArgumentException)
            {
                return false;
            }
            catch (System.Security.SecurityException)
            {
                return false;
            }
        }

        /// <summary>
        /// Check for equality with another instance.
        /// Directly returns false if instance is not of type PathLookupInfo.
        /// </summary>
        /// <param name="obj">Instance to check for equality upon.</param>
        /// <returns>Returns true if equals, false otherwise.</returns>
        public override bool Equals(object obj)
        {
            if ((obj is PathLookupInfo) == false)
            {
                return false;
            }

            return this.Equals((PathLookupInfo)obj);
        }

        /// <summary>
        /// Gets the hash code of this instance, based on
        /// path, sub directories inclusion and search patterns.
        /// </summary>
        /// <returns>Returns the correponding hash code of the instance.</returns>
        public override int GetHashCode()
        {
            return string.Format(
                "{0}{1}{2}",
                this.Path,
                string.Concat(this.SearchPatterns),
                this.IncludeSubDirectories)
                    .GetHashCode();
        }
    }
}
