﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

namespace EffectMaker.Foundation.Command
{
    /// <summary>
    /// コマンドを管理するクラス.
    /// </summary>
    public class CommandStack : IDisposable
    {
        #region Member variables

        /// <summary>The undo stack.</summary>
        private readonly List<CommandBase> undoBuffer = new List<CommandBase>();

        /// <summary>The redo stack.</summary>
        private readonly List<CommandBase> redoBuffer = new List<CommandBase>();

        /// <summary>The tagret object.</summary>
        private object targetObject = null;

        /// <summary>
        /// Undo/Redoバッファの最大サイズ.
        /// -1なら最大サイズは無限.
        /// </summary>
        private int undoRedoMaximum = -1;

        #endregion // Member variables

        /// <summary>
        /// コンストラクタ.
        /// </summary>
        /// <param name="targetObject">ターゲットオブジェクト.</param>
        /// <param name="undoRedoMaximum">Undo/Redoバッファの最大サイズ.-1なら無限サイズ.</param>
        public CommandStack(object targetObject, int undoRedoMaximum)
        {
            this.targetObject = targetObject;
            this.undoRedoMaximum = undoRedoMaximum;
        }

        /// <summary>
        /// Undoバッファのデータ取得
        /// </summary>
        public IEnumerable<CommandBase> UndoBuffer
        {
            get
            {
                return this.undoBuffer;
            }

            private set
            {
            }
        }

        /// <summary>
        /// Redoバッファのデータ取得
        /// </summary>
        public IEnumerable<CommandBase> RedoBuffer
        {
            get
            {
                return this.redoBuffer;
            }

            private set
            {
            }
        }

        /// <summary>
        /// Dispose.
        /// </summary>
        public void Dispose()
        {
            this.targetObject = null;
        }

        /// <summary>
        /// アクティブターゲットオブジェクトを取得する.
        /// </summary>
        /// <returns>アクティブターゲットオブジェクト.</returns>
        public object GetTargetObject()
        {
            return this.targetObject;
        }

        /// <summary>
        /// コマンドをプッシュする.
        /// </summary>
        /// <param name="cmd">コマンド.</param>
        /// <returns>成功した場合、[true]を返す.</returns>
        public bool PushCommand(CommandBase cmd)
        {
            // 1番最後に追加したコマンドに続かないものであれば、バッファの容量を制限する.
            // ※ １番最後に追加したコマンドが、KeepExecuting=trueになっていたら後続のコマンドもあるのでそのまま追加する.
            if (this.undoBuffer.Count > 0 && this.undoBuffer[0].KeepExecuting == false)
            {
                // undoBufferのサイズがUndo/Redoバッファの最大数を上回っていたら、削除する
                if (this.undoRedoMaximum != -1 && this.undoBuffer.Count >= this.undoRedoMaximum)
                {
                    int n = 0;
                    bool keepExecuting = false;

                    // 1番最後のコマンドから数える.連結されたコマンドの場合、連結分のコマンドをまとめて削除する.
                    for (int i = this.undoBuffer.Count - 1; i >= 0; i--)
                    {
                        if (this.undoBuffer[i].KeepExecuting == true)
                        {
                            keepExecuting = true;
                        }
                        else
                        {
                            keepExecuting = false;
                        }

                        if (i < this.undoRedoMaximum && keepExecuting == false)
                        {
                            n = i;
                            break;
                        }
                    }

                    // 古いコマンドを削除
                    this.undoBuffer.RemoveRange(n, this.undoBuffer.Count - n);
                }
            }

            // Undo用リストの最初へ追加する.
            this.undoBuffer.Insert(0, cmd);

            // Redo用バッファをクリアする.
            this.ClearRedoBuffer();

            return true;
        }

        /// <summary>
        /// Undoのためのコマンド操作.
        /// </summary>
        /// <returns>成功した場合、コマンドインスタンスを返す.</returns>
        public CommandBase GetCommandForUndo()
        {
            // アンドゥスタックに履歴があるかチェックする.
            if (this.undoBuffer.Count == 0)
            {
                return null;
            }

            // Undo用リストの最初の要素を、Redo用リストの最初へ移す.
            CommandBase cmd = this.undoBuffer[0];
            this.undoBuffer.RemoveAt(0);
            this.redoBuffer.Insert(0, cmd);

            return cmd;
        }

        /// <summary>
        /// 次にUndoされるコマンドを取得する(PopやPushはしない).
        /// </summary>
        /// <returns>成功した場合、コマンドインスタンスを返す.</returns>
        public CommandBase GetNextUndoCommand()
        {
            // アンドゥスタックに履歴があるかチェックする.
            if (this.undoBuffer.Count == 0)
            {
                return null;
            }

            // Undo用Listの最初の要素を返す.
            CommandBase cmd = this.undoBuffer[0];

            return cmd;
        }

        /// <summary>
        /// Redoのためのコマンド操作.
        /// </summary>
        /// <returns>成功した場合、コマンドインスタンスを返す.</returns>
        public CommandBase GetCommandForRedo()
        {
            // Redoスタックに履歴があるかチェックする.
            if (this.redoBuffer.Count == 0)
            {
                return null;
            }

            // Redo用リストの最初の要素を、Undo用リストの最初へ移す
            CommandBase cmd = this.redoBuffer[0];
            this.redoBuffer.RemoveAt(0);
            this.undoBuffer.Insert(0, cmd);

            return cmd;
        }

        /// <summary>
        /// 次にRedoされるコマンドを取得する(PopやPushはしない).
        /// </summary>
        /// <returns>成功した場合、コマンドインスタンスを返す.</returns>
        public CommandBase GetNextRedoCommand()
        {
            // アンドゥスタックに履歴があるかチェックする.
            if (this.redoBuffer.Count == 0)
            {
                return null;
            }

            // Redo用Listの最初の要素を返す
            CommandBase cmd = this.redoBuffer[0];

            return cmd;
        }

        /// <summary>
        /// 元に戻す用バッファをクリアする.
        /// </summary>
        /// <returns>クリアしたコマンドの数.</returns>
        public int ClearUndoBuffer()
        {
            int count = this.undoBuffer.Count;
            this.undoBuffer.Clear();
            return count;
        }

        /// <summary>
        /// やり直し用バッファをクリアする.
        /// </summary>
        /// <returns>クリアしたコマンドの数.</returns>
        public int ClearRedoBuffer()
        {
            int count = this.redoBuffer.Count;
            this.redoBuffer.Clear();
            return count;
        }
    }
}
