﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.Foundation.Collections
{
    /// <summary>
    /// Holds a pair of strings.
    /// </summary>
    public class StringPair : ICloneable, ISettable, IXmlDocSerializable
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        public StringPair()
        {
            this.Key = string.Empty;
            this.Value = string.Empty;
        }

        /// <summary>
        /// Constructor with values.
        /// </summary>
        /// <param name="key">The string as the key.</param>
        /// <param name="value">The string as the value.</param>
        public StringPair(string key, string value)
        {
            this.Key = key;
            this.Value = value;
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source string pair.</param>
        public StringPair(StringPair src)
        {
            this.Set(src);
        }

        /// <summary>
        /// Get or set the key string.
        /// </summary>
        public string Key { get; set; }

        /// <summary>
        /// Get or set the value string.
        /// </summary>
        public string Value { get; set; }

        /// <summary>
        /// Implicitly converts a KeyValuePair to StringPair.
        /// </summary>
        /// <param name="src">The source KeyValuePair.</param>
        /// <returns>The StringPair.</returns>
        public static implicit operator StringPair(KeyValuePair<string, string> src)
        {
            return new StringPair(src.Key, src.Value);
        }

        /// <summary>
        /// Implicitly converts a StringPair to KeyValuePair.
        /// </summary>
        /// <param name="src">The source StringPair.</param>
        /// <returns>The KeyValuePair.</returns>
        public static implicit operator KeyValuePair<string, string>(StringPair src)
        {
            return new KeyValuePair<string, string>(src.Key, src.Value);
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.Key = this.ReadElement(context, "Key", this.Key);
            this.Value = this.ReadElement(context, "Value", this.Value);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "Key", this.Key);
            this.WriteElement(context, "Value", this.Value);

            return true;
        }

        /// <summary>
        /// Copy the values in the source to this instance.
        /// </summary>
        /// <param name="src">The source object.</param>
        /// <returns>True on success.</returns>
        public bool Set(object src)
        {
            var pair = src as StringPair;
            if (pair == null)
            {
                if (src is KeyValuePair<string, string>)
                {
                    pair = (KeyValuePair<string, string>)src;
                }
                else
                {
                    return false;
                }
            }

            this.Key = pair.Key;
            this.Value = pair.Value;

            return true;
        }

        /// <summary>
        /// Clone the instance.
        /// </summary>
        /// <returns>The created instance.</returns>
        public object Clone()
        {
            return new StringPair(this);
        }
    }
}
