﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.Foundation.Collections.Generic
{
    /// <summary>
    /// 配列コレクションクラスです.
    /// </summary>
    /// <typeparam name="T">データ型です.</typeparam>
    public class ArrayCollection<T> : IXmlSerializable,
                                      IXmlDocSerializable,
                                      ISettable,
                                      ICloneable,
                                      ICollection<T>
    {
        /// <summary>
        /// 配列です.
        /// </summary>
        private T[] items = null;

        /// <summary>
        /// デフォルトコンストラクタ.
        /// </summary>
        public ArrayCollection() :
            this(0)
        {
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="arraySize">配列サイズです.</param>
        public ArrayCollection(int arraySize)
        {
            this.items = new T[arraySize];
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source array collection to copy from.</param>
        public ArrayCollection(ArrayCollection<T> src)
        {
            this.items = new T[src.Count];
            this.Set(src);
        }

        /// <summary>
        /// 配列の要素数を取得します.
        /// </summary>
        public int Count
        {
            get { return this.items.Length; }
        }

        /// <summary>
        /// 固定サイズかどうかを取得します.
        /// </summary>
        public bool IsFixedSize
        {
            get { return true; }
        }

        /// <summary>
        /// 読み取り専用かどうかを取得します.
        /// </summary>
        public bool IsReadOnly
        {
            get { return false; }
        }

        /// <summary>
        /// インデクサです.
        /// </summary>
        /// <param name="index">配列のインデックス.</param>
        /// <returns>指定されたインデックスの値を返却します.</returns>
        public T this[int index]
        {
            get { return this.items[index]; }
            set { this.items[index] = value; }
        }

        /// <summary>
        /// 配列にコピーします.
        /// </summary>
        /// <param name="array">コピー先.</param>
        /// <param name="arrayIndex">コピー開始のインデックス.</param>
        public void CopyTo(T[] array, int arrayIndex)
        {
            Array.Copy(this.items, 0, array, arrayIndex, this.items.Length);
        }

        /// <summary>
        /// ある要素が存在するかどうかを判断します.
        /// </summary>
        /// <param name="value">検索するオブジェクト.</param>
        /// <returns>存在する場合は true。それ以外の場合は false。</returns>
        public bool Contains(T value)
        {
            if (this.items == null || this.Count <= 0)
            {
                return false;
            }

            return this.items.Any(it => object.Equals(it, value));
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します.
        /// </summary>
        /// <returns>コレクションを反復処理するために使用できる System.Collections.IEnumerator オブジェクト.</returns>
        public System.Collections.Generic.IEnumerator<T> GetEnumerator()
        {
            return this.items.Cast<T>().GetEnumerator();
        }

        /// <summary>
        /// 値を設定します.
        /// </summary>
        /// <param name="src">入力データ.</param>
        /// <returns>値の設定に成功したらtrueを返却します.</returns>
        public bool Set(object src)
        {
            var srcData = src as ArrayCollection<T>;
            if (srcData == null)
            {
                return false;
            }

            if (this.Count != srcData.Count)
            {
                this.items = new T[srcData.Count];
            }

            Array.Copy(srcData.items, this.items, srcData.Count);

            return true;
        }

        /// <summary>
        /// 値を比較します.
        /// </summary>
        /// <param name="obj">比較データ.</param>
        /// <returns>値が一致したらtrueを返却します.</returns>
        public override bool Equals(object obj)
        {
            var array = obj as ArrayCollection<T>;
            if (array == null || array.Count != this.Count)
            {
                return false;
            }

            for (int i = 0; i < this.Count; ++i)
            {
                if (object.Equals(this.items[i], array.items[i]) == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// ハッシュコードを取得します.
        /// </summary>
        /// <returns>ハッシュコードを返却します.</returns>
        public override int GetHashCode()
        {
            return this.items.GetHashCode();
        }

        /// <summary>
        ///  現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>現在のインスタンスのコピーである新しいオブジェクト。</returns>
        public object Clone()
        {
            ArrayCollection<T> result = new ArrayCollection<T>(this);
            return result;
        }

        /// <summary>
        /// Read the data model from XML document.
        /// </summary>
        /// <param name="context">Holds the data needed for deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            var element = context.CurrentNode as XmlElement;
            if (element == null)
            {
                return false;
            }

            this.items = this.ReadArrayElement<T>(context, element, this.items);

            return true;
        }

        /// <summary>
        /// Write the data model to XML document.
        /// </summary>
        /// <param name="context">Holds the data needed for serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            var element = context.CurrentNode as XmlElement;
            if (element == null)
            {
                return false;
            }

            return this.WriteEnumerableElement(context, null, this.items);
        }

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml(XmlReader reader)
        {
            var lineInfo = reader as IXmlLineInfo;

            string myLocalName = reader.LocalName;

            // Read array count.
            int count;
            string strCount = reader.GetAttribute("Count");
            if (string.IsNullOrEmpty(strCount) == true ||
                int.TryParse(strCount, out count) == false)
            {
                throw this.GenerateReadXmlException(
                    "The attribute 'Count' is not of correct format.",
                    lineInfo);
            }

            // Create the array.
            this.items = new T[count];

            int index = 0;
            while (reader.Read())
            {
                if (reader.NodeType == XmlNodeType.Element &&
                    reader.Name == "Item" &&
                    index < count)
                {
                    reader.Read();

                    string strIsNull = reader.GetAttribute("xsi:nil");
                    if (string.IsNullOrEmpty(strIsNull) == false && bool.Parse(strIsNull) == true)
                    {
                        this.items[index] = default(T);
                    }
                    else if (typeof(IXmlDocSerializable).IsAssignableFrom(typeof(T)) == true)
                    {
                        var doc = new XmlDocument();
                        doc.Load(reader);

                        this.items[index] = (T)SerializationHelper.LoadXmlDocSerializable(typeof(T), doc, true);
                    }
                    else
                    {
                        try
                        {
                            this.items[index] = SerializationHelper.Load<T>(reader, true);
                        }
                        catch (InvalidOperationException ex)
                        {
                            // If the item is an user data model derived from 'T',
                            // the serializer might fail for some reason.
                            // As a fallback, we look for the type with the Xml tag name from
                            // the extra type array, then create the dedicated serializer
                            // to deserialize this item.
                            var type = SerializationHelper.GetExtraTypes().FirstOrDefault(
                                t => t.Name == reader.Name);
                            if (type != null)
                            {
                                var dedicateSerializer = new XmlSerializer(type);
                                this.items[index] = (T)SerializationHelper.Load(type, reader);
                            }
                            else
                            {
                                // The deserialization still failed, rethrow the exception.
                                throw ex;
                            }
                        }
                    }

                    ++index;
                }
                else if (reader.NodeType == XmlNodeType.EndElement &&
                    reader.LocalName == myLocalName)
                {
                    // This is the end tag of the ArrayCollection.
                    // Read once more, so the next element can be read correctly.
                    reader.Read();
                    break;
                }
            }
        }

        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml(XmlWriter writer)
        {
            writer.WriteAttributeString("Count", this.Count.ToString());

            foreach (T item in this.items)
            {
                writer.WriteStartElement("Item");

                if (item is IXmlDocSerializable)
                {
                    var doc = new XmlDocument();
                    SerializationHelper.SaveXmlDocSerializable((IXmlDocSerializable)item, doc);
                    doc.Save(writer);
                }
                else if (item != null)
                {
                    SerializationHelper.Save(item.GetType(), item, writer);
                }
                else
                {
                    const string XsiNamespaceUri = "http://www.w3.org/2001/XMLSchema-instance";

                    writer.WriteStartElement(typeof(T).Name);
                    writer.WriteAttributeString("xsi", "nil", XsiNamespaceUri, bool.TrueString);
                    writer.WriteEndElement();
                }

                writer.WriteEndElement();
            }
        }

        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します.
        /// </summary>
        /// <returns>コレクションを反復処理するために使用できる System.Collections.IEnumerator オブジェクト.</returns>
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return this.GetEnumerator();
        }

        /// <summary>
        /// ある要素が存在するかどうかを判断します.
        /// </summary>
        /// <param name="value">検索するオブジェクト.</param>
        /// <returns>存在する場合は true。それ以外の場合は false。</returns>
        bool ICollection<T>.Contains(T value)
        {
            return this.Contains(value);
        }

        #region Unsupported Methods

        /// <summary>
        /// 最初に見つかった特定のオブジェクトを削除します.
        /// </summary>
        /// <param name="value">削除するオブジェクト.</param>
        /// <returns>削除が正常終了したらtrue, それ以外はfalseを返却します.</returns>
        bool ICollection<T>.Remove(T value)
        {
            throw new NotSupportedException();
        }

        /// <summary>
        /// 項目を追加します.
        /// </summary>
        /// <param name="value">追加する項目.</param>
        public void Add(T value)
        {
            throw new NotSupportedException();
        }

        /// <summary>
        /// すべての項目の値をリセットします.
        /// </summary>
        public void Clear()
        {
            throw new NotSupportedException();
        }

        #endregion

        /// <summary>
        /// Generate exception for ReadXml().
        /// </summary>
        /// <param name="message">The message for the exception.</param>
        /// <param name="lineInfo">The line info.</param>
        /// <returns>The exception.</returns>
        private XmlException GenerateReadXmlException(string message, IXmlLineInfo lineInfo)
        {
            if (lineInfo != null)
            {
                return new XmlException(message, null, lineInfo.LineNumber, lineInfo.LinePosition);
            }
            else
            {
                return new XmlException(message);
            }
        }
    }
}
