﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.Foundation.Coercers
{
    /// <summary>
    /// A base class for ranged coercion.
    /// </summary>
    /// <typeparam name="T">The specific value type.</typeparam>
    public abstract class RangedCoercerBase<T> : CoercerBase<T> where T : IComparable
    {
        /// <summary>
        /// Backing field for Minimum property.
        /// </summary>
        private T minimum;

        /// <summary>
        /// Backing field for Maximum property.
        /// </summary>
        private T maximum;

        /// <summary>
        /// Initializes the RangedCoercerBase instance.
        /// </summary>
        /// <param name="minimum">The minimum allowed value.</param>
        /// <param name="maximum">The maximum allowed value.</param>
        public RangedCoercerBase(T minimum, T maximum)
        {
            this.Minimum = minimum;
            this.Maximum = maximum;
        }

        /// <summary>
        /// Raised when the minimum value changes.
        /// </summary>
        public event EventHandler<ValueChangedEventArgs<T>> MinimumChanged;

        /// <summary>
        /// Raised when the maximum value changes.
        /// </summary>
        public event EventHandler<ValueChangedEventArgs<T>> MaximumChanged;

        /// <summary>
        /// Gets or sets the minimum allowed value.
        /// </summary>
        public T Minimum
        {
            get
            {
                return this.minimum;
            }

            set
            {
                if (value == null)
                {
                    throw new ArgumentNullException("value");
                }

                if (this.minimum.CompareTo(value) != 0)
                {
                    var oldValue = this.minimum;
                    this.minimum = value;
                    this.OnMinimumChanged(new ValueChangedEventArgs<T>(oldValue, value));
                }
            }
        }

        /// <summary>
        /// Gets or sets the maximum allowed value.
        /// </summary>
        public T Maximum
        {
            get
            {
                return this.maximum;
            }

            set
            {
                if (value == null)
                {
                    throw new ArgumentNullException("value");
                }

                if (this.maximum.CompareTo(value) != 0)
                {
                    var oldValue = this.maximum;
                    this.maximum = value;
                    this.OnMaximumChanged(new ValueChangedEventArgs<T>(oldValue, value));
                }
            }
        }

        /// <summary>
        /// Coerces the input value.
        /// This method already perform the minimum and maximum coercion.
        /// </summary>
        /// <param name="input">The input value to coerce.</param>
        /// <returns>Returns the coerced value.</returns>
        public override T Coerce(T input)
        {
            if (this.Minimum.CompareTo(input) < 0)
            {
                return this.Minimum;
            }
            else if (this.Maximum.CompareTo(input) > 0)
            {
                return this.Maximum;
            }

            return this.OnCoerce(input);
        }

        /// <summary>
        /// When implemented, called from Coerce method, after minimum and maximum coercion.
        /// </summary>
        /// <param name="input">The input value to coerce.</param>
        /// <returns>Returns the coerced value.</returns>
        public abstract T OnCoerce(T input);

        /// <summary>
        /// Raises the MinimumChanged event.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected virtual void OnMinimumChanged(ValueChangedEventArgs<T> e)
        {
            var handler = this.MinimumChanged;

            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// Raises the MinimumChanged event.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected virtual void OnMaximumChanged(ValueChangedEventArgs<T> e)
        {
            var handler = this.MaximumChanged;

            if (handler != null)
            {
                handler(this, e);
            }
        }
    }
}
