﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#ifndef __GFDFILETEXTURE_H__
#define __GFDFILETEXTURE_H__

#ifdef __cplusplus
extern "C"
{
#endif // __cplusplus

/// @addtogroup GFDFileTextureGroup
/// @{

///
/// For Writing gtx file.
///

/// \brief Creates a gtx file based on the passed in GX2Texture
///
/// \param pFilename    Name of the output file
/// \param gpuVer       GPU version
/// \param swapMode     Endian swap mode for image data (GPU data)
/// \param alignMode    Align mode for adding data specific allignment size padding block before image data (GPU data)
/// \param numTextures  Number of textures in pTexture
/// \param pTexture     Input GX2Texture pointer used to create gtx file
///
GFD_DECLSPEC bool GFD_API GFDWriteFileTexture(char* pFilename, GFDGPUVersion gpuVer, GFDEndianSwapMode swapMode, GFDAlignMode alignMode, u32 numTextures, GX2Texture* pTexture);

/// \brief Appends textures based on the passed in GX2Texture to an existing gtx file
///
/// \param pFilename    Name of the file to append textures
/// \param gpuVer       GPU version
/// \param swapMode     Endian swap mode for image data (GPU data)
/// \param alignMode    Align mode for adding data specific allignment size padding block before image data (GPU data)
/// \param numTexture  Number of textures in pTexture
/// \param pTexture     Input GX2Texture pointer used to create gtx file
///
GFD_DECLSPEC bool GFD_API GFDAppendWriteFileTexture(char* pFilename, GFDGPUVersion gpuVer, GFDEndianSwapMode swapMode, GFDAlignMode alignMode, u32 numTexture, GX2Texture* pTexture);

///
/// For Reading gtx file.
///

/// \brief Reads a gtx file and sets data to GX2Texture. Need to call GFDFreeFileTexture after reading.
///
/// \param pTexture     Output GX2Texture pointer
/// \param gpuVer       GPU version
/// \param swapMode     Endian swap mode for image data (GPU data)
/// \param fileName    Name of the input gtx file
///
GFD_DECLSPEC bool GFD_API GFDReadFileTexture(GX2Texture* pTexture, GFDGPUVersion gpuVer, GFDEndianSwapMode swapMode, const char* fileName);

/// \brief Free GX2Texture allocated by with GFDReadFileTexture.
///
/// \param pTexture     GX2Texture pointer to free
///
GFD_DECLSPEC void GFD_API GFDFreeFileTexture(GX2Texture* pTexture);

/// @}

#ifdef __cplusplus
}
#endif // __cplusplus

#endif  //#ifndef __GFDFILETEXTURE_H__
