﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShaderConverterLib_ShaderConverterUtility_H__
#define __ShaderConverterLib_ShaderConverterUtility_H__

namespace EffectMaker {
namespace ShaderConverterLib {

    #define ShaderCodeBufferSize 1048576

    /// <summary>
    /// Profiler to count and output elapsed time within a code block.
    /// </summary>
    class Profiler
    {
    public :
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="szTag">The tag for the profile.</param>
        Profiler(const char *szTag);

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="szTag">The tag for the profile.</param>
        /// <param name="shouldOutput">Should output elapsed time when stopped.</param>
        Profiler(const char *szTag, bool shouldOutput);

        /// <summary>
        /// Destructor.
        /// </summary>
        ~Profiler();

        /// <summary>
        /// Stop the profiling and output elapsed time.
        /// </summary>
        /// <returns>The elapsed seconds.</returns>
        double Stop();

    private :
        bool           shouldOutput;
        bool           isStopped;
        const char    *szTag;
        LARGE_INTEGER  startTime;
    };

    ////////////////////////////////////
    // ShaderCodeParam struct
    ////////////////////////////////////
    struct LIB_API ShaderCodeParam
    {
    public :
        //---------------------------------------------------------------------
        // Default constructor.
        //---------------------------------------------------------------------
        ShaderCodeParam() :
            buf(NULL),
            size(0)
        {
        };

        bool IsAvailable()
        {
            if ( buf != NULL && size != 0 ) return true;
            else return false;
        }

        char* buf;
        u32   size;
    };

    ////////////////////////////////////
    // shaderKeyInfo struct
    ////////////////////////////////////
    struct ShaderKeyInfo
    {
    public:
        ShaderKeyInfo() :
            isField(false),
            reservedShaderIndex(0),
            vertexShaderIndex(0),
            pixelShaderIndex(0),
            customShaderIndex(0),
            userShaderDef(0),
            vsCode(NULL),
            vsSize(0),
            fsCode(NULL),
            fsSize(0),
            name(NULL),
            retCode(0),
            isAlreadyGetShader(false),
            overrideShader(NULL),
            overrideShaderLength(0)
        {
        }

        void SetKey( nw::eft2::ShaderKey* key, bool field, s32 reservedShaderIndex )
        {
            memcpy( (void *)&this->shaderKey, key, sizeof(nw::eft2::ShaderKey) );
            this->isField             = field;
            this->reservedShaderIndex = reservedShaderIndex;
        }

        void SetKey( nw::eft2::ShaderKey* key )
        {
            memcpy( (void *)&this->shaderKey, key, sizeof(nw::eft2::ShaderKey) );
        }

        void SetDefKey( nw::eft2::ShaderKey* key )
        {
            memcpy( (void *)&this->defaultShaderKey, key, sizeof(nw::eft2::ShaderKey) );
        }

        bool IsEqual( ShaderKeyInfo* newKey )
        {
            if ( ( shaderKey.IsEqual( &newKey->shaderKey ) == true ) &&
                 ( isField             == newKey->isField )          &&
                 ( reservedShaderIndex == newKey->reservedShaderIndex ) )
            {
                return true;
            }
            return false;
        }

        void SetShaderKeyInfo( ShaderKeyInfo& newInfo )
        {
            memcpy( (void *)&this->shaderKey, &newInfo.shaderKey, sizeof(nw::eft2::ShaderKey) );
            memcpy( (void *)&this->defaultShaderKey, &newInfo.defaultShaderKey, sizeof(nw::eft2::ShaderKey) );
            isField               = newInfo.isField;
            reservedShaderIndex   = newInfo.reservedShaderIndex;
            vertexShaderIndex     = newInfo.vertexShaderIndex;
            pixelShaderIndex      = newInfo.pixelShaderIndex;
            customShaderIndex     = newInfo.customShaderIndex;
            userShaderDef         = newInfo.userShaderDef;
            vsCode                = newInfo.vsCode;
            vsSize                = newInfo.vsSize;
            fsCode                = newInfo.fsCode;
            fsSize                = newInfo.fsSize;
            name                  = newInfo.name;
            retCode               = newInfo.retCode;
            isAlreadyGetShader    = newInfo.isAlreadyGetShader;
            overrideShader        = newInfo.overrideShader;
            overrideShaderLength  = newInfo.overrideShaderLength;
        }

        nw::eft2::ShaderKey                   shaderKey;
        bool                                  isField;
        s32                                   reservedShaderIndex;
        u32                                   vertexShaderIndex;
        u32                                   pixelShaderIndex;

        nw::eft2::ShaderKey                   defaultShaderKey;       // デフォルトシェーダキー(ShaderConverterInternalで使用)
        u32                                   customShaderIndex;      // カスタムシェーダ番号(ShaderConverterInternalで使用)
        u32                                   userShaderDef;          // UserShaderDefの種類(ShaderConverterInternalで使用)
        char*                                 vsCode;                 // 頂点シェーダコード(ShaderConverterInternalで使用)
        u32                                   vsSize;                 // 頂点シェーダコードサイズ(ShaderConverterInternalで使用)
        char*                                 fsCode;                 // フラグメントシェーダコード(ShaderConverterInternalで使用)
        u32                                   fsSize;                 // フラグメントシェーダコードサイズ(ShaderConverterInternalで使用)
        char*                                 name;                   // エミッタ名(ShaderConverterInternalで使用)
        s32                                   retCode;                // シェーダコンパイルのreturn value (ShaderConverterInternalで使用)
        bool                                  isAlreadyGetShader;     // ShaderCacheから設定されている場合(ShaderConverterInternalで使用)
        char*                                 overrideShader;         // OverRide Shaderが設定されている場合(ShaderConverterInternalで使用)
        u32                                   overrideShaderLength;   // OverRide Shaderが設定されている場合(ShaderConverterInternalで使用)
    };

    ////////////////////////////////////
    // CopyShaderCode class
    ////////////////////////////////////
    class  CopyShaderCode
    {
    public:
        //------------------------------------------------------------------------------
        //      コンストラクタ
        //------------------------------------------------------------------------------
        CopyShaderCode( char* dstbuf );

        //------------------------------------------------------------------------------
        //      シェーダソースコードをコピーする
        //------------------------------------------------------------------------------
        void CopyCode( const char* srcBuf, u32 size );

        //------------------------------------------------------------------------------
        //      コピーしたシェーダソースコードのサイズを取得する
        //------------------------------------------------------------------------------
        u32 GetShaderSize() { return mShaderCodeLength; }

    private:
        u32   mBufAvailSize;
        u32   mShaderCodeLength;
        char* mDstBuf;

    };

    ////////////////////////////////////
    // ShaderCodeList class
    ////////////////////////////////////
    struct ShaderCodeListCpp
    {
    public:
        ShaderCodeParam  mSpecDecShaderSrc;

        ShaderCodeParam  mVertexShaderSrc;
        ShaderCodeParam  mFragShaderSrc;
        ShaderCodeParam  mStripShaderSrc;

        ShaderCodeParam  mParticleGlslSrc;
        ShaderCodeParam  mParticleDecVshSrc;
        ShaderCodeParam  mParticleDecFshSrc;
        ShaderCodeParam  mStreamOutDecVshSrc;
        ShaderCodeParam  mStreamOutVshSrc;
        ShaderCodeParam  mGeneralVshSrc;
        ShaderCodeParam  mGeneralFshSrc;
        ShaderCodeParam  mGeneralCshSrc;
        ShaderCodeParam  mCustomVshSrc[8];
        ShaderCodeParam  mCustomFshSrc[8];
        ShaderCodeParam  mReservedVshSrc[8];
        ShaderCodeParam  mReservedFshSrc[8];
    };

    ////////////////////////////////////
    // ShaderCache class
    ////////////////////////////////////
    class LIB_API ShaderCache
    {
    public:
        //------------------------------------------------------------------------------
        //      コンストラクタ
        //------------------------------------------------------------------------------
        ShaderCache():
            mBinary(NULL),
            mShaderInfo(NULL),
            mShaderTable(NULL),
            mShaderBinary(NULL),
            mNumVertexShader(0),
            mNumFragShader(0),
            mVshArray(NULL),
            mFshArray(NULL)

        {}

        //------------------------------------------------------------------------------
        //      デストラクタ
        //------------------------------------------------------------------------------
        ~ShaderCache()
        {
            if ( mBinary )           delete mBinary;
            if ( mVshArray != NULL ) delete[] mVshArray;
            if ( mFshArray != NULL ) delete[] mFshArray;
        }

        void ReadShaderCache( FILE* fp );
        void WriteShaderCache( void* gshBinary,
                               u32   gshBinarySize,
                               u32            numShader,
                               ShaderKeyInfo* shaderKeyArray,
                               char*          shaderCachePath );

        bool IsValidCache() { return ( mShaderInfo->totalShaderSize > 0 ) ? true : false; }
        s32  IsFindCache( ShaderKeyInfo* shaderKey );

        bool GetShaderCache(s32 index, ShaderTable* dstshader )
        {
            u32 vIdx = mShaderTable[index].shaderKeyInfo.vertexShaderIndex;
            u32 fIdx = mShaderTable[index].shaderKeyInfo.pixelShaderIndex;
            if ( vIdx > mNumVertexShader || fIdx > mNumFragShader ) return false;

            if ( mVshArray[vIdx] == NULL || mFshArray[fIdx] == NULL ) return false;

            dstshader->SetVertexShaderId( gshPushVertexShader( mVshArray[vIdx] ) );
            dstshader->SetVertexShader( mVshArray[vIdx] );
            dstshader->SetPixelShaderId(  gshPushPixelShader(  mFshArray[fIdx] ) );
            dstshader->SetPixelShader( mFshArray[fIdx] );

            return true;
        }

    private:
        struct shaderBinaryInfo
        {
             u32                tag;
             u32                numShader;
             u32                totalShaderSize;
        };

        struct shaderBinaryTable
        {
            ShaderKeyInfo
                                shaderKeyInfo;
            u32                 shaderSize;
            u32                 offset;
        };

        char*                   mBinary;
        shaderBinaryInfo*       mShaderInfo;
        shaderBinaryTable*      mShaderTable;
        char*                   mShaderBinary;
        u32                     mNumVertexShader;
        u32                     mNumFragShader;
        GX2VertexShader**       mVshArray;
        GX2PixelShader**        mFshArray;

    };

} // namespace ShaderConverter
} // namespace EffectMaker


#endif // __ShaderConverterLib_ShaderConverterUtility_H__
