﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define _CRT_SECURE_NO_WARNINGS

#include "ShaderConverterLib_PCH.h"
#include "ShaderConverter.h"

#include <nw/eft/eft2_Data.h>

#include <io.h>
#include <direct.h>
#include <stdafx.h>
#include <iostream>
#include "omp.h"

// To enable profiling of the shader conversion, uncomment the line below.
#define ENABLE_PROFILING

// To enable shader binary duplication checking, uncomment the line below.
//#define CHECK_DUP_BINARY

namespace EffectMaker {
namespace ShaderConverterLib {

/// <summary>
/// Constructor.
/// </summary>
/// <param name="szTag">The tag for the profile.</param>
/// <param name="shouldOutput">Should output elapsed time when stopped.</param>
Profiler::Profiler(const char *szTag, bool shouldOutput) :
    shouldOutput(shouldOutput),
    isStopped(false),
    szTag(szTag)
{
    #ifdef ENABLE_PROFILING
    ::QueryPerformanceCounter(&this->startTime);
    #endif
}

/// <summary>
/// Constructor for Profiler.
/// </summary>
/// <param name="szTag">The tag for the profile.</param>
Profiler::Profiler(const char *szTag) :
    shouldOutput(true),
    isStopped(false),
    szTag(szTag)
{
    #ifdef ENABLE_PROFILING
    ::QueryPerformanceCounter(&this->startTime);
    #endif
}

/// <summary>
/// Destructor for Profiler.
/// </summary>
Profiler::~Profiler()
{
    #ifdef ENABLE_PROFILING

    if (this->isStopped == false)
    {
        this->Stop();
    }

    #endif
}

/// <summary>
/// Stop the profiling and output elapsed time.
/// </summary>
/// <returns>The elapsed seconds.</returns>
double Profiler::Stop()
{
    #ifdef ENABLE_PROFILING

    LARGE_INTEGER currTime;
    LARGE_INTEGER freq;
    ::QueryPerformanceCounter(&currTime);
    ::QueryPerformanceFrequency(&freq);

    this->isStopped = true;

    double elapsedSeconds = (double)(currTime.QuadPart - this->startTime.QuadPart) / (double)freq.QuadPart;

    char szOutput[16];
    sprintf_s(szOutput, 16, "%.6f", elapsedSeconds);

    ////ShaderConverterMsg_ProfilerTime msg;
    ////msg.SetArgument(ShaderConverterMsg_ProfilerTime::Tag, this->szTag);
    ////msg.SetArgument(ShaderConverterMsg_ProfilerTime::ElapsedTime, szOutput);
    ////msg.Send();

    return elapsedSeconds;

    #else

    return 0.0;

    #endif
}

//------------------------------------------------------------------------------
//      Constructor
//------------------------------------------------------------------------------
ShaderConverter::ShaderConverter(const char *szConverterFolderPath,
                                 const char *szShaderFolderPath,
                                 int        convertMode,
                                 int        jobsNumber,
                                 int        shaderAlignment,
                                 int        binaryPosition,
                                 int        IsSubBinaryConverting ) :
    mShaderCachePath(NULL),
    szVshCodeCompositeBuffer(NULL),
    szFshCodeCompositeBuffer(NULL),
    mShaderKeyArray(NULL),
    mShaderTbl(NULL),
    mShaderIdx(0),
    mTotalShaderSize(0),
    mIsField(false),
    mReservedShaderIndex(-1),
    mIsShaderCacheUse(false),
    mIsShaderCacheFileExist(false),
    mIsShaderCacheCreated(false),
    mTmpShaderSourceCnt(0),
    mDefaultShaderKey(NULL),
    mShaderAlignment(shaderAlignment),
    mBinaryPosition(binaryPosition),
    mIsSubBinaryConverting(IsSubBinaryConverting)
{
    // Copy the folder paths.
    this->mConverterFolderPath = szConverterFolderPath;
    this->mShaderFolderPath    = szShaderFolderPath;

    this->szVshCodeCompositeBuffer = new char[ShaderCodeBufferSize];
    this->szFshCodeCompositeBuffer = new char[ShaderCodeBufferSize];

    // convert modeを設定
    mIsCommandLineMode = (( ( convertMode & 0xffff0000 ) >> 16) != 0 ) ? true : false; // 上位16ビットがコマンドラインモード
    mConvertMode       = convertMode & 0x0000ffff;                   // 下位16ビットがコンバートモード

    // Jobsの設定
    int maxThreads     = omp_get_max_threads();
    mJobsNumber        = ( jobsNumber >= maxThreads ) ? maxThreads-1 : jobsNumber;  // 指定されたJobs数がcpu core Maxと同じかそれ以上の時、(cpu core Max)-1数のスレッドで動作する

    if ( mConvertMode < CONVERT_FAST || mConvertMode > CONVERT_BUILD_SHADER_BINARY )
        mConvertMode = CONVERT_FAST;


    if ( ( mConvertMode == CONVERT_USE_SHADER_BINARY ) || ( mConvertMode == CONVERT_BUILD_SHADER_BINARY ) )
    {
        char fileDir[256];
        mShaderCachePath = new char[256];

        mIsShaderCacheUse = true;
        sprintf_s( fileDir, 256, "%s\\shaderCache", this->mConverterFolderPath );
        sprintf_s( mShaderCachePath, 256, "%s\\shaderCache.bin", fileDir );

        FILE* fp;
        if ( fopen_s( &fp, mShaderCachePath, "rb" ) == 0 )
        {
            // シェーダキャッシュを再作成する
            if ( mConvertMode == CONVERT_BUILD_SHADER_BINARY )
            {
                    mIsShaderCacheCreated = true;
            }
            else
            {
                mIsShaderCacheFileExist = true;
                mShaderCache.ReadShaderCache( fp );
            }
            fclose(fp);
        }
        else
        {
            if ( ( _mkdir( fileDir ) == 0 ) || ( errno == EEXIST ) )
            {
                if ( fopen_s( &fp, mShaderCachePath, "wb" ) == 0 )
                {
                    mIsShaderCacheCreated = true;
                    fclose( fp );
                }
                else
                {
                    _rmdir( fileDir );
                }
            }
        }
    }
}

//------------------------------------------------------------------------------
//      初期化処理
//------------------------------------------------------------------------------
void ShaderConverter:: Initialize(ShaderCodeListCpp* pShaderCodes)
{
    /*
    // This is an example of using the message forwarding system.
    // The first message opens a text file at the specified path with the text editor
    // set up in application configuration.
    ShaderConverterMsg_OpenTextFile msg1;
    msg1.SetArgument(ShaderConverterMsg_OpenTextFile::FilePath, "C:\\Dambo\\Prototype\\trunk\\Effect\\EffectMaker4\\Prototype\\ExternalLibraries\\ShaderConverter\\ShaderConverterLib\\MessageBase.cpp");
    msg1.Send();

    // The second message shows a message to the command line console and log view,
    // log level as warning, with a format string set in the string table (resource)
    // of DataModelLogic project, and 2 arguments to compose the message.
    ShaderConverterMsg_SampleStringResource msg2;
    msg2.SetArgument(ShaderConverterMsg_SampleStringResource::Destination, "Console,LogView");
    msg2.SetArgument(ShaderConverterMsg_SampleStringResource::LogLevel, "Warning");
    msg2.SetArgument(ShaderConverterMsg_SampleStringResource::Arg1, "EmitterSet1");
    msg2.SetArgument(ShaderConverterMsg_SampleStringResource::Arg2, "Emitter1");
    msg2.Send();
    */
    mShaderCodes         = pShaderCodes;
    mIsField             = false;
    mReservedShaderIndex = -1;
    mCustomShaderIndex   = -1;
}

//------------------------------------------------------------------------------
//      デストラクタ
//------------------------------------------------------------------------------
ShaderConverter::~ShaderConverter()
{
    // Clean up.
    if (this->szVshCodeCompositeBuffer != NULL)
    {
        delete[] this->szVshCodeCompositeBuffer;
        this->szVshCodeCompositeBuffer = NULL;
    }

    if (this->szFshCodeCompositeBuffer != NULL)
    {
        delete[] this->szFshCodeCompositeBuffer;
        this->szFshCodeCompositeBuffer = NULL;
    }

    if (mShaderCachePath != NULL )
    {
        delete mShaderCachePath;
        mShaderCachePath = NULL;
    }
}

//------------------------------------------------------------------------------
/// Utility method for setting console cursor position.
///
/// \param cursorPos The cursor position to set.
///
/// \return None.
//------------------------------------------------------------------------------
void SetConsoleCursorPosition(const COORD &cursorPos)
{
    HANDLE hConsole = ::GetStdHandle(STD_OUTPUT_HANDLE);
    ::SetConsoleCursorPosition(hConsole, cursorPos);
}

//------------------------------------------------------------------------------
/// Utility method for getting console cursor position.
///
/// \param cursorPos The reference to the cursor position structure.
///
/// \return None.
//------------------------------------------------------------------------------
void GetConsoleCursorPosition(COORD &cursorPos)
{
    HANDLE hConsole = ::GetStdHandle(STD_OUTPUT_HANDLE);

    CONSOLE_SCREEN_BUFFER_INFO info;
    ::GetConsoleScreenBufferInfo(hConsole, &info);

    cursorPos.X = info.dwCursorPosition.X;
    cursorPos.Y = info.dwCursorPosition.Y;
}

void ShaderConverter::AddShaderDataArrayInfo( ShaderConverterEmitterData *pDataArray,
                                                s32 dataCount )
{
    // 全てのEmitterのParamaterを設定する
    for ( s32 i = 0; i < dataCount; i++ )
    {
        DataArrayInfo    dataArrayInfo;
        ShaderKeyInfo    shaderKeyInfo;

        nw::eft2::ResEmitter* emitterData =
            GetEmitter( pDataArray[i].pBinaryHeader, pDataArray[i].pResEmitter );

        shaderKeyInfo.reservedShaderIndex = pDataArray[i].reservedShaderIndex;

        shaderKeyInfo.userShaderDef = 0;

        // フィールドの設定
        u32 fieldFlag = 0;
        if ( IsFieldExist( reinterpret_cast<nw::eft2::BinaryData*>( pDataArray[i].pBinaryHeader ) ) )
        {
            fieldFlag |= EFT_SHADER_KEY_FIELD_RANDOM;
            shaderKeyInfo.isField = true;
        }
        else
        {
            shaderKeyInfo.isField = false;
        }

        void* emitterPluginData = GetEmitterPlugin( reinterpret_cast<nw::eft2::BinaryData*>( pDataArray[i].pBinaryHeader ) );
        nw::eft2::ResFieldCustom* resFieldCustom = reinterpret_cast<nw::eft2::ResFieldCustom*>( GetCustomField( reinterpret_cast<nw::eft2::BinaryData*>( pDataArray[i].pBinaryHeader ) ) );

        //
        // shaderKeyを作成する。
        //
        emitterData->FlipEndian();
        nw::eft2::ShaderKey shaderkey;
        memset( &shaderkey, 0, sizeof(nw::eft2::ShaderKey) );

        s32 reservedShaderIndex = ( pDataArray[i].reservedShaderIndex >= 0 ) ?pDataArray[i].reservedShaderIndex + 1 : pDataArray[i].reservedShaderIndex;

        shaderkey.Initialize( emitterData, reservedShaderIndex, emitterPluginData, fieldFlag, resFieldCustom, NULL );

        shaderKeyInfo.SetKey( &shaderkey );
        shaderKeyInfo.SetDefKey( &shaderkey );

        // Custom Shader Indexの設定
        shaderKeyInfo.customShaderIndex = emitterData->shader.customShaderIndex;
        // エミッタ名を設定
        shaderKeyInfo.name = emitterData->name;

        // Override shaderの設定
        shaderKeyInfo.overrideShader       = pDataArray[i].szOverrideShader;
        shaderKeyInfo.overrideShaderLength = pDataArray[i].iOverrideShaderLength;

        // shaderKeyの重複チェック
        bool detectkey = false;
        for ( u32 j = 0; j < mShaderIdx; j++ )
        {
            if ( mShaderKeyArray[j].IsEqual( &shaderKeyInfo ) )
            {
                // shaderKeyが既にある場合されている場合
                detectkey = true;
                dataArrayInfo.shaderIndex = j;
                break;
            }
        }

        // ShaderCacheから頂点、フラグメントシェーダを所得する
        if ( IsUseShaderCache() == true )
        {
            s32 binNum = mShaderCache.IsFindCache( &shaderKeyInfo );
            if ( binNum != -1 )
            {
                if ( mShaderCache.GetShaderCache( binNum, &mShaderTbl[mShaderIdx] ) == true )
                {
                    shaderKeyInfo.isAlreadyGetShader = true;
                    mShaderKeyArray[mShaderIdx].SetShaderKeyInfo(shaderKeyInfo);

                    dataArrayInfo.shaderIndex = mShaderIdx;

                    mShaderIdx++;
                    detectkey = true;
                }
            }
        }

        if ( detectkey == false )
        {
            GenerateShaderCodeInternal( shaderKeyInfo ); // シェーダソースコードを作成する。

            mShaderKeyArray[mShaderIdx].SetShaderKeyInfo(shaderKeyInfo);

            dataArrayInfo.shaderIndex = mShaderIdx;

            mShaderIdx++;
        }

        dataArrayInfo.shaderConverterEitterData = &pDataArray[i];

        nw::eft2::ResShader* shader          = &emitterData->shader;
        pDataArray[i].shaderIndexOffset      = (u32)((u64)&shader->vertexShaderIndex - (u64)emitterData);
        pDataArray[i].userVertexShaderIndex1 = 0;     // クリアする
        pDataArray[i].userPixelShaderIndex1  = 0;     // クリアする
        pDataArray[i].userVertexShaderIndex2 = 0;     // クリアする
        pDataArray[i].userPixelShaderIndex2  = 0;     // クリアする

        // ShaderCompileDef1
        if ( strlen( shader->userShaderDefine1 ) > 0 )
        {
            nw::eft2::ShaderKey shaderkeyDef1;
            memset( &shaderkeyDef1, 0, sizeof(nw::eft2::ShaderKey) );

            shaderkeyDef1.Initialize( emitterData, reservedShaderIndex, emitterPluginData, fieldFlag, resFieldCustom, shader->userShaderDefine1 );
            shaderKeyInfo.userShaderDef = 1;

            shaderKeyInfo.SetKey( &shaderkeyDef1 );

            detectkey = false;
            // ShaderCacheから頂点、フラグメントシェーダを所得する
            if ( IsUseShaderCache() == true )
            {
                s32 binNum = mShaderCache.IsFindCache( &shaderKeyInfo );
                if ( binNum != -1 )
                {
                    if ( mShaderCache.GetShaderCache( binNum, &mShaderTbl[mShaderIdx] ) == true )
                    {
                        shaderKeyInfo.isAlreadyGetShader = true;
                        mShaderKeyArray[mShaderIdx].SetShaderKeyInfo(shaderKeyInfo);

                        detectkey = true;
                    }
                }
            }

            if ( detectkey == false )
            {
                GenerateShaderCodeInternal( shaderKeyInfo ); // シェーダソースコードを作成する。
            }

            mShaderKeyArray[mShaderIdx].SetShaderKeyInfo(shaderKeyInfo);

            dataArrayInfo.userShaderDef1Index       = mShaderIdx;
            dataArrayInfo.useShaderDef1Index        = true;

            mShaderIdx++;
        }

        // ShaderCompileDef2
        if ( strlen( shader->userShaderDefine2 ) > 0 )
        {
            nw::eft2::ShaderKey shaderkeyDef2;
            memset( &shaderkeyDef2, 0, sizeof(nw::eft2::ShaderKey) );

            shaderkeyDef2.Initialize( emitterData, reservedShaderIndex, emitterPluginData, fieldFlag, resFieldCustom, shader->userShaderDefine2 );
            shaderKeyInfo.userShaderDef = 2;

            shaderKeyInfo.SetKey( &shaderkeyDef2 );

            detectkey = false;
            // ShaderCacheから頂点、フラグメントシェーダを所得する
            if ( IsUseShaderCache() == true )
            {
                s32 binNum = mShaderCache.IsFindCache( &shaderKeyInfo );
                if ( binNum != -1 )
                {
                    if ( mShaderCache.GetShaderCache( binNum, &mShaderTbl[mShaderIdx] ) == true )
                    {
                        shaderKeyInfo.isAlreadyGetShader = true;
                        mShaderKeyArray[mShaderIdx].SetShaderKeyInfo(shaderKeyInfo);

                        detectkey = true;
                    }
                }
            }

            GenerateShaderCodeInternal( shaderKeyInfo ); // シェーダソースコードを作成する。
            mShaderKeyArray[mShaderIdx].SetShaderKeyInfo( shaderKeyInfo );


            dataArrayInfo.userShaderDef2Index       = mShaderIdx;
            dataArrayInfo.useShaderDef2Index        = true;

            mShaderIdx++;
        }

        // DataArrayInfoに追加する
        mDataArrayInfo.push_back( dataArrayInfo );
    }
}

void ShaderConverter::GenerateShaderCodeInternal( ShaderKeyInfo& shaderKeyInfo, bool useDefShaderKey )
{
 //   mReservedShaderIndex = emitterParam.shaderKeyInfo.reservedShaderIndex;
 //   mIsField             = emitterParam.shaderKeyInfo.isField;
 //   mCustomShaderIndex   = emitterParam.customShaderIndex;

    // shader codeを生成する
    char* setting     = ( useDefShaderKey == false ) ? (char *)shaderKeyInfo.shaderKey.GetCompileSetting() :
                                                       (char *)shaderKeyInfo.defaultShaderKey.GetCompileSetting();

    // useDefShaderKeyがfalseの時CustomShaderを使用する
    bool useCustomShader = ( useDefShaderKey == false ) ? true : false;

    u32   settingSize = (u32)strlen( setting );

    memset( this->szVshCodeCompositeBuffer, 0, ShaderCodeBufferSize );
    memset( this->szFshCodeCompositeBuffer, 0, ShaderCodeBufferSize );

   // 頂点シェーダコードを生成する
    u32 shaderVshSize = GenerateVertexShaderCode( this->szVshCodeCompositeBuffer,
                                                  setting,
                                                  settingSize,
                                                  NULL,
                                                  0,
                                                  useCustomShader,
                                                  shaderKeyInfo.customShaderIndex,
                                                  shaderKeyInfo.reservedShaderIndex );

    // フラグメントシェーダコードを生成する
    u32 shaderFshSize = GenerateFragmentShaderCode( this->szFshCodeCompositeBuffer,
                                                    setting,
                                                    settingSize,
                                                    shaderKeyInfo.overrideShader,
                                                    shaderKeyInfo.overrideShaderLength,
                                                    useCustomShader,
                                                    shaderKeyInfo.customShaderIndex,
                                                    shaderKeyInfo.reservedShaderIndex );

    // Set up the shader code and size.
    shaderKeyInfo.vsSize = shaderVshSize;
    shaderKeyInfo.vsCode = new char[shaderVshSize + 1];
    memcpy_s( shaderKeyInfo.vsCode, shaderVshSize + 1, this->szVshCodeCompositeBuffer, shaderVshSize );
    shaderKeyInfo.vsCode[shaderVshSize] = '\0';

    shaderKeyInfo.fsSize = shaderFshSize;
    shaderKeyInfo.fsCode = new char[shaderFshSize + 1];
    memcpy_s(shaderKeyInfo.fsCode, shaderFshSize + 1, this->szFshCodeCompositeBuffer, shaderFshSize);
    shaderKeyInfo.fsCode[shaderFshSize] = '\0';
}

void ShaderConverter::AddShaderIndexToDataArray()
{
    u32 idx = 0;
    for ( u32 i = 0 ; i < mDataArrayInfo.size(); i++ )
    {
        ShaderConverterEmitterData* EmitterData  = mDataArrayInfo[i].shaderConverterEitterData;

        u32 vIdx = mShaderTbl[mDataArrayInfo[i].shaderIndex].GetVertexShaderId();
        u32 fIdx = mShaderTbl[mDataArrayInfo[i].shaderIndex].GetPixelShaderId();

        EmitterData->vertexShaderIndex = vIdx;
        EmitterData->pixelShaderIndex  = fIdx;
        mShaderKeyArray[mDataArrayInfo[i].shaderIndex].vertexShaderIndex = vIdx;
        mShaderKeyArray[mDataArrayInfo[i].shaderIndex].pixelShaderIndex = fIdx;

        // ShaderCompileDef1
        if ( mDataArrayInfo[i].IsUserShaderDef1Enabled() == true )
        {
            vIdx = mShaderTbl[mDataArrayInfo[i].userShaderDef1Index].GetVertexShaderId();
            fIdx = mShaderTbl[mDataArrayInfo[i].userShaderDef1Index].GetPixelShaderId();

            EmitterData->userVertexShaderIndex1 = vIdx;
            EmitterData->userPixelShaderIndex1  = fIdx;
            mShaderKeyArray[mDataArrayInfo[i].userShaderDef1Index].vertexShaderIndex = vIdx;
            mShaderKeyArray[mDataArrayInfo[i].userShaderDef1Index].pixelShaderIndex = fIdx;

        }

        // ShaderCompileDef2
        if ( mDataArrayInfo[i].IsUserShaderDef2Enabled() == true )
        {
            vIdx = mShaderTbl[mDataArrayInfo[i].userShaderDef2Index].GetVertexShaderId();
            fIdx = mShaderTbl[mDataArrayInfo[i].userShaderDef2Index].GetPixelShaderId();

            EmitterData->userVertexShaderIndex2 = vIdx;
            EmitterData->userPixelShaderIndex2  = fIdx;
            mShaderKeyArray[mDataArrayInfo[i].userShaderDef2Index].vertexShaderIndex = vIdx;
            mShaderKeyArray[mDataArrayInfo[i].userShaderDef2Index].pixelShaderIndex = fIdx;

        }
    }
}

u32 ShaderConverter::ConvertInternal( ShaderConverterEmitterData *pDataArray,
                                      ShaderCompileErrorList *pCompileErrorList,
                                      s32 dataCount)
{
    char multiPath[512];

    // DLLのコピー先ディレクトリパスを設定
    DWORD processId = GetCurrentProcessId();
    sprintf_s(multiPath, "%s%u\\", this->mShaderFolderPath, processId);

    COORD startCursorPos;
    GetConsoleCursorPosition(startCursorPos);

    // emitterDataを作成する。
    AddShaderDataArrayInfo( pDataArray, dataCount );

    // gsh Shader DLLを初期化する。
    gshInitializMulti(multiPath, this->mShaderFolderPath, mJobsNumber);

    // シェーダコード分シェーダバイナリを生成する。
    bool isError = false;
    s32  errorNo = 0;
    #pragma omp parallel for num_threads(mJobsNumber)
    for (int i = 0; i < (int)mShaderIdx; i++ )
    {
        if ( mShaderKeyArray[i].isAlreadyGetShader == false )
        {
            mShaderKeyArray[i].retCode = gshCompileMulti( mShaderKeyArray[i].vsCode,
                                                          mShaderKeyArray[i].vsSize,
                                                          mShaderKeyArray[i].fsCode,
                                                          mShaderKeyArray[i].fsSize,
                                                          NULL, 0,
                                                          NULL, 0,
#if CAFE_OS_SDK_VERSION >= 21104
                                                          NULL, 0,
#endif
                                                          NULL,
                                                          &mShaderTbl[i],
                                                          NULL );
            // バイナリコンバートチェックのための場合は次の処理をスキップすること。
            if ( ( mShaderKeyArray[i].retCode != EC_SUCCESS ) && (mShaderKeyArray[i].retCode != EC_OUTPUTWARNING ) )
            {
                isError = true;
                errorNo = i;
            }
        }
        if ((float)i / (float)mShaderIdx * 100.0f <= 100.0f )
            SetConsoleCursorPosition(startCursorPos);
            printf("Compiling Shaders : %.1f%%\n", (float)i / (float)mShaderIdx * 100.0f );
        #pragma omp flush(isError)
        #pragma omp flush(errorNo)
    }

    // エラーの確認をします。エラーがある場合はデフォルト設定でもう再コンパイルします。
    for ( u32 i = 0; i < mShaderIdx; i++ )
    {
        if (mShaderKeyArray[i].retCode == EC_SUCCESS || mShaderKeyArray[i].retCode == EC_OUTPUTWARNING)
        {
            if (mShaderTbl[i].GetVertexShader() == NULL && mShaderTbl[i].GetPixelShader() == NULL)
            {
                pCompileErrorList->AddErrorLog(mShaderKeyArray[i].name,
                    mShaderKeyArray[i].vsCode,
                    mShaderKeyArray[i].fsCode,
                    "Can't get shader binary, but shader compile error doesn't occure\n");

                OutputShaderCompileErrorMessage(mShaderKeyArray[i].name, 0);
                // エラーが出力されたので、一行ずらしておく
                startCursorPos.Y++;
            }
        }
        if ( mShaderKeyArray[i].retCode != EC_SUCCESS && mShaderKeyArray[i].retCode != EC_OUTPUTWARNING )
        {
            OutputShaderCompileErrorMessage( mShaderKeyArray[i].name, mShaderKeyArray[i].userShaderDef );
            // エラーが出力されたので、一行ずらしておく
            startCursorPos.Y++;

            pCompileErrorList->AddErrorLog(  mShaderKeyArray[i].name,
                                             mShaderKeyArray[i].vsCode,
                                             mShaderKeyArray[i].fsCode,
                                             mShaderTbl[i].GetLog() );

            // default settingでコンパイルし直す。
            GenerateShaderCodeInternal( mShaderKeyArray[i], true );  // シェーダソースコードを作成し直す。

            if ( gshCompileMulti( mShaderKeyArray[i].vsCode,
                                  mShaderKeyArray[i].vsSize,
                                  mShaderKeyArray[i].fsCode,
                                  mShaderKeyArray[i].fsSize,
                                  NULL, 0,
                                  NULL, 0,
#if CAFE_OS_SDK_VERSION >= 21104
                                  NULL, 0,
#endif
                                  NULL,
                                  &mShaderTbl[i],
                                  NULL ) != EC_SUCCESS )
            {
                OutputShaderCompileErrorMessage( mShaderKeyArray[i].name, 0 );
                // エラーが出力されたので、一行ずらしておく
                startCursorPos.Y++;
            }
        }
        else if ( mShaderKeyArray[i].retCode == EC_OUTPUTWARNING )
        {
            OutputShaderCompileWarningMessage( mShaderKeyArray[i].name );
            //OutputShaderCompileErrorLog( emitterData->name, shaderTbl[shaderIdx].GetLog() );
            // エラーが出力されたので、一行ずらしておく
            startCursorPos.Y++;

            pCompileErrorList->AddWarningLog( mShaderKeyArray[i].name,
                                              this->szVshCodeCompositeBuffer,
                                              this->szFshCodeCompositeBuffer,
                                              mShaderTbl[i].GetLog() );
        }
    }

    // 重複シェーダバイナリを取り除く。
    gshRemoveDuplicateShader( mShaderTbl, mShaderIdx );

    // shaderIndexを各pDataArrayに振る。
    AddShaderIndexToDataArray();

    // Output shader compiling progress : 100%.
    SetConsoleCursorPosition(startCursorPos);
    printf("Compiling Shaders : 100.0%%\n");

    // ここにgshFinalizeMulti処理を入れる
    gshFinalizeMulti(multiPath);

    return mShaderIdx;
}

//------------------------------------------------------------------------------
//      コンバート処理
//------------------------------------------------------------------------------
u8* ShaderConverter::Convert( ShaderConverterEmitterData *pDataArray,
                               ShaderCompileErrorList *pCompileErrorList,
                               s32 dataCount,
                               s32 *pBinarySize,
                               s32 *pShaderCount)
{
    // シェーダバイナリを保存するメモリ領域
    u8 *pBuffer = NULL;

    bool result = true;
    const static size_t constBinDataSize = sizeof( nw::eft2::BinaryData );

    //
    // dllを初期化する。
    //
    gshInitialize( this->mConverterFolderPath, dataCount );

    mShaderKeyArray   = new ShaderKeyInfo[dataCount * 3];
    mShaderTbl        = new ShaderTable[(dataCount * 3) + 1];
    mShaderIdx        = 0;
    mTotalShaderSize  = 0;
    mDefaultShaderKey = NULL;

    if ( mIsCommandLineMode == true )
    {
        // コマンドラインモードの場合は複数スレッドでコンバートを行う。
        ConvertInternal( pDataArray, pCompileErrorList, dataCount );
    }
    else
    {
        COORD startCursorPos;
        GetConsoleCursorPosition(startCursorPos);

        // ResEmitter分のshaderの確認とshader binaryの作成
        //
        for ( s32 i = 0; i < dataCount; i++ )
        {
            // Output shader compiling progress.
            SetConsoleCursorPosition(startCursorPos);
            printf("Compiling Shaders : %.1f%%\n", (float)i / (float)dataCount * 100.0f);

            // Get the reserved shader index.
            mReservedShaderIndex = pDataArray[i].reservedShaderIndex;

            nw::eft2::ResEmitter* emitterData =
                GetEmitter( pDataArray[i].pBinaryHeader, pDataArray[i].pResEmitter );

            u32 fieldFlag = 0;
            if ( IsFieldExist( reinterpret_cast<nw::eft2::BinaryData*>( pDataArray[i].pBinaryHeader ) ) )
            {
                fieldFlag |= EFT_SHADER_KEY_FIELD_RANDOM;
                mIsField = true;
            }
            else
            {
                mIsField = false;
            }

    		void* emitterPluginData = GetEmitterPlugin( reinterpret_cast<nw::eft2::BinaryData*>( pDataArray[i].pBinaryHeader ) );
            nw::eft2::ResFieldCustom* resFieldCustom = reinterpret_cast<nw::eft2::ResFieldCustom*>( GetCustomField( reinterpret_cast<nw::eft2::BinaryData*>( pDataArray[i].pBinaryHeader ) ) );

            //
            // shaderKeyを作成する。
            //
            emitterData->FlipEndian();
            nw::eft2::ShaderKey shaderKey;
            memset( &shaderKey, 0, sizeof(nw::eft2::ShaderKey) );

            s32 reservedShaderIndex = ( mReservedShaderIndex >= 0 ) ? mReservedShaderIndex + 1 : mReservedShaderIndex;

    		shaderKey.Initialize( emitterData, reservedShaderIndex, emitterPluginData, fieldFlag, NULL );
            mDefaultShaderKey      = &shaderKey;

            // Custom Shader Indexの設定
            mCustomShaderIndex = emitterData->shader.customShaderIndex;

            // offsetを設定する
            nw::eft2::ResShader* shader          = &emitterData->shader;
            pDataArray[i].shaderIndexOffset      = (u32)((u64)&shader->vertexShaderIndex - (u64)emitterData);
            pDataArray[i].userVertexShaderIndex1 = 0;     // クリアする
            pDataArray[i].userPixelShaderIndex1  = 0;     // クリアする
            pDataArray[i].userVertexShaderIndex2 = 0;     // クリアする
            pDataArray[i].userPixelShaderIndex2  = 0;     // クリアする

            //
            // shader keyの重複をチェックする
            //
            bool detectKey = false;
            for ( u32 j = 0; j < mShaderIdx; j++ )
            {
                if ( ( mShaderKeyArray[j].shaderKey.IsEqual( &shaderKey ) == true )      &&
                     ( mShaderKeyArray[j].isField                         == mIsField )  &&
                     ( mShaderKeyArray[j].reservedShaderIndex             == mReservedShaderIndex ) )
                {
                    // shaderKeyが既にバイナリ化 されている場合
                    detectKey = true;
                    pDataArray[i].vertexShaderIndex = mShaderKeyArray[j].vertexShaderIndex;  // VertexShaderIndexを設定
                    pDataArray[i].pixelShaderIndex  = mShaderKeyArray[j].pixelShaderIndex;   // PixelShaderIndexを設定
                    break;
                }
            }

            //
            // shaderKeyの重複がない場合はシェーダバイナリを生成する
            //
            if ( detectKey == false )
            {
                u32 index =  ConvertWithMode( &pDataArray[i],
                                              pCompileErrorList,
                                              emitterData,
                                              &shaderKey,
                                              0 );
                pDataArray[i].vertexShaderIndex = mShaderTbl[index].GetVertexShaderId();
                pDataArray[i].pixelShaderIndex  = mShaderTbl[index].GetPixelShaderId();
            }

            // ShaderCompileDef1
            if ( strlen( shader->userShaderDefine1 ) > 0 )
            {
                nw::eft2::ShaderKey shaderKeyDef1;
    			shaderKeyDef1.Initialize( emitterData, reservedShaderIndex, emitterPluginData, fieldFlag, resFieldCustom, shader->userShaderDefine1 );

                Profiler profilerCompileShaderDef1("Compile shader def 1");

                u32 index = ConvertWithMode( &pDataArray[i],
                                             pCompileErrorList,
                                             emitterData,
                                             &shaderKeyDef1,
                                             1 );

                pDataArray[i].userVertexShaderIndex1 = mShaderTbl[index].GetVertexShaderId();
                pDataArray[i].userPixelShaderIndex1  = mShaderTbl[index].GetPixelShaderId();

                profilerCompileShaderDef1.Stop();
            }

            // ShaderCompileDef2
            if ( strlen( shader->userShaderDefine2 ) > 0 )
            {
                nw::eft2::ShaderKey shaderKeyDef2;
                shaderKeyDef2.Initialize( emitterData, reservedShaderIndex, emitterPluginData, fieldFlag, resFieldCustom, shader->userShaderDefine2 );

                Profiler profilerCompileShaderDef2("Compile shader def 2");

                u32 index = ConvertWithMode( &pDataArray[i],
                                             pCompileErrorList,
                                             emitterData,
                                             &shaderKeyDef2,
                                             2 );
                pDataArray[i].userVertexShaderIndex2 = mShaderTbl[index].GetVertexShaderId();
                pDataArray[i].userPixelShaderIndex2  = mShaderTbl[index].GetPixelShaderId();

                profilerCompileShaderDef2.Stop();
            }
        }

        // Output shader compiling progress : 100%.
        SetConsoleCursorPosition(startCursorPos);
        printf("Compiling Shaders : 100.0%%\n");
    }

    //
    // streamout shaderを追加する(サブバイナリの時はStreamOutのバイナリをつけない）
    //
    if (mIsSubBinaryConverting == false)
    {
        // Compile Shader Binary
        Profiler profilerCompileStreamoutShader("Compile stream out shader");
        s32 ret = CompileShader( this->szVshCodeCompositeBuffer,
                                 this->szFshCodeCompositeBuffer,
                                 NULL,
                                 0,
                                 NULL,
                                 0,
                                 (ShaderTable*)&mShaderTbl[mShaderIdx],
                                 true,
                                 true );
        profilerCompileStreamoutShader.Stop();

        if ( ret == EC_SUCCESS || ret == EC_OUTPUTWARNING )
        {
            if ( ret == EC_OUTPUTWARNING )
            {
                OutputShaderCompileWarningMessage( "StreamOut Shader" );
                pCompileErrorList->AddWarningLog( "StreamOut Shader",
                                                  this->szVshCodeCompositeBuffer,
                                                  this->szFshCodeCompositeBuffer,
                                                  mShaderTbl[mShaderIdx].GetLog() );
            }
            mShaderIdx++;
        }
        else
        {
            OutputShaderCompileErrorMessage( "StreamOut Shader" );
            pCompileErrorList->AddErrorLog( "StreamOut Shader",
                                            this->szVshCodeCompositeBuffer,
                                            this->szFshCodeCompositeBuffer,
                                            mShaderTbl[mShaderIdx].GetLog() );

        }
    }

    // gshシェーダバイナリを作成
    gshGenerateShaderBinary();

    // シェーダバイナリのサイズとポインタを取得
    u32 shaderSize  = gshGetgshBinarySize();
    char* shaderPtr = gshGetgshBinary();

    //
    // shader binary cacheを出力する
    //
    if ( IsWriterShaderCache() == true )
    {
        // char msg[64];
        // sprintf_s( msg, 64, "shader cache binary create\n");
        // OutputDebugMsg( msg );

        ShaderCache shaderCache;
        shaderCache.WriteShaderCache( shaderPtr, shaderSize, mShaderIdx, mShaderKeyArray, mShaderCachePath );
    }

    // シェーダ個数が１つなのでシェーダサイズとともに更新する
    // mShaderIdx = 1;
    mTotalShaderSize = shaderSize;

    //
    // shader分だけBinaryDataを作成する
    //
    u32 BinaryDataInfoSize = constBinDataSize * ( 1 + 1 );  // shaderIdx分のbinData(１個) + SHDA用のbinData
//    u32 BinaryDataInfoSize = constBinDataSize * ( mShaderIdx + 1 );  // shaderIdx分のbinData + SHDA用のbinData

    u32 padding            = (mBinaryPosition + BinaryDataInfoSize) % mShaderAlignment;
    if (padding > 0)
    {
        padding = mShaderAlignment - padding;
    }

    *pBinarySize           = BinaryDataInfoSize + padding + mTotalShaderSize; // shaderIdx分のbinData + SHDA用のbinData
    *pShaderCount          = 1;                      // gshファイルが一つだけなので固定値で設定
   // *pShaderCount          = mShaderIdx;

    // バイナリサイズ分のメモリ領域を確保する.
    pBuffer = new unsigned char [*pBinarySize];

    // メモリの確保に失敗した場合
    if ( pBuffer == NULL )
    {
        char dataSizeStrBuf[16];
        _itoa_s(*pBinarySize, dataSizeStrBuf, 16, 10);

        ShaderMsg_ShaderExceedsBufferSize msg;
        msg.SetArgument(ShaderMsg_ShaderExceedsBufferSize::ShaderSize, dataSizeStrBuf);
        msg.Send();

        return NULL;
    }

    // pBufferの総バイナリサイズ分クリアする
    memset( pBuffer, 0, *pBinarySize );

    nw::eft2::BinaryData* binData = reinterpret_cast<nw::eft2::BinaryData *>( pBuffer );
    u8* shaderBinaryBuffer        = reinterpret_cast<u8 *>( pBuffer + BinaryDataInfoSize + padding );

    binData->Init();
    binData->endian = 1;
    binData->SetBinaryTag('S', 'H', 'D', 'A');
    binData->child     = flip( constBinDataSize );
    binData->childNum  = flip_u16( mShaderIdx );               // shader variation数をセット
    binData            = (nw::eft2::BinaryData*)( (u8*)binData + constBinDataSize );

    size_t offset = shaderBinaryBuffer - (u8*)binData;

    // BinaryDataの設定
    binData->Init();
    binData->endian = 1;
    binData->SetBinaryTag('S', 'H', 'D', 'B');
    binData->binSize = flip( shaderSize );
    binData->next    = flip( BIN_DATA_NOTHING );
    binData->offset  = flip( constBinDataSize + padding );

    memcpy( shaderBinaryBuffer, shaderPtr, shaderSize );

    Profiler profilerGshFinalize("Finalize gshCompiler");

    // shader dll
    free( shaderPtr );
    gshFinalize();

    profilerGshFinalize.Stop();

    delete[] mShaderKeyArray;
    delete[] mShaderTbl;

    // TODO: resultがNULLで戻るとランタイムが落ちるので、false又はshader binary size
    //       が0の時の処理をShaderConverterHelper.cppでする必要があります。

    return pBuffer;
}

//------------------------------------------------------------------------------
//      モード毎にシェーダを生成する
//------------------------------------------------------------------------------
u32 ShaderConverter::ConvertWithMode( ShaderConverterEmitterData* pEmitterData,
                                      ShaderCompileErrorList*     pCompileErrorList,
                                      nw::eft2::ResEmitter*       emitterData,
                                      nw::eft2::ShaderKey*        shaderKey,
                                      u32                         userDefIdx )
{
    u32 retIndex = 0;
    switch (mConvertMode)
    {

        case CONVERT_FAST:
        case CONVERT_BUILD_SHADER_BINARY:
        case CONVERT_DUP_BINARY:
            retIndex = ConvertFast(
                         pEmitterData,
                         pCompileErrorList,
                         emitterData,
                         shaderKey,
                         userDefIdx );
             break;
        case CONVERT_USE_SHADER_BINARY:
            retIndex = ConvertShaderCache(
                         pEmitterData,
                         pCompileErrorList,
                         emitterData,
                         shaderKey);
            break;
    }

    return retIndex;
}

//------------------------------------------------------------------------------
//      コンバート- シェーダバイナリを使用する　
//------------------------------------------------------------------------------
u32 ShaderConverter::ConvertShaderCache(
    ShaderConverterEmitterData* pEmitterData,
    ShaderCompileErrorList*     pCompileErrorList,
    nw::eft2::ResEmitter*       emitterData,
    nw::eft2::ShaderKey*        shaderKey )
{
    u32 retIndex = 0;

    ShaderKeyInfo keyInfo;
    keyInfo.SetKey( shaderKey, mIsField, mReservedShaderIndex );

    //char msg[64];
    //
    //sprintf_s( msg, 64, "ConvertShaderCache\n");
    //OutputDebugMsg(msg);
    bool isGetInShaderCasheAlready = false;

    // ShaderCacheから頂点、フラグメントシェーダを所得する
    if ( IsUseShaderCache() == true )
    {
        s32 binNum = mShaderCache.IsFindCache( &keyInfo );
        if ( binNum != -1 )
        {
            if ( mShaderCache.GetShaderCache( binNum, &mShaderTbl[mShaderIdx] ) == true )
                isGetInShaderCasheAlready = true;
        }
    }

    // ShaderCacheに無い場合は新たに生成する。
    if ( isGetInShaderCasheAlready == false )
    {
        // shader codeを生成する
        char* setting     = (char *)shaderKey->GetCompileSetting();
        u32   settingSize = (u32)strlen( setting );

        // Compile Shader Binary
        Profiler profilerGenerateShader("ConvertShaderCache : Generate shader");
        GenerateShader( pEmitterData,
                        pCompileErrorList,
                        emitterData,
                        setting,
                        settingSize,
                        (ShaderTable*)&mShaderTbl[mShaderIdx]);
        profilerGenerateShader.Stop();
    }
    mShaderKeyArray[mShaderIdx].SetKey( shaderKey, mIsField, mReservedShaderIndex );

    retIndex          = mShaderIdx;

    mShaderIdx++;

    return retIndex;
}

//------------------------------------------------------------------------------
//      コンバート- 速さ重視　
//------------------------------------------------------------------------------
u32 ShaderConverter::ConvertFast(
    ShaderConverterEmitterData* pEmitterData,
    ShaderCompileErrorList*     pCompileErrorList,
    nw::eft2::ResEmitter*       emitterData,
    nw::eft2::ShaderKey*        shaderKey,
    u32                         userDefIdx )
{
    u32 retIndex = 0;

    // shader codeを生成する
    char* setting     = (char *)shaderKey->GetCompileSetting();
    u32   settingSize = (u32)strlen( setting );

    // Compile Shader Binary
    Profiler profilerGenerateShader("ConvertFast : Generate shader");
    GenerateShader( pEmitterData,
                    pCompileErrorList,
                    emitterData,
                    setting,
                    settingSize,
                    (ShaderTable*)&mShaderTbl[mShaderIdx],
                    userDefIdx );
    profilerGenerateShader.Stop();

    mShaderKeyArray[mShaderIdx].SetKey( shaderKey, mIsField, mReservedShaderIndex );

    mShaderKeyArray[mShaderIdx].vertexShaderIndex = mShaderTbl[mShaderIdx].GetVertexShaderId();
    mShaderKeyArray[mShaderIdx].pixelShaderIndex = mShaderTbl[mShaderIdx].GetPixelShaderId();

    retIndex = mShaderIdx;

    mShaderIdx++;

    return retIndex;
}

//------------------------------------------------------------------------------
//      シェーダを生成する　
//------------------------------------------------------------------------------
s32 ShaderConverter::GenerateShader( ShaderConverterEmitterData* pEmitterData,
                                     ShaderCompileErrorList*     pCompileErrorList,
                                     nw::eft2::ResEmitter*       emitterData,
                                     char*                       setting,
                                     u32                         settingSize,
                                     ShaderTable*                shaderTable,
                                     u32                         userShaderDef )
{
    // Compile Shader Binary
    s32 ret = CompileShader( this->szVshCodeCompositeBuffer,
                             this->szFshCodeCompositeBuffer,
                             setting,
                             settingSize,
                             pEmitterData->szOverrideShader,
                             pEmitterData->iOverrideShaderLength,
                             shaderTable,
                             true );


    if ( ret != EC_SUCCESS && ret != EC_OUTPUTWARNING )
    {
   //   result = false; // todo:ランタイムで落ちてしまうのでコメントアウトしました。

        OutputShaderCompileErrorMessage( emitterData->name, userShaderDef );

        pCompileErrorList->AddErrorLog( emitterData->name,
                                        this->szVshCodeCompositeBuffer,
                                        this->szFshCodeCompositeBuffer,
                                        shaderTable->GetLog() );

        // Compile Shader Binary
        char* settingSimple     = setting;
        u32   settingSizeSimple = settingSize;

        if ( userShaderDef > 0 )
        {
            settingSimple     = (char *)mDefaultShaderKey->GetCompileSetting();
            settingSizeSimple = (u32)strlen( settingSimple );
        }

        if ( CompileShader( this->szVshCodeCompositeBuffer,
                            this->szFshCodeCompositeBuffer,
                            settingSimple,
                            settingSizeSimple,
                            NULL,
                            0,
                            shaderTable,
                            false ) !=  EC_SUCCESS )
        {
            OutputShaderCompileErrorMessage( emitterData->name, 0 );
        }
    }
    else if ( ret == EC_OUTPUTWARNING )
    {
        OutputShaderCompileWarningMessage( emitterData->name );
        //OutputShaderCompileErrorLog( emitterData->name, shaderTbl[shaderIdx].GetLog() );

        pCompileErrorList->AddWarningLog( emitterData->name,
                                          this->szVshCodeCompositeBuffer,
                                          this->szFshCodeCompositeBuffer,
                                          shaderTable->GetLog() );
    }

    return ret;

}

//------------------------------------------------------------------------------
//      ストライプデータを取得する
//------------------------------------------------------------------------------
nw::eft2::ResEmitter* ShaderConverter::GetEmitter( u8 *pHeaderData, u8 *pEmitterData )
{
    nw::eft2::BinaryData* header = reinterpret_cast<nw::eft2::BinaryData*>(pHeaderData);
    return reinterpret_cast<nw::eft2::ResEmitter*>(header->GetBinaryData());
}

//------------------------------------------------------------------------------
//      カスタムフィールドを取得する
//------------------------------------------------------------------------------
void* ShaderConverter::GetCustomField( nw::eft2::BinaryData* binaryData )
{
    nw::eft2::BinaryData* subData = binaryData->GetSubData();
    while( subData )
    {
        if ( ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'F', 'C', 'S', 'F' ) ) )
        {
            return ( subData->GetBinaryData() );
            break;
        }

        subData = subData->GetNextData();
    }
    return NULL;
}


//------------------------------------------------------------------------------
//      エミッタプラグインを取得する
//------------------------------------------------------------------------------
void* ShaderConverter::GetEmitterPlugin( nw::eft2::BinaryData* binaryData )
{
    nw::eft2::BinaryData* subData = binaryData->GetSubData();
    while( subData )
    {
        if (   ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'E', 'P', '0', '1' ) )
            || ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'E', 'P', '0', '2' ) )
            || ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'E', 'P', '0', '3' ) )
            || ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'E', 'P', '0', '4' ) )
            || ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'E', 'P', '0', '5' ) )
            || ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'E', 'P', '0', '6' ) )
            || ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'E', 'P', '0', '7' ) )
            || ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'E', 'P', '0', '8' ) ) )
        {

            return ( subData->GetBinaryData() );
            break;
        }

        subData = subData->GetNextData();
    }
    return NULL;
}


//------------------------------------------------------------------------------
//      フィールドデータの存在チェック
//------------------------------------------------------------------------------
bool ShaderConverter::IsFieldExist( nw::eft2::BinaryData* binaryData )
{
    nw::eft2::BinaryData* subData = binaryData->GetSubData();
    while( subData )
    {
        if ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'F', 'R', 'N', 'D' ) )
        {
            return true;
            break;
        }

        subData = subData->GetNextData();
    }
    return false;
}

//------------------------------------------------------------------------------
//      コンパイルしてシェーダバイナリを生成する
//------------------------------------------------------------------------------
s32 ShaderConverter::CompileShader(
    char* shaderVsh,
    char* shaderFsh,
    char* compileSettings,
    u32   settingSize,
    char* overrideShader,
    s32   overrideShaderLength,
    ShaderTable* shaderTbl,
    bool  useCustomShader,
    bool  useStreamOut)
{
    u32   shaderVshSize    = 0;
    u32   shaderFshSize    = 0;
    s32   ret              = 0;

    memset( shaderVsh, 0, ShaderCodeBufferSize );
    memset( shaderFsh, 0, ShaderCodeBufferSize );

    // 頂点シェーダコードを生成する
    shaderVshSize = GenerateVertexShaderCode( shaderVsh,
                                              compileSettings,
                                              settingSize,
                                              NULL,
                                              0,
                                              useCustomShader,
                                              mCustomShaderIndex,
                                              mReservedShaderIndex,
                                              useStreamOut );

    // フラグメントシェーダコードを生成する
    shaderFshSize = GenerateFragmentShaderCode( shaderFsh,
                                                compileSettings,
                                                settingSize,
                                                overrideShader,
                                                overrideShaderLength,
                                                useCustomShader,
                                                mCustomShaderIndex,
                                                mReservedShaderIndex,
                                                useStreamOut );
#if 0
#ifdef NW_DEBUG
    char fileName[512];
    sprintf( (char*)&fileName, "%s\\shaderVsh%d.vs", mConverterFolderPath, mTmpShaderSourceCnt );
    FILE* fp = fopen( (char*)&fileName, "wb" );
    fwrite( shaderVsh, 1, shaderVshSize, fp );
    fclose(fp);

    memset( &fileName, 0, 512 );
    sprintf( (char*)&fileName, "%s\\shaderFsh%d.fs", mConverterFolderPath, mTmpShaderSourceCnt );
    fp = fopen( (char*)&fileName, "wb" );
    fwrite( shaderFsh, 1, shaderFshSize, fp );
    fclose(fp);
    mTmpShaderSourceCnt++;
#endif
#endif
    if ( useStreamOut == true )
    {
        char* sodata = "sysOutPos\ngl_NextBuffer\nsysOutVec\n";
        u32 sosize = (u32)strlen(sodata);

        ret = gshCompile(shaderVsh, shaderVshSize,
                         shaderFsh, shaderFshSize,
                         NULL, 0,
                         sodata, sosize,
#if CAFE_OS_SDK_VERSION >= 21104
                         NULL, 0,
#endif
                         NULL,
                         shaderTbl,
                         NULL );
    }
    else
    {
        // shader Compileする
        ret = gshCompile(shaderVsh, shaderVshSize,
                         shaderFsh, shaderFshSize,
                         NULL, 0,
                         NULL, 0,
#if CAFE_OS_SDK_VERSION >= 21104
                         NULL, 0,
#endif
                         NULL,
                         shaderTbl,
                         NULL );
    }

    return ret;
}

//------------------------------------------------------------------------------
//      頂点シェーダソースの生成
//------------------------------------------------------------------------------
u32 ShaderConverter::GenerateVertexShaderCode(
    char* shaderCodeBuf,
    char* compileSettings,
    u32   settingSize,
    char* overrideVertexShader,
    s32   overrideVertexShaderLength,
    bool  useCustomShader,
    s32   customShaderIndex,
    s32   reservedShaderIndex,
    bool  useStreamOut )
{
    CopyShaderCode shaderCode( shaderCodeBuf );

    shaderCode.CopyCode( mShaderCodes->mSpecDecShaderSrc.buf,  mShaderCodes->mSpecDecShaderSrc.size  );
    if (!useStreamOut)
    {
        shaderCode.CopyCode( compileSettings, settingSize );
    }
    else
    {
        // StreamOut用に "#define COMPUTE_SHADER" を追加する
        // ここの処理はShaderVariationに対応後に削除する
        const char* streamOutCompileSettings = "#define COMPUTE_SHADER\n";
        shaderCode.CopyCode( streamOutCompileSettings,
                             (u32)strlen( streamOutCompileSettings ) );
    }

    shaderCode.CopyCode( mShaderCodes->mParticleGlslSrc.buf,   mShaderCodes->mParticleGlslSrc.size   );
    shaderCode.CopyCode( mShaderCodes->mParticleDecVshSrc.buf, mShaderCodes->mParticleDecVshSrc.size );

    if ( overrideVertexShader && overrideVertexShaderLength != 0 )
    {
        shaderCode.CopyCode( overrideVertexShader, overrideVertexShaderLength );
    }

    // Custom shaderのコピー
    if ( useCustomShader == true )
    {
        if ( ( customShaderIndex <= 0 ) || ( useStreamOut == true ) )
        {
            if ( mShaderCodes->mGeneralVshSrc.buf  != NULL &&
                 mShaderCodes->mGeneralVshSrc.size != 0 )
            {
                shaderCode.CopyCode( mShaderCodes->mGeneralVshSrc.buf,
                                     mShaderCodes->mGeneralVshSrc.size );
            }
        }
        else
        {
            const ShaderCodeParam &shaderCodeParam =
                mShaderCodes->mCustomVshSrc[customShaderIndex - 1];

            if ( shaderCodeParam.buf  != NULL &&
                 shaderCodeParam.size != 0 )
            {
                shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
            }
        }
    }

    // Reserved shaderのコピー
    if ( ( reservedShaderIndex >= 0 ) && ( useStreamOut == false ) )
    {
        const ShaderCodeParam &shaderCodeParam =
            mShaderCodes->mReservedVshSrc[reservedShaderIndex];

        if ( shaderCodeParam.buf  != NULL &&
             shaderCodeParam.size != 0 )
        {
            shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
        }
    }

    // StreamOut shaderのコピー
    if ( useStreamOut == true )
    {
        shaderCode.CopyCode( mShaderCodes->mStreamOutDecVshSrc.buf,
                             mShaderCodes->mStreamOutDecVshSrc.size );

        if ( mShaderCodes->mGeneralCshSrc.buf != NULL &&
             mShaderCodes->mGeneralCshSrc.size != 0 )
        {
            shaderCode.CopyCode( mShaderCodes->mGeneralCshSrc.buf,
                                 mShaderCodes->mGeneralCshSrc.size );
        }

        shaderCode.CopyCode( mShaderCodes->mStreamOutVshSrc.buf,
                             mShaderCodes->mStreamOutVshSrc.size );
    }

    // 共通コードをコピー (StreamOut shader以外)
    if ( useStreamOut == false )
    {
        shaderCode.CopyCode( mShaderCodes->mVertexShaderSrc.buf,
                             mShaderCodes->mVertexShaderSrc.size );
    }

    return  shaderCode.GetShaderSize();
}

//------------------------------------------------------------------------------
//      フラグメントシェーダソースの生成
//------------------------------------------------------------------------------
u32 ShaderConverter::GenerateFragmentShaderCode(
    char* shaderCodeBuf,
    char* compileSettings,
    u32   settingSize,
    char* overrideFragmentSader,
    s32   overrideFragmentShaderLength,
    bool  useCustomShader,
    s32   customShaderIndex,
    s32   reservedShaderIndex,
    bool  useStreamOut )
{
    CopyShaderCode shaderCode( shaderCodeBuf );

    shaderCode.CopyCode( mShaderCodes->mSpecDecShaderSrc.buf,  mShaderCodes->mSpecDecShaderSrc.size  );
    if (!useStreamOut)
    {
        shaderCode.CopyCode( compileSettings, settingSize );
    }

    shaderCode.CopyCode( mShaderCodes->mParticleGlslSrc.buf,   mShaderCodes->mParticleGlslSrc.size   );
    shaderCode.CopyCode( mShaderCodes->mParticleDecFshSrc.buf, mShaderCodes->mParticleDecFshSrc.size );

    // Custum shaderのコピー
    if ( useCustomShader == true )
    {
        if ( ( customShaderIndex <= 0 ) || ( useStreamOut == true ) )
        {
            if ( mShaderCodes->mGeneralFshSrc.buf  != NULL &&
                 mShaderCodes->mGeneralFshSrc.size != 0 )
            {
                shaderCode.CopyCode( mShaderCodes->mGeneralFshSrc.buf,
                                     mShaderCodes->mGeneralFshSrc.size );
            }
        }
        else
        {
            const ShaderCodeParam &shaderCodeParam =
                mShaderCodes->mCustomFshSrc[customShaderIndex - 1];

            if ( shaderCodeParam.buf  != NULL &&
                 shaderCodeParam.size != 0 )
            {
                shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
            }
        }
    }

    // Reserved shaderのコピー
    if ( reservedShaderIndex >= 0 )
    {
        const ShaderCodeParam &shaderCodeParam =
            mShaderCodes->mReservedFshSrc[reservedShaderIndex];

        if ( shaderCodeParam.buf  != NULL &&
             shaderCodeParam.size != 0 )
        {
            shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
        }
    }

    if ( useStreamOut )
    {
        char nonMainFshShaderCode[] = "void main(){}\n";
        shaderCode.CopyCode( nonMainFshShaderCode,
                             (u32)strlen( nonMainFshShaderCode ) );

    }
    else if ( overrideFragmentSader && overrideFragmentShaderLength != 0 )
    {
        shaderCode.CopyCode( mShaderCodes->mFragShaderSrc.buf,
                             mShaderCodes->mFragShaderSrc.size );
        // CombinerShaderのコピー
        shaderCode.CopyCode( overrideFragmentSader, overrideFragmentShaderLength );
    }
    else
    {
        shaderCode.CopyCode( mShaderCodes->mFragShaderSrc.buf,
                             mShaderCodes->mFragShaderSrc.size );
    }

    return  shaderCode.GetShaderSize();
}

//------------------------------------------------------------------------------
//      シェーダソースコードの生成
//------------------------------------------------------------------------------
bool ShaderConverter::GenerateShaderCode( ShaderConverterEmitterData* pEmitterData,
                                          char**  ppVertexShader,
                                          s32* pVertexShaderSize,
                                          char**  ppFragmentShader,
                                          s32* pFragmentShaderSize,
                                          char**  ppComputeShader,
                                          s32* pComputeShaderSize,
                                          u32 userDefineIndex  )
{
    nw::eft2::ShaderKey shaderKey;

    // Get the reserved shader index.
    mReservedShaderIndex = pEmitterData->reservedShaderIndex;
    void* emitterPluginData = GetEmitterPlugin( reinterpret_cast<nw::eft2::BinaryData*>( pEmitterData->pBinaryHeader ) );
    nw::eft2::ResFieldCustom* resFieldCustom = reinterpret_cast<nw::eft2::ResFieldCustom*>( GetCustomField( reinterpret_cast<nw::eft2::BinaryData*>( pEmitterData->pBinaryHeader ) ) );

    nw::eft2::ResEmitter* emitterData =
        GetEmitter( pEmitterData->pBinaryHeader, pEmitterData->pResEmitter );

    u32 fieldFlag = 0;
    if ( IsFieldExist( reinterpret_cast<nw::eft2::BinaryData*>( pEmitterData->pBinaryHeader ) ) )
    {
        fieldFlag |= EFT_SHADER_KEY_FIELD_RANDOM;
        mIsField = true;
    }
    else
    {
        mIsField = false;
    }


    // TODO: CustomShader Codeを設定すること

    //
    // shaderKeyを作成する。
    //
    emitterData->FlipEndian();
    s32 reservedShaderIndex = ( mReservedShaderIndex >= 0 ) ? mReservedShaderIndex + 1 : mReservedShaderIndex;

    char *userDefineStr = NULL;
    if ( userDefineIndex == 1 ) userDefineStr = emitterData->shader.userShaderDefine1;
    else if ( userDefineIndex == 2 ) userDefineStr = emitterData->shader.userShaderDefine2;

    // コンバイナシェーダがない場合は、indexを-1にしておく
    if( pEmitterData->iOverrideShaderLength < 1 )
    {
        emitterData->shader.eftCombinerShaderIndex = static_cast<s32>(-1);
    }

    shaderKey.Initialize( emitterData, reservedShaderIndex, emitterPluginData, fieldFlag, resFieldCustom, userDefineStr );
    // Custom Shader Indexの設定
    mCustomShaderIndex = emitterData->shader.customShaderIndex;
    emitterData->FlipEndian();

    // shader codeを生成する
    char* setting     = (char *)shaderKey.GetCompileSetting();
    u32   settingSize = (u32)strlen( setting );

    memset( this->szVshCodeCompositeBuffer, 0, ShaderCodeBufferSize );
    memset( this->szFshCodeCompositeBuffer, 0, ShaderCodeBufferSize );

    // 頂点シェーダコードを生成する
    u32 shaderVshSize = GenerateVertexShaderCode( this->szVshCodeCompositeBuffer,
                                                  setting,
                                                  settingSize,
                                                  NULL,
                                                  0,
                                                  true,
                                                  mCustomShaderIndex,
                                                  mReservedShaderIndex );

    // フラグメントシェーダコードを生成する
    u32 shaderFshSize = GenerateFragmentShaderCode( this->szFshCodeCompositeBuffer,
                                                    setting,
                                                    settingSize,
                                                    pEmitterData->szOverrideShader,
                                                    pEmitterData->iOverrideShaderLength,
                                                    true,
                                                    mCustomShaderIndex,
                                                    mReservedShaderIndex );

    // Set up the shader code and size.
    *pVertexShaderSize = shaderVshSize;
    *ppVertexShader = new char[shaderVshSize + 1];
    memcpy_s(*ppVertexShader, shaderVshSize + 1, this->szVshCodeCompositeBuffer, shaderVshSize);
    (*ppVertexShader)[shaderVshSize] = '\0';

    *pFragmentShaderSize = shaderFshSize;
    *ppFragmentShader = new char[shaderFshSize + 1];
    memcpy_s(*ppFragmentShader, shaderFshSize + 1, this->szFshCodeCompositeBuffer, shaderFshSize);
    (*ppFragmentShader)[shaderFshSize] = '\0';

    // コンピュートシェーダを生成する (バッファはバーテックスシェーダのものを使いまわす)
    memset( this->szVshCodeCompositeBuffer, 0, ShaderCodeBufferSize );
    u32 shaderCshSize = GenerateVertexShaderCode( this->szVshCodeCompositeBuffer,
                                                  NULL,
                                                  0,
                                                  NULL,
                                                  0,
                                                  true,
                                                  mCustomShaderIndex,
                                                  mReservedShaderIndex,
                                                  true );

    *pComputeShaderSize = shaderCshSize;
    *ppComputeShader = new char[shaderCshSize + 1];
    memcpy_s(*ppComputeShader, shaderCshSize + 1, this->szVshCodeCompositeBuffer, shaderCshSize);
    (*ppComputeShader)[shaderCshSize] = '\0';

    return true;
}

//------------------------------------------------------------------------------
//      シェーダアセンブリコードの生成
//------------------------------------------------------------------------------
bool ShaderConverter::GenerateAssemblyCode( ShaderConverterEmitterData* pEmitterData,
                                            ShaderCompileErrorList *pCompileErrorList,
                                            char**  ppVertexAssembly,
                                            s32* pVertexAssemblySize,
                                            char**  ppFragmentAssembly,
                                            s32* pFragmentAssemblySize,
                                            char**  ppComputeAssembly,
                                            s32* pComputeAssemblySize,
                                            u32 userDefineIndex )
{
    //
    // dllを初期化する。
    //
    Profiler p1("gshInitialize");
    gshInitialize( this->mConverterFolderPath, 1 );
    p1.Stop();

    nw::eft2::ShaderKey shaderKey;

    // Get the reserved shader index.
    mReservedShaderIndex = pEmitterData->reservedShaderIndex;
    void* emitterPluginData = GetEmitterPlugin( reinterpret_cast<nw::eft2::BinaryData*>( pEmitterData->pBinaryHeader ) );
    nw::eft2::ResFieldCustom* resFieldCustom = reinterpret_cast<nw::eft2::ResFieldCustom*>( GetCustomField( reinterpret_cast<nw::eft2::BinaryData*>( pEmitterData->pBinaryHeader ) ) );

    nw::eft2::ResEmitter* emitterData =
        GetEmitter( pEmitterData->pBinaryHeader, pEmitterData->pResEmitter );

    u32 fieldFlag = 0;
    if ( IsFieldExist( reinterpret_cast<nw::eft2::BinaryData*>( pEmitterData->pBinaryHeader ) ) )
    {
        fieldFlag |= EFT_SHADER_KEY_FIELD_RANDOM;
        mIsField = true;
    }
    else
    {
        mIsField = false;
    }

    //
    // shaderKeyを作成する。
    //
    Profiler p3("Flip endian and make shader key");
    emitterData->FlipEndian();
    s32 reservedShaderIndex = ( mReservedShaderIndex >= 0 ) ? mReservedShaderIndex + 1 : mReservedShaderIndex;

    char *userDefineStr = NULL;
    if ( userDefineIndex == 1 ) userDefineStr = emitterData->shader.userShaderDefine1;
    else if ( userDefineIndex == 2 ) userDefineStr = emitterData->shader.userShaderDefine2;

    // コンバイナシェーダがない場合は、indexを-1にしておく
    if( pEmitterData->iOverrideShaderLength < 1 )
    {
        emitterData->shader.eftCombinerShaderIndex = static_cast<s32>(-1);
    }

    shaderKey.Initialize( emitterData, reservedShaderIndex, emitterPluginData, fieldFlag, resFieldCustom, userDefineStr );

    // Custom Shader Indexの設定
    mCustomShaderIndex = emitterData->shader.customShaderIndex;
    emitterData->FlipEndian();
    p3.Stop();

    Profiler p4("Compose source code");
    // shader codeを生成する
    char* setting     = (char *)shaderKey.GetCompileSetting();
    u32   settingSize = (u32)strlen( setting );

    memset( this->szVshCodeCompositeBuffer, 0, ShaderCodeBufferSize );
    memset( this->szFshCodeCompositeBuffer, 0, ShaderCodeBufferSize );

        // 頂点シェーダコードを生成する
    u32 shaderVshSize = GenerateVertexShaderCode( this->szVshCodeCompositeBuffer,
                                                  setting,
                                                  settingSize,
                                                  NULL,
                                                  0,
                                                  true,
                                                  mCustomShaderIndex,
                                                  mReservedShaderIndex );

    // フラグメントシェーダコードを生成する
    u32 shaderFshSize = GenerateFragmentShaderCode( this->szFshCodeCompositeBuffer,
                                                    setting,
                                                    settingSize,
                                                    pEmitterData->szOverrideShader,
                                                    pEmitterData->iOverrideShaderLength,
                                                    true,
                                                    mCustomShaderIndex,
                                                    mReservedShaderIndex );

    p4.Stop();

    bool result = false;
    {
        // Compile Shader Binary
        ShaderAsmTable asmCodeTable;

        // shader Compileする
        ShaderTable shaderTbl;
        Profiler p5("Compile shader");

        s32 compileResult = gshCompile( this->szVshCodeCompositeBuffer, shaderVshSize,
                                        this->szFshCodeCompositeBuffer, shaderFshSize,
                                        NULL, 0,
                                        NULL, 0,
    #if CAFE_OS_SDK_VERSION >= 21104
                                        NULL, 0,
    #endif
                                        NULL,
                                        &shaderTbl,
                                        &asmCodeTable );

        if ( compileResult == EC_SUCCESS || compileResult == EC_OUTPUTWARNING )
        {
            p5.Stop();

            Profiler p6("Copy shader code");
            // Set up the shader code and size.
            *pVertexAssemblySize = (s32)strlen( asmCodeTable.vertexAsmCode );
            *ppVertexAssembly = new char[*pVertexAssemblySize + 1];
            memcpy_s(*ppVertexAssembly, *pVertexAssemblySize + 1, asmCodeTable.vertexAsmCode, *pVertexAssemblySize);
            (*ppVertexAssembly)[*pVertexAssemblySize] = '\0';

            *pFragmentAssemblySize = (s32)strlen( asmCodeTable.fragmentAsmCode );
            *ppFragmentAssembly = new char[*pFragmentAssemblySize + 1];
            memcpy_s(*ppFragmentAssembly, *pFragmentAssemblySize + 1, asmCodeTable.fragmentAsmCode, *pFragmentAssemblySize);
            (*ppFragmentAssembly)[*pFragmentAssemblySize] = '\0';
            p6.Stop();

            if ( compileResult == EC_OUTPUTWARNING )
            {
                pCompileErrorList->AddWarningLog( emitterData->name,
                                                  this->szVshCodeCompositeBuffer,
                                                  this->szFshCodeCompositeBuffer,
                                                  shaderTbl.GetLog() );
            }

            result = true;
        }
        else
        {
            pCompileErrorList->AddErrorLog( emitterData->name,
                                            this->szVshCodeCompositeBuffer,
                                            this->szFshCodeCompositeBuffer,
                                            shaderTbl.GetLog() );

            *pVertexAssemblySize = 0;
            *pFragmentAssemblySize = 0;

            result = false;
        }
    }

    if ( result )
    {
        // コンピュートシェーダを生成する (バッファはバーテックスシェーダのものを使いまわす)
        memset( this->szVshCodeCompositeBuffer, 0, ShaderCodeBufferSize );
        u32 shaderCshSize = GenerateVertexShaderCode( this->szVshCodeCompositeBuffer,
                                                      NULL,
                                                      0,
                                                      NULL,
                                                      0,
                                                      true,
                                                      mCustomShaderIndex,
                                                      mReservedShaderIndex,
                                                      true );

        // Compile Shader Binary
        ShaderAsmTable asmCodeTable;

        // shader Compileする
        ShaderTable shaderTbl;

        char* sodata = "sysOutPos\ngl_NextBuffer\nsysOutVec\n";
        u32 sosize = (u32)strlen(sodata);

        s32 compileResult = gshCompile( this->szVshCodeCompositeBuffer, shaderCshSize,
                                        this->szFshCodeCompositeBuffer, shaderFshSize,
                                        NULL, 0,
                                        sodata, sosize,
#if CAFE_OS_SDK_VERSION >= 21104
                                        NULL, 0,
#endif
                                        NULL,
                                        &shaderTbl,
                                        &asmCodeTable );

        if ( compileResult == EC_SUCCESS || compileResult == EC_OUTPUTWARNING )
        {
            *pComputeAssemblySize = (s32)strlen( asmCodeTable.vertexAsmCode );
            *ppComputeAssembly = new char[*pComputeAssemblySize + 1];
            memcpy_s(*ppComputeAssembly, *pComputeAssemblySize + 1, asmCodeTable.vertexAsmCode, *pComputeAssemblySize);
            (*ppComputeAssembly)[*pComputeAssemblySize] = '\0';

            if ( compileResult == EC_OUTPUTWARNING )
            {
                pCompileErrorList->AddWarningLog( emitterData->name,
                                                  this->szVshCodeCompositeBuffer,
                                                  NULL,
                                                  shaderTbl.GetLog() );
            }

            result = true;
        }
        else
        {
            pCompileErrorList->AddErrorLog( emitterData->name,
                                            this->szVshCodeCompositeBuffer,
                                            NULL,
                                            shaderTbl.GetLog() );

            *pComputeAssemblySize = 0;

            result = false;
        }
    }

    Profiler p7("Finalize compiler");
    gshFinalize();
    p7.Stop();

    return result;
}

//------------------------------------------------------------------------------
//     テキストエディタに表示する
//------------------------------------------------------------------------------
void ShaderConverter::OpenTextFile( char* outPath )
{
    ShaderConverterMsg_OpenTextFile msg;
    msg.SetArgument(ShaderConverterMsg_OpenTextFile::FilePath, outPath);
    msg.Send();
}

//------------------------------------------------------------------------------
// Output shader compile error message.
//------------------------------------------------------------------------------
void ShaderConverter::OutputShaderCompileErrorMessage( const char *szEmitterName, u32 userShaderDef )
{
    ShaderConverterMsg_ShaderCompileError msg;
    msg.SetArgument( ShaderConverterMsg_ShaderCompileError::Destination, "Console,LogView" );
    msg.SetArgument( ShaderConverterMsg_ShaderCompileError::LogLevel, "Error" );
    msg.SetArgument( ShaderConverterMsg_ShaderCompileError::EmitterName, szEmitterName );
    if ( userShaderDef == 1 )
    {
        msg.SetArgument( ShaderConverterMsg_ShaderCompileError::UserShaderDefine, "userShaderDef 1" );
    }
    if ( userShaderDef == 2 )
    {
        msg.SetArgument( ShaderConverterMsg_ShaderCompileError::UserShaderDefine, "userShaderDef 2" );
    }

    msg.Send();
}

//------------------------------------------------------------------------------
// Output error message when shader compile failed even without custom shader.
//------------------------------------------------------------------------------
void ShaderConverter::OutputShaderRecompileErrorMessage( const char *szEmitterName )
{
    ShaderConverterMsg_ShaderRecompileError msg;
    msg.SetArgument( ShaderConverterMsg_ShaderRecompileError::Destination, "Console,LogView" );
    msg.SetArgument( ShaderConverterMsg_ShaderRecompileError::LogLevel, "Error" );
    msg.SetArgument( ShaderConverterMsg_ShaderRecompileError::EmitterName, szEmitterName );
    msg.Send();
}

//------------------------------------------------------------------------------
// Output shader compile warning message.
//------------------------------------------------------------------------------
void ShaderConverter::OutputShaderCompileWarningMessage( const char *szEmitterName )
{
    ShaderConverterMsg_ShaderCompileWarning msg;
    msg.SetArgument( ShaderConverterMsg_ShaderCompileWarning::Destination, "Console,LogView" );
    msg.SetArgument( ShaderConverterMsg_ShaderCompileWarning::LogLevel, "Warning" );
    msg.SetArgument( ShaderConverterMsg_ShaderCompileWarning::EmitterName, szEmitterName );
    msg.Send();
}

//------------------------------------------------------------------------------
// Output shader compile error log to debug console.
//------------------------------------------------------------------------------
void ShaderConverter::OutputShaderCompileErrorLog( const char *szEmitterName, const char *szErrorLog )
{
    ShaderConverterMsg_ShaderCompileErrorLog msg;
    msg.SetArgument( ShaderConverterMsg_ShaderCompileErrorLog::Destination, "DebugConsole" );
    msg.SetArgument( ShaderConverterMsg_ShaderCompileErrorLog::LogLevel, "Warning" );
    msg.SetArgument( ShaderConverterMsg_ShaderCompileErrorLog::EmitterName, szEmitterName );
    msg.SetArgument( ShaderConverterMsg_ShaderCompileErrorLog::ErrorLog, szErrorLog );
    msg.Send();
}

//------------------------------------------------------------------------------
// Output debug messge to debug console.
//------------------------------------------------------------------------------
void ShaderConverter::OutputDebugMsg( const char* msgString )
{
    ShaderMsg_General msg;
    msg.SetArgument( ShaderMsg_General::LogLevel, "Debug" );
    msg.SetArgument( ShaderMsg_General::Destination, "DebugConsole" );
    msg.SetArgument( ShaderMsg_General::OutputMsg, msgString );
    msg.Send();
}

} // ShaderConverter
} // EffectMaker

