﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShaderCodeGeneratorGeneric_PCH.h"
#include <nn/vfx/vfx_Enum.h>
#include "vfx_ShaderKey.h"

namespace nn {
namespace vfx {
namespace detail {

// シェーダセッティング用バッファ
char ShaderKey::g_CompileSetting[ ShaderCompileSettings_BufferSize ];

//---------------------------------------------------------------------------
//! @brief  シェーダキー文字列生成クラス
//---------------------------------------------------------------------------
struct _ShaderKeyString
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンパイルセッティングを取得します。
    //---------------------------------------------------------------------------
    void Initialize( char* top, size_t size ) NN_NOEXCEPT
    {
        keyTop = top;
        keySize = static_cast<uint32_t>(size);
        current = top;
    }

    //---------------------------------------------------------------------------
    //! @brief  コンパイルセッティングを取得します。
    //---------------------------------------------------------------------------
    void AddKey( const char* format, ... ) NN_NOEXCEPT
    {
        va_list vargs;
        va_start( vargs, format );
        current += vsprintf_s( current, keySize - ( static_cast< uint32_t >( reinterpret_cast< uint64_t >( current ) - reinterpret_cast< uint64_t >( keyTop ) ) ), format, vargs );
        va_end( vargs );
    }

    //---------------------------------------------------------------------------
    //! @brief  EOF。
    //---------------------------------------------------------------------------
    void Eof() NN_NOEXCEPT
    {
        *current = 0;
    }

    char* GetKeyStr() const NN_NOEXCEPT
    {
        return keyTop;
    }

private:
    char* keyTop;
    uint32_t   keySize;
    char* current;
};


//---------------------------------------------------------------------------
//! @brief  頂点シェーダキー定義1
//---------------------------------------------------------------------------
enum : uint64_t
{
    // 以下は 通常 & 制限 版スペックで利用できる定義
    _PARTICLE_TYPE_BILLBOARD                = ( static_cast< uint64_t >( 0x01 ) << 0 ),    //!< ビルボード
    _PARTICLE_TYPE_COMPLEX_BILLBOARD        = ( static_cast< uint64_t >( 0x01 ) << 1 ),    //!< 豪華版ビルボード
    _PARTICLE_TYPE_Y_BILLBOARD              = ( static_cast< uint64_t >( 0x01 ) << 2 ),    //!< Y軸ビルボード
    _PARTICLE_TYPE_POLYGON_XY               = ( static_cast< uint64_t >( 0x01 ) << 3 ),    //!< XYポリゴン
    _PARTICLE_TYPE_POLYGON_XZ               = ( static_cast< uint64_t >( 0x01 ) << 4 ),    //!< XZポリゴン
    _PARTICLE_TYPE_VEL_LOOK                 = ( static_cast< uint64_t >( 0x01 ) << 5 ),    //!< ディレクショナルY
    _PARTICLE_TYPE_VEL_LOOK_POLYGON         = ( static_cast< uint64_t >( 0x01 ) << 6 ),    //!< ディレクショナル
    _EMITTER_FOLLOW_TYPE_ALL                = ( static_cast< uint64_t >( 0x01 ) << 7 ),    //!< 完全追従
    _EMITTER_FOLLOW_TYPE_NONE               = ( static_cast< uint64_t >( 0x01 ) << 8 ),    //!< 追従なし
    _EMITTER_FOLLOW_TYPE_POS                = ( static_cast< uint64_t >( 0x01 ) << 9 ),    //!< 位置のみ追従
    _PARTICLE                               = ( static_cast< uint64_t >( 0x01 ) << 10 ),   //!< パーティクル
    _PRIMITIVE                              = ( static_cast< uint64_t >( 0x01 ) << 11 ),   //!< プリミティブ

    // 以下は通常版スペックで利用できる定義
    _CALC_TYPE_CPU                          = ( static_cast< uint64_t >( 0x01 ) << 12 ),    //!< CPUエミッタ
    _CALC_TYPE_GPU_TIME                     = ( static_cast< uint64_t >( 0x01 ) << 13 ),    //!< GPUタイムエミッタ
    _CALC_TYPE_GPU_SO                       = ( static_cast< uint64_t >( 0x01 ) << 14 ),    //!< GPU StreamOut エミッタ
    _SHADER_ANIM                            = ( static_cast< uint64_t >( 0x01 ) << 15 ),    //!< シェーダアニメ
    _WORLD_GRAVITY                          = ( static_cast< uint64_t >( 0x01 ) << 16 ),    //!< 世界座標系で重力を計算する
    _CAMERA_OFFSET                          = ( static_cast< uint64_t >( 0x01 ) << 17 ),    //!< カメラオフセット
    _DEPTH_OFFSET                           = ( static_cast< uint64_t >( 0x01 ) << 18 ),    //!< デプスオフセット
    _USE_ROTATE                             = ( static_cast< uint64_t >( 0x01 ) << 19 ),    //!< パーティクル回転
    _ROTATE_YZX                             = ( static_cast< uint64_t >( 0x01 ) << 20 ),    //!< YZX回転
    _ROTATE_XYZ                             = ( static_cast< uint64_t >( 0x01 ) << 21 ),    //!< XYZ回転
    _ROTATE_ZXY                             = ( static_cast< uint64_t >( 0x01 ) << 22 ),    //!< ZXY回転
    _COLOR_0_FIXED                          = ( static_cast< uint64_t >( 0x01 ) << 23 ),    //!< カラー0 固定
    _COLOR_0_RANDOM                         = ( static_cast< uint64_t >( 0x01 ) << 24 ),    //!< カラー0 ランダム
    _COLOR0_INHERIT                         = ( static_cast< uint64_t >( 0x01 ) << 25 ),    //!< カラー0 継承
    _COLOR_1_FIXED                          = ( static_cast< uint64_t >( 0x01 ) << 26 ),    //!< カラー1 固定
    _COLOR_1_RANDOM                         = ( static_cast< uint64_t >( 0x01 ) << 27 ),    //!< カラー1 ランダム
    _COLOR1_INHERIT                         = ( static_cast< uint64_t >( 0x01 ) << 28 ),    //!< カラー1 継承
    _ALPHA_0_FIXED                          = ( static_cast< uint64_t >( 0x01 ) << 29 ),    //!< アルファ0 固定
    _ALPHA0_INHERIT                         = ( static_cast< uint64_t >( 0x01 ) << 30 ),    //!< アルファ0 継承
    _ALPHA_1_FIXED                          = ( static_cast< uint64_t >( 0x01 ) << 31 ),    //!< アルファ1 固定
    _ALPHA1_INHERIT                         = ( static_cast< uint64_t >( 0x01 ) << 32 ),    //!< アルファ1 継承
    _NEAR_DIST_ALPHA                        = ( static_cast< uint64_t >( 0x01 ) << 33 ),    //!< 近距離アルファ
    _FAR_DIST_ALPHA                         = ( static_cast< uint64_t >( 0x01 ) << 34 ),    //!< 近距離アルファ
    _MASKING_CHECK                          = ( static_cast< uint64_t >( 0x01 ) << 35 ),    //!< 遮蔽チェック
    _VERTEX_SOFT                            = ( static_cast< uint64_t >( 0x01 ) << 36 ),    //!< 頂点ソフト
    _VERTEX_FRESNEL_ALPHA                   = ( static_cast< uint64_t >( 0x01 ) << 37 ),    //!< 頂点ソフト
    _FLUCTUATION_ALPHA_ENABLE               = ( static_cast< uint64_t >( 0x01 ) << 38 ),    //!< アルファ揺らぎ 有効/無効
    _FLUCTUATION_SCALE_ENABLE               = ( static_cast< uint64_t >( 0x01 ) << 39 ),    //!< スケール揺らぎ 有効/無効
    _FLUCTUATION_SCALE_Y_SEPARATE           = ( static_cast< uint64_t >( 0x01 ) << 40 ),    //!< スケール揺らぎYを別個で計算するか
    _PARTICLE_SCALE_LIMIT_NEAR              = ( static_cast< uint64_t >( 0x01 ) << 41 ),    //!< カメラ距離スケール「近」
    _VELOCITY_SCALE_Y                       = ( static_cast< uint64_t >( 0x01 ) << 42 ),    //!< 速度スケールY
    _TEXTURE0_USE_UV1                       = ( static_cast< uint64_t >( 0x01 ) << 43 ),    //!< テクスチャ0 uv1を使用
    _TEXTURE1_USE_UV1                       = ( static_cast< uint64_t >( 0x01 ) << 44 ),    //!< テクスチャ1 uv1を使用
    _TEXTURE2_USE_UV1                       = ( static_cast< uint64_t >( 0x01 ) << 45 ),    //!< テクスチャ2 uv1を使用
    _TEX_0_SHIFT_ANIM                       = ( static_cast< uint64_t >( 0x01 ) << 46 ),    //!< テクスチャ0 座標アニメ
    _TEX_1_SHIFT_ANIM                       = ( static_cast< uint64_t >( 0x01 ) << 47 ),    //!< テクスチャ1 座標アニメ
    _TEX_2_SHIFT_ANIM                       = ( static_cast< uint64_t >( 0x01 ) << 48 ),    //!< テクスチャ2 座標アニメ
    _TEX_0_PATTERN_ANIM                     = ( static_cast< uint64_t >( 0x01 ) << 49 ),    //!< テクスチャ0 パタンアニメ
    _TEX_1_PATTERN_ANIM                     = ( static_cast< uint64_t >( 0x01 ) << 50 ),    //!< テクスチャ1 パタンアニメ
    _TEX_2_PATTERN_ANIM                     = ( static_cast< uint64_t >( 0x01 ) << 51 ),    //!< テクスチャ2 パタンアニメ
    _TEX_0_SPHERE_MAP                       = ( static_cast< uint64_t >( 0x01 ) << 52 ),    //!< テクスチャ0 スフィアマップ
    _TEX_1_SPHERE_MAP                       = ( static_cast< uint64_t >( 0x01 ) << 53 ),    //!< テクスチャ1 スフィアマップ
    _TEX_2_SPHERE_MAP                       = ( static_cast< uint64_t >( 0x01 ) << 54 ),    //!< テクスチャ2 スフィアマップ
    _TEX_0_SHIFT_ANIM_ROTATE                = ( static_cast< uint64_t >( 0x01 ) << 55 ),    //!< テクスチャ1 座標アニメ回転
    _TEX_1_SHIFT_ANIM_ROTATE                = ( static_cast< uint64_t >( 0x01 ) << 56 ),    //!< テクスチャ2 座標アニメ回転
    _TEX_2_SHIFT_ANIM_ROTATE                = ( static_cast< uint64_t >( 0x01 ) << 57 ),    //!< テクスチャ3 座標アニメ回転
    _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL     = ( static_cast< uint64_t >( 0x01 ) << 58 ),    //!< Y軸ビルボード（カメラに平行）
    _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL = ( static_cast< uint64_t >( 0x01 ) << 59 ),    //!< 立体風ビルボード（ディレクショナル版）
    _PARTICLE_TYPE_WITH_SCALE_Z             = ( static_cast< uint64_t >( 0x01 ) << 60 ),    //!< 立体風ビルボード
    _PARTICLE_SCALE_LIMIT_FAR               = ( static_cast< uint64_t >( 0x01 ) << 61 ),    //!< カメラ距離スケール「遠」
};

//---------------------------------------------------------------------------
//! @brief  頂点シェーダキー定義2
//---------------------------------------------------------------------------
enum : uint64_t
{
    _FIELD_RANDOM                           = ( static_cast< uint64_t >( 0x01 ) << 0  ),    //!< フィールドランダム
    _TEX0_KEEP_TEXTURE_SIZE_U               = ( static_cast< uint64_t >( 0x01 ) << 1  ),    //!< テクスチャ0 描画サイズをスケール値によって変えない
    _TEX1_KEEP_TEXTURE_SIZE_U               = ( static_cast< uint64_t >( 0x01 ) << 2  ),    //!< テクスチャ1 描画サイズをスケール値によって変えない
    _TEX2_KEEP_TEXTURE_SIZE_U               = ( static_cast< uint64_t >( 0x01 ) << 3  ),    //!< テクスチャ2 描画サイズをスケール値によって変えない
    _TEX0_KEEP_TEXTURE_SIZE_V               = ( static_cast< uint64_t >( 0x01 ) << 4  ),    //!< テクスチャ0 描画サイズをスケール値によって変えない
    _TEX1_KEEP_TEXTURE_SIZE_V               = ( static_cast< uint64_t >( 0x01 ) << 5  ),    //!< テクスチャ1 描画サイズをスケール値によって変えない
    _TEX2_KEEP_TEXTURE_SIZE_V               = ( static_cast< uint64_t >( 0x01 ) << 6  ),    //!< テクスチャ2 描画サイズをスケール値によって変えない
    _CONNECT_PTCL_SCALE_TO_Z_OFFSET         = ( static_cast< uint64_t >( 0x01 ) << 7  ),    //!< ZオフセットとPtclスケールを連動
    _USE_TEXCOORD_ATTR                      = ( static_cast< uint64_t >( 0x01 ) << 8  ),    //!< テクスチャ座標アトリビュート入力
    _USE_NORMAL_ATTR                        = ( static_cast< uint64_t >( 0x01 ) << 9  ),    //!< 法線アトリビュート入力
    _USE_TANGENT_ATTR                       = ( static_cast< uint64_t >( 0x01 ) << 10 ),    //!< 接線アトリビュート入力
    _USE_COLOR0_ATTR                        = ( static_cast< uint64_t >( 0x01 ) << 11 ),    //!< カラー0アトリビュート入力
    _USE_COLOR1_ATTR                        = ( static_cast< uint64_t >( 0x01 ) << 12 ),    //!< カラー1アトリビュート入力
    _LINK_FOVY_TO_SCALE_LIMIT               = ( static_cast< uint64_t >( 0x01 ) << 13 ),    //!< カメラ距離スケール制限に画角を連動させる
    _AVOID_Z_FIGHTING                       = ( static_cast< uint64_t >( 0x01 ) << 14 ),    //!< Z-Fighting 抑制
    _CAMERA_OFFSET_FIXED_SIZE               = ( static_cast< uint64_t >( 0x01 ) << 15 ),    //!< サイズ固定カメラオフセット
};


//---------------------------------------------------------------------------
//! @brief  ピクセルシェーダキー定義1
//---------------------------------------------------------------------------
enum : uint64_t
{
    _COLOR_PROCESS_COLOR                    = ( static_cast< uint64_t >( 0x01 ) << 0  ),    //!< Color0
    _COLOR_PROCESS_TEXTURE                  = ( static_cast< uint64_t >( 0x01 ) << 1  ),    //!< Color0 * Texture
    _COLOR_PROCESS_TEXTURE_INTERPOLATE      = ( static_cast< uint64_t >( 0x01 ) << 2  ),    //!< Texture * Color0 + ( 1 - Texture ) * Color1
    _COLOR_PROCESS_TEXTURE_ADD              = ( static_cast< uint64_t >( 0x01 ) << 3  ),    //!< Texture * Color0 + Color1
    _ALPHA_PROCESS_MOD                      = ( static_cast< uint64_t >( 0x01 ) << 4  ),    //!< Alpha0 * Texture
    _ALPHA_PROCESS_SUB                      = ( static_cast< uint64_t >( 0x01 ) << 5  ),    //!< ( Texture - ( 1 - Alpha0 ) ) * 2
    _ALPHA_PROCESS_MOD_A0_A1                = ( static_cast< uint64_t >( 0x01 ) << 6  ),    //!< Texture * Alpha0 * Alpha1
    _ALPHA_PROCESS_SUB_A0_MOD_A1            = ( static_cast< uint64_t >( 0x01 ) << 7  ),    //!< ( Texture - Alpha0 ) * Alpha1
    _TEX0_COLOR_INPUT_ONE                   = ( static_cast< uint64_t >( 0x01 ) << 8  ),    //!< テクスチャ0 入力カラーを1.0で置き換え
    _TEX1_COLOR_INPUT_ONE                   = ( static_cast< uint64_t >( 0x01 ) << 9  ),    //!< テクスチャ1 入力カラーを1.0で置き換え
    _TEX2_COLOR_INPUT_ONE                   = ( static_cast< uint64_t >( 0x01 ) << 10 ),    //!< テクスチャ2 入力カラーを1.0で置き換え
    _TEX0_ALPHA_INPUT_RED                   = ( static_cast< uint64_t >( 0x01 ) << 11 ),    //!< テクスチャ0 入力アルファをR値で置き換え
    _TEX0_ALPHA_INPUT_ONE                   = ( static_cast< uint64_t >( 0x01 ) << 12 ),    //!< テクスチャ0 入力アルファを1.0で置き換え
    _TEX1_ALPHA_INPUT_RED                   = ( static_cast< uint64_t >( 0x01 ) << 13 ),    //!< テクスチャ1 入力アルファをR値で置き換え
    _TEX1_ALPHA_INPUT_ONE                   = ( static_cast< uint64_t >( 0x01 ) << 14 ),    //!< テクスチャ1 入力アルファを1.0で置き換え
    _TEX2_ALPHA_INPUT_RED                   = ( static_cast< uint64_t >( 0x01 ) << 15 ),    //!< テクスチャ2 入力アルファをR値で置き換え
    _TEX2_ALPHA_INPUT_ONE                   = ( static_cast< uint64_t >( 0x01 ) << 16 ),    //!< テクスチャ2 入力アルファを1.0で置き換え
    _SHADER_TYPE_NORMAL                     = ( static_cast< uint64_t >( 0x01 ) << 17 ),    //!< 通常シェーダ
    _SHADER_TYPE_REFRACT                    = ( static_cast< uint64_t >( 0x01 ) << 18 ),    //!< 屈折シェーダ
    _SHADER_TYPE_DISTORTION                 = ( static_cast< uint64_t >( 0x01 ) << 19 ),    //!< 歪みシェーダ
    _TEXTURE0_ENABLE                        = ( static_cast< uint64_t >( 0x01 ) << 20 ),    //!< テクスチャ2 を有効にする
    _TEXTURE1_ENABLE                        = ( static_cast< uint64_t >( 0x01 ) << 21 ),    //!< テクスチャ1 を有効にする
    _TEXTURE2_ENABLE                        = ( static_cast< uint64_t >( 0x01 ) << 22 ),    //!< テクスチャ2 を有効にする
    _TEXTURE1_COLOR_BLEND_MOD               = ( static_cast< uint64_t >( 0x01 ) << 23 ),    //!< テクスチャ0RGB * テクスチャ1RGB
    _TEXTURE1_COLOR_BLEND_ADD               = ( static_cast< uint64_t >( 0x01 ) << 24 ),    //!< テクスチャ0RGB + テクスチャ1RGB
    _TEXTURE1_COLOR_BLEND_SUB               = ( static_cast< uint64_t >( 0x01 ) << 25 ),    //!< テクスチャ0RGB - テクスチャ1RGB
    _TEXTURE2_COLOR_BLEND_MOD               = ( static_cast< uint64_t >( 0x01 ) << 26 ),    //!< テクスチャ0RGB * テクスチャ2RGB
    _TEXTURE2_COLOR_BLEND_ADD               = ( static_cast< uint64_t >( 0x01 ) << 27 ),    //!< テクスチャ0RGB + テクスチャ2RGB
    _TEXTURE2_COLOR_BLEND_SUB               = ( static_cast< uint64_t >( 0x01 ) << 28 ),    //!< テクスチャ0RGB - テクスチャ2RGB
    _TEXTURE1_ALPHA_BLEND_MOD               = ( static_cast< uint64_t >( 0x01 ) << 29 ),    //!< テクスチャ0A * テクスチャ1A
    _TEXTURE1_ALPHA_BLEND_ADD               = ( static_cast< uint64_t >( 0x01 ) << 30 ),    //!< テクスチャ0A + テクスチャ1A
    _TEXTURE1_ALPHA_BLEND_SUB               = ( static_cast< uint64_t >( 0x01 ) << 31 ),    //!< テクスチャ0A - テクスチャ1A
    _TEXTURE2_ALPHA_BLEND_MOD               = ( static_cast< uint64_t >( 0x01 ) << 32 ),    //!< テクスチャ0A * テクスチャ2A
    _TEXTURE2_ALPHA_BLEND_ADD               = ( static_cast< uint64_t >( 0x01 ) << 33 ),    //!< テクスチャ0A + テクスチャ2A
    _TEXTURE2_ALPHA_BLEND_SUB               = ( static_cast< uint64_t >( 0x01 ) << 34 ),    //!< テクスチャ0A - テクスチャ2A
    _PRIMITIVE_COLOR_BLEND_MOD              = ( static_cast< uint64_t >( 0x01 ) << 35 ),    //!< プリミティブカラー 乗算
    _PRIMITIVE_COLOR_BLEND_ADD              = ( static_cast< uint64_t >( 0x01 ) << 36 ),    //!< プリミティブカラー 加算
    _PRIMITIVE_COLOR_BLEND_SUB              = ( static_cast< uint64_t >( 0x01 ) << 37 ),    //!< プリミティブカラー 減算
    _PRIMITIVE_ALPHA_BLEND_MOD              = ( static_cast< uint64_t >( 0x01 ) << 38 ),    //!< プリミティブアルファ 乗算
    _PRIMITIVE_ALPHA_BLEND_ADD              = ( static_cast< uint64_t >( 0x01 ) << 39 ),    //!< プリミティブアルファ 加算
    _PRIMITIVE_ALPHA_BLEND_SUB              = ( static_cast< uint64_t >( 0x01 ) << 40 ),    //!< プリミティブアルファ 減算
    _PRIMITIVE_COLOR_INPUT_SRC              = ( static_cast< uint64_t >( 0x01 ) << 41 ),    //!< プリミティブ入力カラーをR値で置き換え
    _PRIMITIVE_COLOR_INPUT_ONE              = ( static_cast< uint64_t >( 0x01 ) << 42 ),    //!< プリミティブ入力カラーを1.0で置き換え
    _PRIMITIVE_ALPHA_INPUT_A                = ( static_cast< uint64_t >( 0x01 ) << 43 ),    //!< プリミティブ入力アルファをR値で置き換え
    _PRIMITIVE_ALPHA_INPUT_RED              = ( static_cast< uint64_t >( 0x01 ) << 44 ),    //!< プリミティブ入力アルファを1.0で置き換え
    _PRIMITIVE_ALPHA_INPUT_ONE              = ( static_cast< uint64_t >( 0x01 ) << 45 ),    //!< プリミティブ入力アルファを1.0で置き換え
    _FRAGMENT_SOFT                          = ( static_cast< uint64_t >( 0x01 ) << 46 ),    //!< ピクセルソフト
    _DECAL                                  = ( static_cast< uint64_t >( 0x01 ) << 47 ),    //!< デカール
    _FRAGMENT_FRESNEL_ALPHA                 = ( static_cast< uint64_t >( 0x01 ) << 48 ),    //!< デカール
    _REFRACT_APPLY_ALPHA                    = ( static_cast< uint64_t >( 0x01 ) << 49 ),    //!< 屈折アルファ値を適用する
    _ALPHA_PROCESS_DIST_FILED               = ( static_cast< uint64_t >( 0x01 ) << 50 ),    //!< clamp( ( Texture - Alpha0 ) *4 )* Alpha1
    _ALPHA_COMPARE_NEVER                    = ( static_cast< uint64_t >( 0x01 ) << 51 ),    //!< アルファテスト比較モード(常に通過しない)
    _ALPHA_COMPARE_LESS                     = ( static_cast< uint64_t >( 0x01 ) << 52 ),    //!< アルファテスト比較モード(＜ 基準値)
    _ALPHA_COMPARE_EQUAL                    = ( static_cast< uint64_t >( 0x01 ) << 53 ),    //!< アルファテスト比較モード(＝ 基準値)
    _ALPHA_COMPARE_LEQUAL                   = ( static_cast< uint64_t >( 0x01 ) << 54 ),    //!< アルファテスト比較モード(≦ 基準値)
    _ALPHA_COMPARE_GREATER                  = ( static_cast< uint64_t >( 0x01 ) << 55 ),    //!< アルファテスト比較モード(＞ 基準値)
    _ALPHA_COMPARE_NOTEQUAL                 = ( static_cast< uint64_t >( 0x01 ) << 56 ),    //!< アルファテスト比較モード(≠ 基準値)
    _ALPHA_COMPARE_GEQUAL                   = ( static_cast< uint64_t >( 0x01 ) << 57 ),    //!< アルファテスト比較モード(≧ 基準値)
    _ALPHA_COMPARE_ALWAYS                   = ( static_cast< uint64_t >( 0x01 ) << 58 ),    //!< アルファテスト比較モード(常に通過)
    _TEX0_COLOR_INPUT_SQUARE                = ( static_cast< uint64_t >( 0x01 ) << 59 ),    //!< テクスチャ0 入力カラーを二乗して取得
    _TEX1_COLOR_INPUT_SQUARE                = ( static_cast< uint64_t >( 0x01 ) << 60 ),    //!< テクスチャ1 入力カラーを二乗して取得
    _TEX2_COLOR_INPUT_SQUARE                = ( static_cast< uint64_t >( 0x01 ) << 61 ),    //!< テクスチャ2 入力カラーを二乗して取得
    _REFRACT_BY_CAMERA_DISTANCE             = ( static_cast< uint64_t >( 0x01 ) << 62 ),    //!< カメラ距離に応じてカラーバッファ屈折の強度を強くする
};

//---------------------------------------------------------------------------
//! @brief  ピクセルシェーダキー定義2
//---------------------------------------------------------------------------
enum : uint64_t
{
    _TEX0_COLOR_INPUT_ONE_MINUS_RGB         = ( static_cast< uint64_t >( 0x01 ) <<  0 ),    //!< テクスチャ0 入力カラーを (1.0 - RGB) で置き換え
    _TEX1_COLOR_INPUT_ONE_MINUS_RGB         = ( static_cast< uint64_t >( 0x01 ) <<  1 ),    //!< テクスチャ1 入力カラーを (1.0 - RGB) で置き換え
    _TEX2_COLOR_INPUT_ONE_MINUS_RGB         = ( static_cast< uint64_t >( 0x01 ) <<  2 ),    //!< テクスチャ2 入力カラーを (1.0 - RGB) で置き換え
    _TEX0_COLOR_INPUT_ALPHA                 = ( static_cast< uint64_t >( 0x01 ) <<  3 ),    //!< テクスチャ0 入力カラーを Alpha で置き換え
    _TEX1_COLOR_INPUT_ALPHA                 = ( static_cast< uint64_t >( 0x01 ) <<  4 ),    //!< テクスチャ1 入力カラーを Alpha で置き換え
    _TEX2_COLOR_INPUT_ALPHA                 = ( static_cast< uint64_t >( 0x01 ) <<  5 ),    //!< テクスチャ2 入力カラーを Alpha で置き換え
    _TEX0_COLOR_INPUT_ONE_MINUS_ALPHA       = ( static_cast< uint64_t >( 0x01 ) <<  6 ),    //!< テクスチャ0 入力カラーを (1.0 - Alpha) で置き換え
    _TEX1_COLOR_INPUT_ONE_MINUS_ALPHA       = ( static_cast< uint64_t >( 0x01 ) <<  7 ),    //!< テクスチャ1 入力カラーを (1.0 - Alpha) で置き換え
    _TEX2_COLOR_INPUT_ONE_MINUS_ALPHA       = ( static_cast< uint64_t >( 0x01 ) <<  8 ),    //!< テクスチャ2 入力カラーを (1.0 - Alpha) で置き換え
    _TEX0_ALPHA_INPUT_ONE_MINUS_RED         = ( static_cast< uint64_t >( 0x01 ) <<  9 ),    //!< テクスチャ0 入力アルファを (1.0 - Red) で置き換え
    _TEX1_ALPHA_INPUT_ONE_MINUS_RED         = ( static_cast< uint64_t >( 0x01 ) << 10 ),    //!< テクスチャ1 入力アルファを (1.0 - Red) で置き換え
    _TEX2_ALPHA_INPUT_ONE_MINUS_RED         = ( static_cast< uint64_t >( 0x01 ) << 11 ),    //!< テクスチャ2 入力アルファを (1.0 - Red) で置き換え
    _TEX0_ALPHA_INPUT_ONE_MINUS_ALPHA       = ( static_cast< uint64_t >( 0x01 ) << 12 ),    //!< テクスチャ0 入力アルファを (1.0 - Alpha) で置き換え
    _TEX1_ALPHA_INPUT_ONE_MINUS_ALPHA       = ( static_cast< uint64_t >( 0x01 ) << 13 ),    //!< テクスチャ1 入力アルファを (1.0 - Alpha) で置き換え
    _TEX2_ALPHA_INPUT_ONE_MINUS_ALPHA       = ( static_cast< uint64_t >( 0x01 ) << 14 ),    //!< テクスチャ2 入力アルファを (1.0 - Alpha) で置き換え
    _PRIMITIVE_COLOR_INPUT_ONE_MINUS_RGB    = ( static_cast< uint64_t >( 0x01 ) << 15 ),    //!< プリミティブ入力カラーを (1.0 - RGB) で置き換え
    _PRIMITIVE_COLOR_INPUT_ALPHA            = ( static_cast< uint64_t >( 0x01 ) << 16 ),    //!< プリミティブ入力カラーを Alpha で置き換え
    _PRIMITIVE_COLOR_INPUT_ONE_MINUS_ALPHA  = ( static_cast< uint64_t >( 0x01 ) << 17 ),    //!< プリミティブ入力カラーを (1.0 - Alpha) で置き換え
    _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_RED    = ( static_cast< uint64_t >( 0x01 ) << 18 ),    //!< プリミティブ入力アルファを (1.0 - Red) で置き換え
    _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_ALPHA  = ( static_cast< uint64_t >( 0x01 ) << 19 ),    //!< プリミティブ入力アルファを (1.0 - Alpha) で置き換え
    _TEX0_MIPMAP_LEVEL_LIMIT                = ( static_cast< uint64_t >( 0x01 ) << 20 ),    //!< テクスチャ0 ミップマップレベルを制限
    _TEX1_MIPMAP_LEVEL_LIMIT                = ( static_cast< uint64_t >( 0x01 ) << 21 ),    //!< テクスチャ1 ミップマップレベルを制限
    _TEX2_MIPMAP_LEVEL_LIMIT                = ( static_cast< uint64_t >( 0x01 ) << 22 ),    //!< テクスチャ2 ミップマップレベルを制限
    _TEX_0_SURFACE_ANIM                     = ( static_cast< uint64_t >( 0x01 ) << 23 ),    //!< テクスチャ0 サーフェイスアニメ
    _TEX_1_SURFACE_ANIM                     = ( static_cast< uint64_t >( 0x01 ) << 24 ),    //!< テクスチャ1 サーフェイスアニメ
    _TEX_2_SURFACE_ANIM                     = ( static_cast< uint64_t >( 0x01 ) << 25 ),    //!< テクスチャ2 サーフェイスアニメ
    _TEX_0_SURFACE_ANIM_CROSSFADE           = ( static_cast< uint64_t >( 0x01 ) << 26 ),    //!< テクスチャ0 サーフェイスアニメクロスフェード
    _TEX_1_SURFACE_ANIM_CROSSFADE           = ( static_cast< uint64_t >( 0x01 ) << 27 ),    //!< テクスチャ1 サーフェイスアニメクロスフェード
    _TEX_2_SURFACE_ANIM_CROSSFADE           = ( static_cast< uint64_t >( 0x01 ) << 28 ),    //!< テクスチャ2 サーフェイスアニメクロスフェード
};

//---------------------------------------------------------------------------
//! @brief  EmitterPlugin用 頂点シェーダキー定義
//---------------------------------------------------------------------------
enum : uint64_t
{
    _STRIPE_BILLBOARD                       = ( static_cast< uint64_t >( 0x01 ) << 0  ),    //!< ストライプ形状：ビルボード
    _STRIPE_EMITTER_MATRIX                  = ( static_cast< uint64_t >( 0x01 ) << 1  ),    //!< ストライプ形状：エミッタ行列ストライプ
    _STRIPE_EMITTER_UPDOWN                  = ( static_cast< uint64_t >( 0x01 ) << 2  ),    //!< ストライプ形状：エミッタの上下に伸ばす
    _STRIPE_FOLLOW_EMITTER_MATRIX           = ( static_cast< uint64_t >( 0x01 ) << 3  ),    //!< ストライプ：エミッタに完全追従
    _STRIPE_RIBBON                          = ( static_cast< uint64_t >( 0x01 ) << 4  ),    //!< ストライプ形状：リボン
};

//---------------------------------------------------------------------------
//! @brief  EmitterPlugin用 ピクセルシェーダキー定義
//---------------------------------------------------------------------------
enum : uint64_t
{
    //_EMITTER_PLUGIN_FRAGMENT_DUMMY          = ( static_cast< uint64_t >( 0x01 ) << 0 )    //!< Dummy
};

//---------------------------------------------------------------------------
//  StreamOutを利用するか
//---------------------------------------------------------------------------
bool ShaderKey::IsStreamOut() const NN_NOEXCEPT
{
    return m_VertexShaderFlag.Is( _CALC_TYPE_GPU_SO );
}

//---------------------------------------------------------------------------
//  初期化
//---------------------------------------------------------------------------
void ShaderKey::Initialize( ResEmitter* resEmitter, uint32_t emitterPluginIndex, void* emitterPluginData, uint32_t fieldFlag, ResFieldCustom* resCustomField, const char* userDefine ) NN_NOEXCEPT
{
    Invalidate();

    //------------------------------------------------
    // 頂点シェーダ向けキー
    //------------------------------------------------
    // アトリビュート
    enum _PrimitiveVertexInfoFlags
    {
        _VERTEX_INFO_FLAG_POSITION  = 0x01 << 0,
        _VERTEX_INFO_FLAG_NORMAL    = 0x01 << 1,
        _VERTEX_INFO_FLAG_TANGENT   = 0x01 << 2,
        _VERTEX_INFO_FLAG_COLOR     = 0x01 << 3,
        _VERTEX_INFO_FLAG_UV0       = 0x01 << 4,
        _VERTEX_INFO_FLAG_UV1       = 0x01 << 5,
    };

    if( resEmitter->ptcl.primitiveVertexInfoFlags & _VERTEX_INFO_FLAG_NORMAL )
    {
        m_VertexShaderFlag2.Set( _USE_NORMAL_ATTR );
    }
    if( resEmitter->ptcl.primitiveVertexInfoFlags & _VERTEX_INFO_FLAG_TANGENT )
    {
        m_VertexShaderFlag2.Set( _USE_TANGENT_ATTR );
    }
    if( resEmitter->ptcl.primitiveVertexInfoFlags & _VERTEX_INFO_FLAG_COLOR )
    {
        m_VertexShaderFlag2.Set( _USE_COLOR0_ATTR );
    }
    if( resEmitter->ptcl.primitiveVertexInfoFlags & _VERTEX_INFO_FLAG_UV0 || resEmitter->ptcl.primitiveVertexInfoFlags & _VERTEX_INFO_FLAG_UV1 ||
        resEmitter->ptcl.trimmingPrimitiveIndex != InvalidValueId_PrimitiveIndex )
    {
        m_VertexShaderFlag2.Set( _USE_TEXCOORD_ATTR );
    }

    // パーティクル or プリミティブ
    SetFigureType( ( resEmitter->ptcl.primitiveIndex != InvalidValueId_PrimitiveIndex || resEmitter->ptcl.trimmingPrimitiveIndex != InvalidValueId_PrimitiveIndex ) ? ParticleFigureType_Primitive : ParticleFigureType_Particle );

    // パーティクルタイプ
    SetParticleType( static_cast< detail::ParticleBillboardType >( resEmitter->ptcl.billboardType ) );

    // 追従タイプ
    SetEmitterFollowType( static_cast< detail::ParticleFollowType >( resEmitter->emitter.followType ) );

    // パーティクル回転
    if( resEmitter->ptcl.isRotateX || resEmitter->ptcl.isRotateY || resEmitter->ptcl.isRotateZ )
    {
        m_VertexShaderFlag.Set( _USE_ROTATE );
    }
    SetParticleRotateType( static_cast< detail::ParticleRotationType >( resEmitter->ptcl.rotType ) );

    // 挙動計算タイプ
    SetEmitterCalculationType( static_cast< detail::EmitterCalculationMode >( resEmitter->emitter.calcType ) );

    // 描画パス
    SetDrawPath( resEmitter->emitter.drawPath );

    // パーティクルタイプ
    if( resEmitter->emission.isWorldGravity )
    {
        SetParticleWorldGravity( true );
    }

    // パーティクルカラー計算タイプ
    SetParticleColorCalculationType( static_cast< detail::ParticleColorCalculationMode >( resEmitter->ptclColor.color0Type ),
                              static_cast< detail::ParticleColorCalculationMode >( resEmitter->ptclColor.color1Type ) );

    // パーティクルアルファ計算タイプ
    SetParticleAlphaCalculationType( static_cast< detail::ParticleColorCalculationMode >( resEmitter->ptclColor.alpha0Type ),
                              static_cast< detail::ParticleColorCalculationMode >( resEmitter->ptclColor.alpha1Type ) );

    // テクスチャパタンアニメ 有効/無効
    SetParticleTexturePatternAnim( ( resEmitter->textureAnim0.patternAnimType != TexturePatternAnimationType_None ),
        ( resEmitter->textureAnim1.patternAnimType != TexturePatternAnimationType_None ),
        ( resEmitter->textureAnim2.patternAnimType != TexturePatternAnimationType_None ) );

    // テクスチャ座標アニメ 有効/無効
    SetParticleTextureShiftAnim( ( resEmitter->textureAnim0.isScroll || resEmitter->textureAnim0.isScale ),
        ( resEmitter->textureAnim1.isScroll || resEmitter->textureAnim1.isScale ),
        ( resEmitter->textureAnim2.isScroll || resEmitter->textureAnim2.isScale ) );

    if( resEmitter->textureAnim0.isRotate )
    {
        m_VertexShaderFlag.Set( _TEX_0_SHIFT_ANIM_ROTATE );
    }
    if( resEmitter->textureAnim1.isRotate )
    {
        m_VertexShaderFlag.Set( _TEX_1_SHIFT_ANIM_ROTATE );
    }
    if( resEmitter->textureAnim2.isRotate )
    {
        m_VertexShaderFlag.Set( _TEX_2_SHIFT_ANIM_ROTATE );
    }

    // テクスチャサーフェイスアニメ 有効/無効
    SetParticleTextureSurfaceAnim( ( resEmitter->textureAnim0.patternAnimType == TexturePatternAnimationType_Surface ),
        ( resEmitter->textureAnim1.patternAnimType == TexturePatternAnimationType_Surface ),
        ( resEmitter->textureAnim2.patternAnimType == TexturePatternAnimationType_Surface ) );

    if( resEmitter->textureAnim0.isCrossfade )
    {
        m_PixelShaderFlag2.Set( _TEX_0_SURFACE_ANIM_CROSSFADE );
    }
    if( resEmitter->textureAnim1.isCrossfade )
    {
        m_PixelShaderFlag2.Set( _TEX_1_SURFACE_ANIM_CROSSFADE );
    }
    if( resEmitter->textureAnim2.isCrossfade )
    {
        m_PixelShaderFlag2.Set( _TEX_2_SURFACE_ANIM_CROSSFADE );
    }

    // スフィアマップの 有効/無効
    SetParticleSphereMap( ( resEmitter->textureSampler0.isSphereMap != 0 ),
        ( resEmitter->textureSampler1.isSphereMap != 0 ),
        ( resEmitter->textureSampler2.isSphereMap != 0 ) );

    // スケール キー数を見て変化させる
    //{
    //    SetParticleScaleAnim( resEmitter->staticUbo.scaleAnimKeyNum );
    //}
    m_VertexScaleAnimKey = static_cast< uint8_t >( resEmitter->staticUbo.scaleAnimKeyNum );
    m_VertexColor0AnimKey = ( resEmitter->ptclColor.color0Type == ParticleColorCalculationMode_8KeysAnimation ) ? static_cast< uint8_t >( resEmitter->staticUbo.color0AnimKeynum ) : 0;
    m_VertexColor1AnimKey = ( resEmitter->ptclColor.color1Type == ParticleColorCalculationMode_8KeysAnimation ) ? static_cast< uint8_t >( resEmitter->staticUbo.color1AnimKeynum ) : 0;
    m_VertexAlpha0AnimKey = ( resEmitter->ptclColor.alpha0Type == ParticleColorCalculationMode_8KeysAnimation ) ? static_cast< uint8_t >( resEmitter->staticUbo.alpha0AnimKeyNum ) : 0;
    m_VertexAlpha1AnimKey = ( resEmitter->ptclColor.alpha1Type == ParticleColorCalculationMode_8KeysAnimation ) ? static_cast< uint8_t >( resEmitter->staticUbo.alpha1AnimKeyNum ) : 0;

    // 揺らぎ 有効/無効
    if( resEmitter->ptclFluctuation.isApplayScale )
    {
        m_VertexShaderFlag.Set( _FLUCTUATION_SCALE_ENABLE );
    }
    if( resEmitter->ptclFluctuation.isApplyAlpha )
    {
        m_VertexShaderFlag.Set( _FLUCTUATION_ALPHA_ENABLE );
    }
    if( resEmitter->ptclFluctuation.isApplayScaleY )
    {
        m_VertexShaderFlag.Set( _FLUCTUATION_SCALE_Y_SEPARATE );
    }

    // 近距離アルファ
    if( resEmitter->ptclColor.isNearDistAlpha )
    {
        m_VertexShaderFlag.Set( _NEAR_DIST_ALPHA );
    }

    // 遠距離アルファ
    if( resEmitter->ptclColor.isFarDistAlpha )
    {
        m_VertexShaderFlag.Set( _FAR_DIST_ALPHA );
    }

    // ソフトパーティクル
    if( resEmitter->ptclColor.isSoftParticle == 1 )
    {
        m_VertexShaderFlag.Set( _VERTEX_SOFT );
    }

    // 遮蔽チェック
    if( resEmitter->ptclColor.isSoftParticle == 3 )
    {
        m_VertexShaderFlag.Set( _MASKING_CHECK );
    }

    // 頂点フレネル
    if( resEmitter->ptclColor.isFresnelAlpha == 1 )
    {
        m_VertexShaderFlag.Set( _VERTEX_FRESNEL_ALPHA );
    }

    // フィールドフラグを立てる
    if( fieldFlag & VFX_SHADER_KEY_FIELD_RANDOM )
    {
        m_VertexShaderFlag2.Set( _FIELD_RANDOM );
    }

    // カメラ前でのスケールを抑制
    if( resEmitter->ptclScale.enableScalingByCameraDistNear == 1 )
    {
        m_VertexShaderFlag.Set( _PARTICLE_SCALE_LIMIT_NEAR );
    }
    if( resEmitter->ptclScale.enableScalingByCameraDistFar == 1 )
    {
        m_VertexShaderFlag.Set( _PARTICLE_SCALE_LIMIT_FAR );
    }
    if( resEmitter->ptclScale.enableLinkFovyToScaleValue == 1 )
    {
        m_VertexShaderFlag2.Set( _LINK_FOVY_TO_SCALE_LIMIT );
    }

    // パーティクルオフセット
    if( resEmitter->ptcl.offsetType == ParticleOffsetType_Depth )
    {
        m_VertexShaderFlag.Set( _DEPTH_OFFSET );
    }
    if( resEmitter->ptcl.offsetType == ParticleOffsetType_Camera )
    {
        m_VertexShaderFlag.Set( _CAMERA_OFFSET );
    }
    if( resEmitter->ptcl.offsetType == ParticleOffsetType_CameraFixedsize )
    {
        m_VertexShaderFlag2.Set( _CAMERA_OFFSET_FIXED_SIZE );
    }

    if( resEmitter->ptcl.connectPtclScaleAndZOffset )
    {
        m_VertexShaderFlag2.Set( _CONNECT_PTCL_SCALE_TO_Z_OFFSET );
    }
    if( resEmitter->ptcl.enableAvoidZFighting == 1 )
    {
        m_VertexShaderFlag2.Set( _AVOID_Z_FIGHTING );
    }


    // 速度スケールY
    if( resEmitter->ptclScale.enableAddScaleY )
    {
        m_VertexShaderFlag.Set( _VELOCITY_SCALE_Y );
    }

    // テクスチャ描画サイズを固定
    if( resEmitter->textureSampler0.isDensityFixedU )
    {
        m_VertexShaderFlag2.Set( _TEX0_KEEP_TEXTURE_SIZE_U );
    }
    if( resEmitter->textureSampler1.isDensityFixedU )
    {
        m_VertexShaderFlag2.Set( _TEX1_KEEP_TEXTURE_SIZE_U );
    }
    if( resEmitter->textureSampler2.isDensityFixedU )
    {
        m_VertexShaderFlag2.Set( _TEX2_KEEP_TEXTURE_SIZE_U );
    }
    if( resEmitter->textureSampler0.isDensityFixedV )
    {
        m_VertexShaderFlag2.Set( _TEX0_KEEP_TEXTURE_SIZE_V );
    }
    if( resEmitter->textureSampler1.isDensityFixedV )
    {
        m_VertexShaderFlag2.Set( _TEX1_KEEP_TEXTURE_SIZE_V );
    }
    if( resEmitter->textureSampler2.isDensityFixedV )
    {
        m_VertexShaderFlag2.Set( _TEX2_KEEP_TEXTURE_SIZE_V );
    }

    // シェーダ係数アニメ
    if( ( resEmitter->combiner.shaderType != PixelShaderMode_Normal || resEmitter->shader.eftCombinerShaderIndex != -1 ) &&
        resEmitter->staticUbo.shaderAnimKeyNum > 1 )
    {
        m_VertexShaderFlag.Set( _SHADER_ANIM );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _SHADER_ANIM );
    }

    // カラー0 継承
    if( resEmitter->inherit.color0 )
    {
        m_VertexShaderFlag.Set( _COLOR0_INHERIT );
    }

    // カラー1 継承
    if( resEmitter->inherit.color1 )
    {
        m_VertexShaderFlag.Set( _COLOR1_INHERIT );
    }

    // アルファ0 継承
    if( resEmitter->inherit.alpha0 )
    {
        m_VertexShaderFlag.Set( _ALPHA0_INHERIT );
    }

    // アルファ1 継承
    if( resEmitter->inherit.alpha1 )
    {
        m_VertexShaderFlag.Set( _ALPHA1_INHERIT );
    }

    //------------------------------------------------
    // エミッタプラグイン：現在は型を見てチートしてフラグを立てる。
    //------------------------------------------------
    if( emitterPluginIndex > 0 )
    {
        switch( emitterPluginIndex )
        {
        case 1: // 連結式ストライプ
        {
            ResStripeConnection* data = reinterpret_cast< ResStripeConnection* >( emitterPluginData );
            switch( data->calcType )
            {
            case 0: m_VertexShaderFlagEP.Set( _STRIPE_BILLBOARD );         break;
            case 1: m_VertexShaderFlagEP.Set( _STRIPE_EMITTER_MATRIX );    break;
            case 2: m_VertexShaderFlagEP.Set( _STRIPE_EMITTER_UPDOWN );    break;
            default: break;
            }
            if( data->emitterFollow )
            {
                m_VertexShaderFlagEP.Set( _STRIPE_FOLLOW_EMITTER_MATRIX );
            }
        }
            break;
        case 2: // 履歴式ストライプ
        {
            ResStripeHistory* data = reinterpret_cast< ResStripeHistory* >( emitterPluginData );
            switch( data->calcType )
            {
            case 0: m_VertexShaderFlagEP.Set( _STRIPE_BILLBOARD );         break;
            case 1: m_VertexShaderFlagEP.Set( _STRIPE_EMITTER_MATRIX );    break;
            case 2: m_VertexShaderFlagEP.Set( _STRIPE_EMITTER_UPDOWN );    break;
            default: break;
            }
            if( data->emitterFollow )
            {
                m_VertexShaderFlagEP.Set( _STRIPE_FOLLOW_EMITTER_MATRIX );
            }
        }
            break;
        case 3: // 履歴式ストライプ2（スーパーストライプ）
        {
            ResStripeSuper* data = reinterpret_cast< ResStripeSuper* >( emitterPluginData );
            switch( data->calcType )
            {
            case 0: m_VertexShaderFlagEP.Set( _STRIPE_BILLBOARD );         break;
            case 1: m_VertexShaderFlagEP.Set( _STRIPE_EMITTER_MATRIX );    break;
            case 2: m_VertexShaderFlagEP.Set( _STRIPE_EMITTER_UPDOWN );    break;
            case 3: m_VertexShaderFlagEP.Set( _STRIPE_RIBBON );            break;
            default: break;
            }
            if( data->emitterFollow )
            {
                m_VertexShaderFlagEP.Set( _STRIPE_FOLLOW_EMITTER_MATRIX );
            }
        }
            break;
        case 4: // 範囲内ループ
        {

        }
            break;
        default:
            break;
        }
    }

    // UVチャンネル
    if( resEmitter->textureAnim0.uvChannel == 1 )
    {
        m_VertexShaderFlag.Set( _TEXTURE0_USE_UV1 );
    }
    if( resEmitter->textureAnim1.uvChannel == 1 )
    {
        m_VertexShaderFlag.Set( _TEXTURE1_USE_UV1 );
    }
    if( resEmitter->textureAnim2.uvChannel == 1 )
    {
        m_VertexShaderFlag.Set( _TEXTURE2_USE_UV1 );
    }

    //------------------------------------------------
    // ピクセルシェーダ向けキー
    //------------------------------------------------

    // シェーダタイプ
    // コンバイナシェーダがセットされてない場合のみ、シェーダタイプのフラグをセットする
    if( resEmitter->shader.eftCombinerShaderIndex == static_cast< uint32_t >( -1 ) )
    {
        switch( resEmitter->combiner.shaderType )
        {
        case PixelShaderMode_Normal:
            {
                m_PixelShaderFlag.Set( _SHADER_TYPE_NORMAL );
                break;
            }
        case PixelShaderMode_Refraction:
            {
                m_PixelShaderFlag.Set( _SHADER_TYPE_REFRACT );
                if( resEmitter->combiner.applyAlpha )
                {
                    m_PixelShaderFlag.Set( _REFRACT_APPLY_ALPHA );
                }
                if( resEmitter->combiner.isDistortionByCameraDistance )
                {
                    m_PixelShaderFlag.Set( _REFRACT_BY_CAMERA_DISTANCE );
                }
                break;
            }
        case PixelShaderMode_Distortion:
            {
                m_PixelShaderFlag.Set( _SHADER_TYPE_DISTORTION );
                break;
            }
        default:
            break;
        }
    }

    // カラー計算式
    switch( resEmitter->combiner.colorCombinerProcess )
    {
    case CombinerColorProcessType_Color:                m_PixelShaderFlag.Set( _COLOR_PROCESS_COLOR );                  break;
    case CombinerColorProcessType_Texture:              m_PixelShaderFlag.Set( _COLOR_PROCESS_TEXTURE );                break;
    case CombinerColorProcessType_TextureInterpolate:   m_PixelShaderFlag.Set( _COLOR_PROCESS_TEXTURE_INTERPOLATE );    break;
    case CombinerColorProcessType_TextureAdd:           m_PixelShaderFlag.Set( _COLOR_PROCESS_TEXTURE_ADD );            break;
    default: break;
    }

    // アルファ計算式
    switch( resEmitter->combiner.alphaCombinerProcess )
    {
    case CombinerAlphaProcessType_MulAlpha0:            m_PixelShaderFlag.Set( _ALPHA_PROCESS_MOD );            break;
    case CombinerAlphaProcessType_MulAlpha01:           m_PixelShaderFlag.Set( _ALPHA_PROCESS_MOD_A0_A1 );      break;
    case CombinerAlphaProcessType_SubAlpha0:            m_PixelShaderFlag.Set( _ALPHA_PROCESS_SUB );            break;
    case CombinerAlphaProcessType_SubAlpha0MulAlpha1:   m_PixelShaderFlag.Set( _ALPHA_PROCESS_SUB_A0_MOD_A1 );  break;
    case CombinerAlphaProcessType_DistanceField:        m_PixelShaderFlag.Set( _ALPHA_PROCESS_DIST_FILED );     break;
    default: break;
    }

    // テクスチャブレンドタイプ
    switch( resEmitter->combiner.texture1ColorBlend )
    {
    case CombinerBlendType_Mul:      m_PixelShaderFlag.Set( _TEXTURE1_COLOR_BLEND_MOD );  break;
    case CombinerBlendType_Add:      m_PixelShaderFlag.Set( _TEXTURE1_COLOR_BLEND_ADD );  break;
    case CombinerBlendType_Sub:      m_PixelShaderFlag.Set( _TEXTURE1_COLOR_BLEND_SUB );  break;
    default: break;
    }
    switch( resEmitter->combiner.texture2ColorBlend )
    {
    case CombinerBlendType_Mul:      m_PixelShaderFlag.Set( _TEXTURE2_COLOR_BLEND_MOD );  break;
    case CombinerBlendType_Add:      m_PixelShaderFlag.Set( _TEXTURE2_COLOR_BLEND_ADD );  break;
    case CombinerBlendType_Sub:      m_PixelShaderFlag.Set( _TEXTURE2_COLOR_BLEND_SUB );  break;
    default: break;
    }
    switch( resEmitter->combiner.texture1AlphaBlend )
    {
    case CombinerBlendType_Mul:      m_PixelShaderFlag.Set( _TEXTURE1_ALPHA_BLEND_MOD );  break;
    case CombinerBlendType_Add:      m_PixelShaderFlag.Set( _TEXTURE1_ALPHA_BLEND_ADD );  break;
    case CombinerBlendType_Sub:      m_PixelShaderFlag.Set( _TEXTURE1_ALPHA_BLEND_SUB );  break;
    default: break;
    }
    switch( resEmitter->combiner.texture2AlphaBlend )
    {
    case CombinerBlendType_Mul:      m_PixelShaderFlag.Set( _TEXTURE2_ALPHA_BLEND_MOD );  break;
    case CombinerBlendType_Add:      m_PixelShaderFlag.Set( _TEXTURE2_ALPHA_BLEND_ADD );  break;
    case CombinerBlendType_Sub:      m_PixelShaderFlag.Set( _TEXTURE2_ALPHA_BLEND_SUB );  break;
    default: break;
    }

    // テクスチャカラー入力タイプ
    switch( resEmitter->combiner.texColor0InputType )
    {
    case CombinerColorInputType_One:           m_PixelShaderFlag.Set( _TEX0_COLOR_INPUT_ONE );              break;
    case CombinerColorInputType_OneMinusRGB:   m_PixelShaderFlag2.Set( _TEX0_COLOR_INPUT_ONE_MINUS_RGB );   break;
    case CombinerColorInputType_Alpha:         m_PixelShaderFlag2.Set( _TEX0_COLOR_INPUT_ALPHA );           break;
    case CombinerColorInputType_OneMinusAlpha: m_PixelShaderFlag2.Set( _TEX0_COLOR_INPUT_ONE_MINUS_ALPHA ); break;
    default: break;
    }
    switch( resEmitter->combiner.texColor1InputType )
    {
    case CombinerColorInputType_One:           m_PixelShaderFlag.Set( _TEX1_COLOR_INPUT_ONE );              break;
    case CombinerColorInputType_OneMinusRGB:   m_PixelShaderFlag2.Set( _TEX1_COLOR_INPUT_ONE_MINUS_RGB );   break;
    case CombinerColorInputType_Alpha:         m_PixelShaderFlag2.Set( _TEX1_COLOR_INPUT_ALPHA );           break;
    case CombinerColorInputType_OneMinusAlpha: m_PixelShaderFlag2.Set( _TEX1_COLOR_INPUT_ONE_MINUS_ALPHA ); break;
    default: break;
    }
    switch( resEmitter->combiner.texColor2InputType )
    {
    case CombinerColorInputType_One:           m_PixelShaderFlag.Set( _TEX2_COLOR_INPUT_ONE );              break;
    case CombinerColorInputType_OneMinusRGB:   m_PixelShaderFlag2.Set( _TEX2_COLOR_INPUT_ONE_MINUS_RGB );   break;
    case CombinerColorInputType_Alpha:         m_PixelShaderFlag2.Set( _TEX2_COLOR_INPUT_ALPHA );           break;
    case CombinerColorInputType_OneMinusAlpha: m_PixelShaderFlag2.Set( _TEX2_COLOR_INPUT_ONE_MINUS_ALPHA ); break;
    default: break;
    }

    // テクスチャアルファ入力タイプ
    switch( resEmitter->combiner.texAlpha0InputType )
    {
    case CombinerAlphaInputType_One:             m_PixelShaderFlag.Set( _TEX0_ALPHA_INPUT_ONE );              break;
    case CombinerAlphaInputType_Red:             m_PixelShaderFlag.Set( _TEX0_ALPHA_INPUT_RED );              break;
    case CombinerAlphaInputType_OneMinusAlpha:   m_PixelShaderFlag2.Set( _TEX0_ALPHA_INPUT_ONE_MINUS_ALPHA ); break;
    case CombinerAlphaInputType_OneMinusRed:     m_PixelShaderFlag2.Set( _TEX0_ALPHA_INPUT_ONE_MINUS_RED );   break;
    default: break;
    }
    switch( resEmitter->combiner.texAlpha1InputType )
    {
    case CombinerAlphaInputType_One:             m_PixelShaderFlag.Set( _TEX1_ALPHA_INPUT_ONE );              break;
    case CombinerAlphaInputType_Red:             m_PixelShaderFlag.Set( _TEX1_ALPHA_INPUT_RED );              break;
    case CombinerAlphaInputType_OneMinusAlpha:   m_PixelShaderFlag2.Set( _TEX1_ALPHA_INPUT_ONE_MINUS_ALPHA ); break;
    case CombinerAlphaInputType_OneMinusRed:     m_PixelShaderFlag2.Set( _TEX1_ALPHA_INPUT_ONE_MINUS_RED );   break;
    default: break;
    }
    switch( resEmitter->combiner.texAlpha2InputType )
    {
    case CombinerAlphaInputType_One:             m_PixelShaderFlag.Set( _TEX2_ALPHA_INPUT_ONE );              break;
    case CombinerAlphaInputType_Red:             m_PixelShaderFlag.Set( _TEX2_ALPHA_INPUT_RED );              break;
    case CombinerAlphaInputType_OneMinusAlpha:   m_PixelShaderFlag2.Set( _TEX2_ALPHA_INPUT_ONE_MINUS_ALPHA ); break;
    case CombinerAlphaInputType_OneMinusRed:     m_PixelShaderFlag2.Set( _TEX2_ALPHA_INPUT_ONE_MINUS_RED );   break;
    default: break;
    }

    // テクスチャカラー二乗オプション
    if( resEmitter->textureSampler0.isSquareRgb )
    {
        m_PixelShaderFlag.Set( _TEX0_COLOR_INPUT_SQUARE );
    }
    if( resEmitter->textureSampler1.isSquareRgb )
    {
        m_PixelShaderFlag.Set( _TEX1_COLOR_INPUT_SQUARE );
    }
    if( resEmitter->textureSampler2.isSquareRgb )
    {
        m_PixelShaderFlag.Set( _TEX2_COLOR_INPUT_SQUARE );
    }

    // プリミティブカラーブレンドタイプ
    switch( resEmitter->combiner.primitiveColorBlend )
    {
    case CombinerBlendType_Mul: m_PixelShaderFlag.Set( _PRIMITIVE_COLOR_BLEND_MOD ); break;
    case CombinerBlendType_Add: m_PixelShaderFlag.Set( _PRIMITIVE_COLOR_BLEND_ADD ); break;
    case CombinerBlendType_Sub: m_PixelShaderFlag.Set( _PRIMITIVE_COLOR_BLEND_SUB ); break;
    default: break;
    }
    switch( resEmitter->combiner.primitiveAlphaBlend )
    {
    case CombinerBlendType_Mul: m_PixelShaderFlag.Set( _PRIMITIVE_ALPHA_BLEND_MOD ); break;
    case CombinerBlendType_Add: m_PixelShaderFlag.Set( _PRIMITIVE_ALPHA_BLEND_ADD ); break;
    case CombinerBlendType_Sub: m_PixelShaderFlag.Set( _PRIMITIVE_ALPHA_BLEND_SUB ); break;
    default: break;
    }

    // プリミティブカラー入力タイプ
    switch( resEmitter->combiner.primitiveColorInputType )
    {
    case CombinerColorInputType_Source:        m_PixelShaderFlag.Set( _PRIMITIVE_COLOR_INPUT_SRC );              break;
    case CombinerColorInputType_One:           m_PixelShaderFlag.Set( _PRIMITIVE_COLOR_INPUT_ONE );              break;
    case CombinerColorInputType_OneMinusRGB:   m_PixelShaderFlag2.Set( _PRIMITIVE_COLOR_INPUT_ONE_MINUS_RGB );   break;
    case CombinerColorInputType_Alpha:         m_PixelShaderFlag2.Set( _PRIMITIVE_COLOR_INPUT_ALPHA );           break;
    case CombinerColorInputType_OneMinusAlpha: m_PixelShaderFlag2.Set( _PRIMITIVE_COLOR_INPUT_ONE_MINUS_ALPHA ); break;
    default: break;
    }

    // プリミティブアルファ入力タイプ
    switch( resEmitter->combiner.primitiveAlphaInputType )
    {
    case CombinerAlphaInputType_Source:          m_PixelShaderFlag.Set( _PRIMITIVE_ALPHA_INPUT_A );                break;
    case CombinerAlphaInputType_One:             m_PixelShaderFlag.Set( _PRIMITIVE_ALPHA_INPUT_ONE );              break;
    case CombinerAlphaInputType_Red:             m_PixelShaderFlag.Set( _PRIMITIVE_ALPHA_INPUT_RED );              break;
    case CombinerAlphaInputType_OneMinusAlpha:   m_PixelShaderFlag2.Set( _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_ALPHA ); break;
    case CombinerAlphaInputType_OneMinusRed:     m_PixelShaderFlag2.Set( _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_RED );   break;
    default: break;
    }

    // テクスチャの有効無効
    if( resEmitter->textureSampler0.guid != InvalidValueId_TextureId )
    {
        m_PixelShaderFlag.Set( _TEXTURE0_ENABLE );
    }
    if( resEmitter->textureSampler1.guid != InvalidValueId_TextureId )
    {
        m_PixelShaderFlag.Set( _TEXTURE1_ENABLE );
    }
    if( resEmitter->textureSampler2.guid != InvalidValueId_TextureId )
    {
        m_PixelShaderFlag.Set( _TEXTURE2_ENABLE );
    }

    // テクスチャのミップマップレベルを制限の有無
    if ( resEmitter->textureSampler0.mipLevelLimit )
    {
        m_PixelShaderFlag2.Set( _TEX0_MIPMAP_LEVEL_LIMIT );
    }

    if ( resEmitter->textureSampler1.mipLevelLimit )
    {
        m_PixelShaderFlag2.Set( _TEX1_MIPMAP_LEVEL_LIMIT );
    }

    if ( resEmitter->textureSampler2.mipLevelLimit )
    {
        m_PixelShaderFlag2.Set( _TEX2_MIPMAP_LEVEL_LIMIT );
    }

    if( resEmitter->ptclColor.isSoftParticle == 2 )
    {
        m_PixelShaderFlag.Set( _FRAGMENT_SOFT );
    }
    if( resEmitter->ptclColor.isDecal )
    {
        m_PixelShaderFlag.Set( _DECAL );
    }

    // 頂点フレネル
    if( resEmitter->ptclColor.isFresnelAlpha == 2 )
    {
        m_PixelShaderFlag.Set( _FRAGMENT_FRESNEL_ALPHA );
    }

    // アルファ比較モード
    if ( resEmitter->renderState.isAlphaTest )
    {
        switch (resEmitter->renderState.alphaFunc)
        {
        case TestCompareFunction_Never:             m_PixelShaderFlag.Set( _ALPHA_COMPARE_NEVER );      break;
        case TestCompareFunction_Less:              m_PixelShaderFlag.Set( _ALPHA_COMPARE_LESS );       break;
        case TestCompareFunction_Equal:             m_PixelShaderFlag.Set( _ALPHA_COMPARE_EQUAL );      break;
        case TestCompareFunction_LessOrEqual:       m_PixelShaderFlag.Set( _ALPHA_COMPARE_LEQUAL );     break;
        case TestCompareFunction_GreaterThan:       m_PixelShaderFlag.Set( _ALPHA_COMPARE_GREATER );    break;
        case TestCompareFunction_NotEqual:          m_PixelShaderFlag.Set( _ALPHA_COMPARE_NOTEQUAL );   break;
        case TestCompareFunction_GreaterOrEqual:    m_PixelShaderFlag.Set( _ALPHA_COMPARE_GEQUAL );     break;
        case TestCompareFunction_Always:            m_PixelShaderFlag.Set( _ALPHA_COMPARE_ALWAYS );     break;
        default:
            NN_SDK_ASSERT( 0 );
            break;
        }
    }

    //------------------------------------------------
    // カスタムシェーダ向けキー
    //------------------------------------------------
    m_CustomShaderIndex = resEmitter->shader.customShaderIndex;
    m_CustomShaderFlag = resEmitter->shader.customShaderFlag;
    m_CustomShaderSwitch = resEmitter->shader.customShaderSwitch;

    //------------------------------------------------
    // カスタムフィールド向けキー
    //------------------------------------------------
    if( resCustomField )
    {
#ifdef NEED_EP_FLIP
        resCustomField->FlipEndian();
#endif
        m_CustomFieldEnabled = 1;
        m_CustomFieldFlag = resCustomField->flags;
#ifdef NEED_EP_FLIP
        resCustomField->FlipEndian();
#endif
    }
    else
    {
        m_CustomFieldEnabled = 0;
        m_CustomFieldFlag = 0;
    }

    if( userDefine )
    {
        strcpy_s( m_CustomShaderDefine, 16, userDefine );
    }
    else
    {
        memset( m_CustomShaderDefine, 0, 16 );
    }

    //------------------------------------------------
    // コンバイナシェーダ向けキー
    //------------------------------------------------
    m_CombinerShaderIndex = resEmitter->shader.eftCombinerShaderIndex;

}// NOLINT(readability/fn_size)


//---------------------------------------------------------------------------
//  コンパイルセッティングを生成します。
//---------------------------------------------------------------------------
void ShaderKey::Generate() NN_NOEXCEPT
{
    _ShaderKeyString keyString;
    keyString.Initialize( g_CompileSetting, ShaderCompileSettings_BufferSize );
    //    OutputLog( mCompileSetting );

#ifdef VFX_USE_INSTANS_DRAW
    keyString.AddKey( "#define VFX_USE_INSTANS_DRAW\n" );
#endif
#ifdef VFX_USE_REGISTER_DRAW
    keyString.AddKey( "#define VFX_USE_REGISTER_DRAW\n" );
#endif

    //----------------------------------
    // 頂点シェーダ向けキー
    //----------------------------------
    if ( m_VertexShaderFlag2.Is( _USE_NORMAL_ATTR ) )
    {
        keyString.AddKey( "#define _USE_NORMAL_ATTR\n" );
    }
    if ( m_VertexShaderFlag2.Is( _USE_TANGENT_ATTR ) )
    {
        keyString.AddKey( "#define _USE_TANGENT_ATTR\n" );
    }
    if ( m_VertexShaderFlag2.Is( _USE_COLOR0_ATTR ) )
    {
        keyString.AddKey( "#define _USE_COLOR0_ATTR\n" );
    }
    if ( m_VertexShaderFlag2.Is( _USE_TEXCOORD_ATTR ) )
    {
        keyString.AddKey( "#define _USE_TEXCOORD_ATTR\n" );
    }

    if( m_VertexShaderFlag.Is( _CALC_TYPE_CPU ) )
    {
        keyString.AddKey( "#define _CALC_TYPE_CPU\n" );
    }
    if( m_VertexShaderFlag.Is( _CALC_TYPE_GPU_TIME ) )
    {
        keyString.AddKey( "#define _CALC_TYPE_GPU_TIME\n" );
    }
    if( m_VertexShaderFlag.Is( _CALC_TYPE_GPU_SO ) )
    {
        keyString.AddKey( "#define _CALC_TYPE_GPU_SO\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_TYPE_BILLBOARD ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_BILLBOARD\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_TYPE_COMPLEX_BILLBOARD ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_COMPLEX_BILLBOARD\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_TYPE_Y_BILLBOARD ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_Y_BILLBOARD\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_TYPE_WITH_SCALE_Z ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_WITH_SCALE_Z\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_TYPE_POLYGON_XY ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_POLYGON_XY\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_TYPE_POLYGON_XZ ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_POLYGON_XZ\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_TYPE_VEL_LOOK ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_VEL_LOOK\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_TYPE_VEL_LOOK_POLYGON ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_VEL_LOOK_POLYGON\n" );
    }
    if( m_VertexShaderFlag.Is( _EMITTER_FOLLOW_TYPE_ALL ) )
    {
        keyString.AddKey( "#define _EMITTER_FOLLOW_TYPE_ALL\n" );
    }
    if( m_VertexShaderFlag.Is( _EMITTER_FOLLOW_TYPE_NONE ) )
    {
        keyString.AddKey( "#define _EMITTER_FOLLOW_TYPE_NONE\n" );
    }
    if( m_VertexShaderFlag.Is( _EMITTER_FOLLOW_TYPE_POS ) )
    {
        keyString.AddKey( "#define _EMITTER_FOLLOW_TYPE_POS\n" );
    }
    if( m_VertexShaderFlag.Is( _FLUCTUATION_SCALE_ENABLE ) )
    {
        keyString.AddKey( "#define _FLUCTUATION_SCALE_ENABLE\n" );
    }
    if( m_VertexShaderFlag.Is( _FLUCTUATION_ALPHA_ENABLE ) )
    {
        keyString.AddKey( "#define _FLUCTUATION_ALPHA_ENABLE\n" );
    }
    if( m_VertexShaderFlag.Is( _FLUCTUATION_SCALE_Y_SEPARATE ) )
    {
        keyString.AddKey( "#define _FLUCTUATION_SCALE_Y_SEPARATE\n" );
    }
    if( m_VertexShaderFlag.Is( _COLOR_0_FIXED ) )
    {
        keyString.AddKey( "#define _COLOR_0_FIXED\n" );
    }
    if( m_VertexShaderFlag.Is( _COLOR_0_RANDOM ) )
    {
        keyString.AddKey( "#define _COLOR_0_RANDOM\n" );
    }
    if( m_VertexShaderFlag.Is( _COLOR_1_FIXED ) )
    {
        keyString.AddKey( "#define _COLOR_1_FIXED\n" );
    }
    if( m_VertexShaderFlag.Is( _COLOR_1_RANDOM ) )
    {
        keyString.AddKey( "#define _COLOR_1_RANDOM\n" );
    }
    if( m_VertexShaderFlag.Is( _ALPHA_0_FIXED ) )
    {
        keyString.AddKey( "#define _ALPHA_0_FIXED\n" );
    }
    if( m_VertexShaderFlag.Is( _ALPHA_1_FIXED ) )
    {
        keyString.AddKey( "#define _ALPHA_1_FIXED\n" );
    }

    if( m_VertexShaderFlag.Is( _TEX_0_SHIFT_ANIM_ROTATE ) )
    {
        keyString.AddKey( "#define _TEX_0_SHIFT_ANIM_ROTATE\n" );
    }
    else if( m_VertexShaderFlag.Is( _TEX_0_SHIFT_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_0_SHIFT_ANIM\n" );
    }

    if( m_VertexShaderFlag.Is( _TEX_1_SHIFT_ANIM_ROTATE ) )
    {
        keyString.AddKey( "#define _TEX_1_SHIFT_ANIM_ROTATE\n" );
    }
    else if( m_VertexShaderFlag.Is( _TEX_1_SHIFT_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_1_SHIFT_ANIM\n" );
    }

    if( m_VertexShaderFlag.Is( _TEX_2_SHIFT_ANIM_ROTATE ) )
    {
        keyString.AddKey( "#define _TEX_2_SHIFT_ANIM_ROTATE\n" );
    }
    else if( m_VertexShaderFlag.Is( _TEX_2_SHIFT_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_2_SHIFT_ANIM\n" );
    }

    if( m_PixelShaderFlag2.Is( _TEX_0_SURFACE_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_0_SURFACE_ANIM\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX_0_SURFACE_ANIM_CROSSFADE ) )
    {
        keyString.AddKey( "#define _TEX_0_SURFACE_ANIM_CROSSFADE\n" );
    }

    if( m_PixelShaderFlag2.Is( _TEX_1_SURFACE_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_1_SURFACE_ANIM\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX_1_SURFACE_ANIM_CROSSFADE ) )
    {
        keyString.AddKey( "#define _TEX_1_SURFACE_ANIM_CROSSFADE\n" );
    }

    if( m_PixelShaderFlag2.Is( _TEX_2_SURFACE_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_2_SURFACE_ANIM\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX_2_SURFACE_ANIM_CROSSFADE ) )
    {
        keyString.AddKey( "#define _TEX_2_SURFACE_ANIM_CROSSFADE\n" );
    }

    if( m_VertexShaderFlag.Is( _TEX_0_PATTERN_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_0_PATTERN_ANIM\n" );
    }
    if( m_VertexShaderFlag.Is( _TEX_1_PATTERN_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_1_PATTERN_ANIM\n" );
    }
    if( m_VertexShaderFlag.Is( _TEX_2_PATTERN_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_2_PATTERN_ANIM\n" );
    }
    if( m_VertexShaderFlag.Is( _TEX_0_SPHERE_MAP ) )
    {
        keyString.AddKey( "#define _TEX_0_SPHERE_MAP\n" );
    }
    if( m_VertexShaderFlag.Is( _TEX_1_SPHERE_MAP ) )
    {
        keyString.AddKey( "#define _TEX_1_SPHERE_MAP\n" );
    }
    if( m_VertexShaderFlag.Is( _TEX_2_SPHERE_MAP ) )
    {
        keyString.AddKey( "#define _TEX_2_SPHERE_MAP\n" );
    }

    if( m_VertexShaderFlag.Is( _USE_ROTATE ) )
    {
        keyString.AddKey( "#define _USE_ROTATE\n" );
    }
    if( m_VertexShaderFlag.Is( _ROTATE_YZX ) )
    {
        keyString.AddKey( "#define _ROTATE_YZX\n" );
    }
    if( m_VertexShaderFlag.Is( _ROTATE_XYZ ) )
    {
        keyString.AddKey( "#define _ROTATE_XYZ\n" );
    }
    if( m_VertexShaderFlag.Is( _ROTATE_ZXY ) )
    {
        keyString.AddKey( "#define _ROTATE_ZXY\n" );
    }
    if( m_VertexShaderFlag.Is( _NEAR_DIST_ALPHA ) )
    {
        keyString.AddKey( "#define _NEAR_DIST_ALPHA\n" );
    }
    if( m_VertexShaderFlag.Is( _FAR_DIST_ALPHA ) )
    {
        keyString.AddKey( "#define _FAR_DIST_ALPHA\n" );
    }
    if( m_VertexShaderFlag.Is( _MASKING_CHECK ) )
    {
        keyString.AddKey( "#define _MASKING_CHECK\n" );
    }
    if( m_VertexShaderFlag.Is( _VERTEX_SOFT ) )
    {
        keyString.AddKey( "#define _VERTEX_SOFT\n" );
    }
    if( m_VertexShaderFlag.Is( _VERTEX_FRESNEL_ALPHA ) )
    {
        keyString.AddKey( "#define _VERTEX_FRESNEL_ALPHA\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE ) )
    {
        keyString.AddKey( "#define _PARTICLE\n" );
    }
    if( m_VertexShaderFlag.Is( _PRIMITIVE ) )
    {
        keyString.AddKey( "#define _PRIMITIVE\n" );
    }
    if( m_VertexShaderFlag2.Is( _LINK_FOVY_TO_SCALE_LIMIT ) )
    {
        keyString.AddKey( "#define _LINK_FOVY_TO_SCALE_LIMIT\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_SCALE_LIMIT_NEAR ) )
    {
        keyString.AddKey( "#define _PARTICLE_SCALE_LIMIT_NEAR\n" );
    }
    if( m_VertexShaderFlag.Is( _PARTICLE_SCALE_LIMIT_FAR ) )
    {
        keyString.AddKey( "#define _PARTICLE_SCALE_LIMIT_FAR\n" );
    }
    if( m_VertexShaderFlag2.Is( _AVOID_Z_FIGHTING ) )
    {
        keyString.AddKey( "#define _AVOID_Z_FIGHTING\n" );
    }
    if( m_VertexShaderFlag.Is( _SHADER_ANIM ) )
    {
        keyString.AddKey( "#define _SHADER_ANIM\n" );
    }

    if( m_VertexShaderFlag.Is( _COLOR0_INHERIT ) )
    {
        keyString.AddKey( "#define _COLOR0_INHERIT\n" );
    }
    if( m_VertexShaderFlag.Is( _COLOR1_INHERIT ) )
    {
        keyString.AddKey( "#define _COLOR1_INHERIT\n" );
    }
    if( m_VertexShaderFlag.Is( _ALPHA0_INHERIT ) )
    {
        keyString.AddKey( "#define _ALPHA0_INHERIT\n" );
    }
    if( m_VertexShaderFlag.Is( _ALPHA1_INHERIT ) )
    {
        keyString.AddKey( "#define _ALPHA1_INHERIT\n" );
    }

    if( m_VertexShaderFlag2.Is( _CONNECT_PTCL_SCALE_TO_Z_OFFSET ) )
    {
        keyString.AddKey( "#define _CONNECT_PTCL_SCALE_TO_Z_OFFSET\n" );
    }

    if( m_VertexShaderFlag.Is( _VELOCITY_SCALE_Y ) )
    {
        keyString.AddKey( "#define _VELOCITY_SCALE_Y\n" );
    }
    if( m_VertexShaderFlag.Is( _DEPTH_OFFSET ) )
    {
        keyString.AddKey( "#define _DEPTH_OFFSET\n" );
    }
    if( m_VertexShaderFlag.Is( _CAMERA_OFFSET ) )
    {
        keyString.AddKey( "#define _CAMERA_OFFSET\n" );
    }
    if( m_VertexShaderFlag2.Is( _CAMERA_OFFSET_FIXED_SIZE ) )
    {
        keyString.AddKey( "#define _CAMERA_OFFSET_FIXED_SIZE\n" );
    }
    if( m_VertexShaderFlag.Is( _WORLD_GRAVITY ) )
    {
        keyString.AddKey( "#define _WORLD_GRAVITY\n" );
    }

    if( m_VertexShaderFlag2.Is( _FIELD_RANDOM ) )
    {
        keyString.AddKey( "#define _FIELD_RANDOM\n" );
    }

    if( m_VertexShaderFlag2.Is( _TEX0_KEEP_TEXTURE_SIZE_U ) )
    {
        keyString.AddKey( "#define _TEX0_KEEP_TEXTURE_SIZE_U\n" );
    }
    if( m_VertexShaderFlag2.Is( _TEX1_KEEP_TEXTURE_SIZE_U ) )
    {
        keyString.AddKey( "#define _TEX1_KEEP_TEXTURE_SIZE_U\n" );
    }
    if( m_VertexShaderFlag2.Is( _TEX2_KEEP_TEXTURE_SIZE_U ) )
    {
        keyString.AddKey( "#define _TEX2_KEEP_TEXTURE_SIZE_U\n" );
    }

    if( m_VertexShaderFlag2.Is( _TEX0_KEEP_TEXTURE_SIZE_V ) )
    {
        keyString.AddKey( "#define _TEX0_KEEP_TEXTURE_SIZE_V\n" );
    }
    if( m_VertexShaderFlag2.Is( _TEX1_KEEP_TEXTURE_SIZE_V ) )
    {
        keyString.AddKey( "#define _TEX1_KEEP_TEXTURE_SIZE_V\n" );
    }
    if( m_VertexShaderFlag2.Is( _TEX2_KEEP_TEXTURE_SIZE_V ) )
    {
        keyString.AddKey( "#define _TEX2_KEEP_TEXTURE_SIZE_V\n" );
    }

    // TODO:エミッタプラグイン対応
    // ストライプタイプ
    if( m_VertexShaderFlagEP.Is( _STRIPE_BILLBOARD ) )
    {
        keyString.AddKey( "#define _STRIPE_BILLBOARD\n" );
    }
    if( m_VertexShaderFlagEP.Is( _STRIPE_EMITTER_MATRIX ) )
    {
        keyString.AddKey( "#define _STRIPE_EMITTER_MATRIX\n" );
    }
    if( m_VertexShaderFlagEP.Is( _STRIPE_EMITTER_UPDOWN ) )
    {
        keyString.AddKey( "#define _STRIPE_EMITTER_UPDOWN\n" );
    }
    if( m_VertexShaderFlagEP.Is( _STRIPE_FOLLOW_EMITTER_MATRIX ) )
    {
        keyString.AddKey( "#define _STRIPE_FOLLOW_EMITTER_MATRIX\n" );
    }
    if( m_VertexShaderFlagEP.Is( _STRIPE_RIBBON ) )
    {
        keyString.AddKey( "#define _STRIPE_RIBBON\n" );
    }

    //----------------------------------
    // ピクセルシェーダ向けキー
    //----------------------------------
    if( m_PixelShaderFlag.Is( _SHADER_TYPE_NORMAL ) )
    {
        keyString.AddKey( "#define _SHADER_TYPE_NORMAL\n" );
    }
    if( m_PixelShaderFlag.Is( _SHADER_TYPE_REFRACT ) )
    {
        keyString.AddKey( "#define _SHADER_TYPE_REFRACT\n" );
    }
    if( m_PixelShaderFlag.Is( _SHADER_TYPE_DISTORTION ) )
    {
        keyString.AddKey( "#define _SHADER_TYPE_DISTORTION\n" );
    }
    if( m_PixelShaderFlag.Is( _COLOR_PROCESS_COLOR ) )
    {
        keyString.AddKey( "#define _COLOR_PROCESS_COLOR\n" );
    }
    if( m_PixelShaderFlag.Is( _COLOR_PROCESS_TEXTURE ) )
    {
        keyString.AddKey( "#define _COLOR_PROCESS_TEXTURE\n" );
    }
    if( m_PixelShaderFlag.Is( _COLOR_PROCESS_TEXTURE_INTERPOLATE ) )
    {
        keyString.AddKey( "#define _COLOR_PROCESS_TEXTURE_INTERPOLATE\n" );
    }
    if( m_PixelShaderFlag.Is( _COLOR_PROCESS_TEXTURE_ADD ) )
    {
        keyString.AddKey( "#define _COLOR_PROCESS_TEXTURE_ADD\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_PROCESS_MOD ) )
    {
        keyString.AddKey( "#define _ALPHA_PROCESS_MOD\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_PROCESS_SUB ) )
    {
        keyString.AddKey( "#define _ALPHA_PROCESS_SUB\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_PROCESS_MOD_A0_A1 ) )
    {
        keyString.AddKey( "#define _ALPHA_PROCESS_MOD_A0_A1\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_PROCESS_SUB_A0_MOD_A1 ) )
    {
        keyString.AddKey( "#define _ALPHA_PROCESS_SUB_A0_MOD_A1\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_PROCESS_DIST_FILED ) )
    {
        keyString.AddKey( "#define _ALPHA_PROCESS_DIST_FILED\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX0_COLOR_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX0_COLOR_INPUT_ONE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX1_COLOR_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX1_COLOR_INPUT_ONE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX2_COLOR_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX2_COLOR_INPUT_ONE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX0_ALPHA_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX0_ALPHA_INPUT_ONE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX1_ALPHA_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX1_ALPHA_INPUT_ONE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX2_ALPHA_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX2_ALPHA_INPUT_ONE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX0_ALPHA_INPUT_RED ) )
    {
        keyString.AddKey( "#define _TEX0_ALPHA_INPUT_RED\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX1_ALPHA_INPUT_RED ) )
    {
        keyString.AddKey( "#define _TEX1_ALPHA_INPUT_RED\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX2_ALPHA_INPUT_RED ) )
    {
        keyString.AddKey( "#define _TEX2_ALPHA_INPUT_RED\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE0_ENABLE ) )
    {
        keyString.AddKey( "#define _TEXTURE0_ENABLE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE1_ENABLE ) )
    {
        keyString.AddKey( "#define _TEXTURE1_ENABLE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE2_ENABLE ) )
    {
        keyString.AddKey( "#define _TEXTURE2_ENABLE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE1_COLOR_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _TEXTURE1_COLOR_BLEND_MOD\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE1_COLOR_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _TEXTURE1_COLOR_BLEND_ADD\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE1_COLOR_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _TEXTURE1_COLOR_BLEND_SUB\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE2_COLOR_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _TEXTURE2_COLOR_BLEND_MOD\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE2_COLOR_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _TEXTURE2_COLOR_BLEND_ADD\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE2_COLOR_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _TEXTURE2_COLOR_BLEND_SUB\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE1_ALPHA_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _TEXTURE1_ALPHA_BLEND_MOD\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE1_ALPHA_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _TEXTURE1_ALPHA_BLEND_ADD\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE1_ALPHA_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _TEXTURE1_ALPHA_BLEND_SUB\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE2_ALPHA_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _TEXTURE2_ALPHA_BLEND_MOD\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE2_ALPHA_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _TEXTURE2_ALPHA_BLEND_ADD\n" );
    }
    if( m_PixelShaderFlag.Is( _TEXTURE2_ALPHA_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _TEXTURE2_ALPHA_BLEND_SUB\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_COLOR_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_BLEND_MOD\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_COLOR_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_BLEND_ADD\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_COLOR_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_BLEND_SUB\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_ALPHA_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_BLEND_MOD\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_ALPHA_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_BLEND_ADD\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_ALPHA_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_BLEND_SUB\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_COLOR_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_INPUT_ONE\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_COLOR_INPUT_SRC ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_INPUT_SRC\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_ALPHA_INPUT_A ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_INPUT_A\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_ALPHA_INPUT_RED ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_INPUT_RED\n" );
    }
    if( m_PixelShaderFlag.Is( _PRIMITIVE_ALPHA_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_INPUT_ONE\n" );
    }
    if( m_PixelShaderFlag.Is( _FRAGMENT_SOFT ) )
    {
        keyString.AddKey( "#define _FRAGMENT_SOFT\n" );
    }
    if( m_PixelShaderFlag.Is( _DECAL ) )
    {
        keyString.AddKey( "#define _DECAL\n" );
    }
    if( m_PixelShaderFlag.Is( _FRAGMENT_FRESNEL_ALPHA ) )
    {
        keyString.AddKey( "#define _FRAGMENT_FRESNEL_ALPHA\n" );
    }
    if( m_PixelShaderFlag.Is( _REFRACT_APPLY_ALPHA ) )
    {
        keyString.AddKey( "#define _REFRACT_APPLY_ALPHA\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_COMPARE_NEVER ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_NEVER\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_COMPARE_LESS ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_LESS\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_COMPARE_EQUAL ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_EQUAL\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_COMPARE_LEQUAL ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_LEQUAL\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_COMPARE_GREATER ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_GREATER\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_COMPARE_NOTEQUAL ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_NOTEQUAL\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_COMPARE_GEQUAL ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_GEQUAL\n" );
    }
    if( m_PixelShaderFlag.Is( _ALPHA_COMPARE_ALWAYS ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_ALWAYS\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX0_COLOR_INPUT_SQUARE ) )
    {
        keyString.AddKey( "#define _TEX0_COLOR_INPUT_SQUARE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX1_COLOR_INPUT_SQUARE ) )
    {
        keyString.AddKey( "#define _TEX1_COLOR_INPUT_SQUARE\n" );
    }
    if( m_PixelShaderFlag.Is( _TEX2_COLOR_INPUT_SQUARE ) )
    {
        keyString.AddKey( "#define _TEX2_COLOR_INPUT_SQUARE\n" );
    }
    if( m_PixelShaderFlag.Is( _REFRACT_BY_CAMERA_DISTANCE ) )
    {
        keyString.AddKey( "#define _REFRACT_BY_CAMERA_DISTANCE\n" );
    }



    if( m_PixelShaderFlag2.Is( _TEX0_COLOR_INPUT_ONE_MINUS_RGB ) )
    {
        keyString.AddKey( "#define _TEX0_COLOR_INPUT_ONE_MINUS_RGB\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX1_COLOR_INPUT_ONE_MINUS_RGB ) )
    {
        keyString.AddKey( "#define _TEX1_COLOR_INPUT_ONE_MINUS_RGB\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX2_COLOR_INPUT_ONE_MINUS_RGB ) )
    {
        keyString.AddKey( "#define _TEX2_COLOR_INPUT_ONE_MINUS_RGB\n" );
    }

    if( m_PixelShaderFlag2.Is( _TEX0_COLOR_INPUT_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX0_COLOR_INPUT_ALPHA\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX1_COLOR_INPUT_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX1_COLOR_INPUT_ALPHA\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX2_COLOR_INPUT_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX2_COLOR_INPUT_ALPHA\n" );
    }

    if( m_PixelShaderFlag2.Is( _TEX0_COLOR_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX0_COLOR_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX1_COLOR_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX1_COLOR_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX2_COLOR_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX2_COLOR_INPUT_ONE_MINUS_ALPHA\n" );
    }

    if( m_PixelShaderFlag2.Is( _TEX0_ALPHA_INPUT_ONE_MINUS_RED ) )
    {
        keyString.AddKey( "#define _TEX0_ALPHA_INPUT_ONE_MINUS_RED\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX1_ALPHA_INPUT_ONE_MINUS_RED ) )
    {
        keyString.AddKey( "#define _TEX1_ALPHA_INPUT_ONE_MINUS_RED\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX2_ALPHA_INPUT_ONE_MINUS_RED ) )
    {
        keyString.AddKey( "#define _TEX2_ALPHA_INPUT_ONE_MINUS_RED\n" );
    }

    if( m_PixelShaderFlag2.Is( _TEX0_ALPHA_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX0_ALPHA_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX1_ALPHA_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX1_ALPHA_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if( m_PixelShaderFlag2.Is( _TEX2_ALPHA_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX2_ALPHA_INPUT_ONE_MINUS_ALPHA\n" );
    }

    if( m_PixelShaderFlag2.Is( _PRIMITIVE_COLOR_INPUT_ONE_MINUS_RGB ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_INPUT_ONE_MINUS_RGB\n" );
    }
    if( m_PixelShaderFlag2.Is( _PRIMITIVE_COLOR_INPUT_ALPHA ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_INPUT_ALPHA\n" );
    }
    if( m_PixelShaderFlag2.Is( _PRIMITIVE_COLOR_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if( m_PixelShaderFlag2.Is( _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_RED ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_RED\n" );
    }
    if( m_PixelShaderFlag2.Is( _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if ( m_PixelShaderFlag2.Is( _TEX0_MIPMAP_LEVEL_LIMIT ))
    {
        keyString.AddKey( "#define _TEX0_MIPMAP_LEVEL_LIMIT\n" );
    }
    if ( m_PixelShaderFlag2.Is( _TEX1_MIPMAP_LEVEL_LIMIT ) )
    {
        keyString.AddKey( "#define _TEX1_MIPMAP_LEVEL_LIMIT\n" );
    }
    if ( m_PixelShaderFlag2.Is( _TEX2_MIPMAP_LEVEL_LIMIT ) )
    {
        keyString.AddKey( "#define _TEX2_MIPMAP_LEVEL_LIMIT\n" );
    }
    //----------------------------------
    // 8キーアニメ
    //----------------------------------
    if( m_VertexScaleAnimKey == 1 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_1_KEY\n" );
    }
    else if( m_VertexScaleAnimKey == 2 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_2_KEY\n" );
    }
    else if( m_VertexScaleAnimKey == 3 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_3_KEY\n" );
    }
    else if( m_VertexScaleAnimKey == 4 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_4_KEY\n" );
    }
    else if( m_VertexScaleAnimKey == 5 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_5_KEY\n" );
    }
    else if( m_VertexScaleAnimKey == 6 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_6_KEY\n" );
    }
    else if( m_VertexScaleAnimKey == 7 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_7_KEY\n" );
    }
    else if( m_VertexScaleAnimKey == 8 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_8_KEY\n" );
    }

    if( m_VertexColor0AnimKey == 1 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_1_KEY\n" );
    }
    else if( m_VertexColor0AnimKey == 2 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_2_KEY\n" );
    }
    else if( m_VertexColor0AnimKey == 3 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_3_KEY\n" );
    }
    else if( m_VertexColor0AnimKey == 4 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_4_KEY\n" );
    }
    else if( m_VertexColor0AnimKey == 5 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_5_KEY\n" );
    }
    else if( m_VertexColor0AnimKey == 6 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_6_KEY\n" );
    }
    else if( m_VertexColor0AnimKey == 7 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_7_KEY\n" );
    }
    else if( m_VertexColor0AnimKey == 8 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_8_KEY\n" );
    }

    if( m_VertexColor1AnimKey == 1 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_1_KEY\n" );
    }
    else if( m_VertexColor1AnimKey == 2 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_2_KEY\n" );
    }
    else if( m_VertexColor1AnimKey == 3 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_3_KEY\n" );
    }
    else if( m_VertexColor1AnimKey == 4 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_4_KEY\n" );
    }
    else if( m_VertexColor1AnimKey == 5 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_5_KEY\n" );
    }
    else if( m_VertexColor1AnimKey == 6 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_6_KEY\n" );
    }
    else if( m_VertexColor1AnimKey == 7 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_7_KEY\n" );
    }
    else if( m_VertexColor1AnimKey == 8 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_8_KEY\n" );
    }

    if( m_VertexAlpha0AnimKey == 1 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_1_KEY\n" );
    }
    else if( m_VertexAlpha0AnimKey == 2 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_2_KEY\n" );
    }
    else if( m_VertexAlpha0AnimKey == 3 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_3_KEY\n" );
    }
    else if( m_VertexAlpha0AnimKey == 4 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_4_KEY\n" );
    }
    else if( m_VertexAlpha0AnimKey == 5 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_5_KEY\n" );
    }
    else if( m_VertexAlpha0AnimKey == 6 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_6_KEY\n" );
    }
    else if( m_VertexAlpha0AnimKey == 7 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_7_KEY\n" );
    }
    else if( m_VertexAlpha0AnimKey == 8 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_8_KEY\n" );
    }

    if( m_VertexAlpha1AnimKey == 1 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_1_KEY\n" );
    }
    else if( m_VertexAlpha1AnimKey == 2 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_2_KEY\n" );
    }
    else if( m_VertexAlpha1AnimKey == 3 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_3_KEY\n" );
    }
    else if( m_VertexAlpha1AnimKey == 4 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_4_KEY\n" );
    }
    else if( m_VertexAlpha1AnimKey == 5 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_5_KEY\n" );
    }
    else if( m_VertexAlpha1AnimKey == 6 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_6_KEY\n" );
    }
    else if( m_VertexAlpha1AnimKey == 7 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_7_KEY\n" );
    }
    else if( m_VertexAlpha1AnimKey == 8 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_8_KEY\n" );
    }

    //----------------------------------
    // UVチャンネル
    //----------------------------------
    if( m_VertexShaderFlag.Is( _TEXTURE0_USE_UV1 ) )
    {
        keyString.AddKey( "#define _TEXTURE0_USE_UV1\n" );
    }
    if( m_VertexShaderFlag.Is( _TEXTURE1_USE_UV1 ) )
    {
        keyString.AddKey( "#define _TEXTURE1_USE_UV1\n" );
    }
    if( m_VertexShaderFlag.Is( _TEXTURE2_USE_UV1 ) )
    {
        keyString.AddKey( "#define _TEXTURE2_USE_UV1\n" );
    }


    //----------------------------------
    // ユーザー向けに定義する文字列
    //----------------------------------

    // 描画パス
    if ( m_DrawPathFlag != -1 )
    {
        keyString.AddKey( "#define DRAW_PATH_%d\n", m_DrawPathFlag );

        // 互換性保持
        keyString.AddKey( "#define DRAW_PATH (%d)\n", m_DrawPathFlag );
    }
    else
    {
        keyString.AddKey( "#define DRAW_PATH_NONE\n" );
    }

    // カスタムシェーダインデックス
    if( m_CustomShaderIndex != 0 )
    {
        keyString.AddKey( "#define CUSTOM_SHADER_INDEX_%d\n", m_CustomShaderIndex );
        // 互換性保持
        keyString.AddKey( "#define USR_SETTING_%d\n", m_CustomShaderIndex );
    }
    else
    {
        keyString.AddKey( "#define CUSTOM_SHADER_NONE\n" );
        // 互換性保持
        keyString.AddKey( "#define USR_SETTING_NONE\n" );
    }

    // カスタムシェーダ チェックボックス
    if( m_CustomShaderFlag != 0 )
    {
        for( uint64_t i = 0; i < 64; ++i )
        {
            if( m_CustomShaderFlag & ( static_cast<uint64_t>(0x1) << i ) )
            {
                keyString.AddKey( "#define CUSTOM_SHADER_FLAG_%d\n", i );
                // 互換性保持
                keyString.AddKey( "#define USR_FLAG_%d\n", i );
            }
        }
    }
    else
    {
        keyString.AddKey( "#define CUSTOM_SHADER_FLAG_NONE\n" );
    }

    // カスタムシェーダ スイッチフラグ
    if( m_CustomShaderSwitch != 0 )
    {
        for( uint64_t i = 0; i < 64; ++i )
        {
            if( m_CustomShaderSwitch & ( static_cast<uint64_t>(0x1) << i ) )
            {
                keyString.AddKey( "#define CUSTOM_SHADER_SWITCH_FLAG_%d\n", i );
                keyString.AddKey( "#define USR_SWITCH_FLAG_%d\n", i );
            }
        }
    }
    else
    {
        keyString.AddKey( "#define CUSTOM_SHADER_SWITCH_FLAG_NONE\n" );
    }

    // カスタムフィールド向けキー
    if( m_CustomFieldEnabled != 0 )
    {
        keyString.AddKey( "#define CUSTOM_FIELD_ENABLED\n" );
        for( uint32_t i = 0; i < 16; ++i )
        {
            if( m_CustomFieldFlag & ( 0x1 << i ) )
            {
                keyString.AddKey( "#define CUSTOM_FIELD_FLAG_%d\n", i );
            }
        }
    }
    else
    {
        keyString.AddKey( "#define CUSTOM_FIELD_NONE\n" );
    }

    // ユーザー定義
    if( strlen( m_CustomShaderDefine ) > 0 )
    {
        char temp[ 32 ];
        sprintf_s( temp, 32, "#define %s\n", m_CustomShaderDefine );
        keyString.AddKey( temp );
    }
    keyString.Eof();

}// NOLINT(readability/fn_size)



//---------------------------------------------------------------------------
//  描画パスを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetDrawPath( uint32_t drawPath ) NN_NOEXCEPT
{
    m_DrawPathFlag = drawPath;
}

//---------------------------------------------------------------------------
//  エミッタ挙動タイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetEmitterCalculationType( EmitterCalculationMode type ) NN_NOEXCEPT
{
    if( type == EmitterCalculationMode_Cpu )
    {
        m_VertexShaderFlag.Set( _CALC_TYPE_CPU );
    }
    if( type == EmitterCalculationMode_Gpu )
    {
        m_VertexShaderFlag.Set( _CALC_TYPE_GPU_TIME );
    }
    if( type == EmitterCalculationMode_GpuStreamOut )
    {
        m_VertexShaderFlag.Set( _CALC_TYPE_GPU_SO );
    }
}

//---------------------------------------------------------------------------
//  形状タイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetFigureType( ParticleFigureType type ) NN_NOEXCEPT
{
    switch( type )
    {
    case ParticleFigureType_Particle:  m_VertexShaderFlag.Set( _PARTICLE );  break;
    case ParticleFigureType_Primitive: m_VertexShaderFlag.Set( _PRIMITIVE ); break;
    default: break;
    }
}

//---------------------------------------------------------------------------
//! @brief  世界座標系で重力を適用 を設定します。
//---------------------------------------------------------------------------
void  ShaderKey::SetParticleWorldGravity( bool worldGravity ) NN_NOEXCEPT
{
    if( worldGravity )
    {
        m_VertexShaderFlag.Set( _WORLD_GRAVITY );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _WORLD_GRAVITY );
    }
}

//---------------------------------------------------------------------------
//  パーティクルタイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleType( ParticleBillboardType type ) NN_NOEXCEPT
{
    switch( type )
    {
    case ParticleBillboardType_Billboard:                           m_VertexShaderFlag.Set( _PARTICLE_TYPE_BILLBOARD );                 break;
    case ParticleBillboardType_AdvancedBillboard:                   m_VertexShaderFlag.Set( _PARTICLE_TYPE_COMPLEX_BILLBOARD );         break;
    case ParticleBillboardType_YBillboard:                          m_VertexShaderFlag.Set( _PARTICLE_TYPE_Y_BILLBOARD );               break;
    case ParticleBillboardType_PolygonOnXyPlane:                    m_VertexShaderFlag.Set( _PARTICLE_TYPE_POLYGON_XY );                break;
    case ParticleBillboardType_PolygonOnXzPlane:                    m_VertexShaderFlag.Set( _PARTICLE_TYPE_POLYGON_XZ );                break;
    case ParticleBillboardType_VelocityDirectionalBillboard:        m_VertexShaderFlag.Set( _PARTICLE_TYPE_VEL_LOOK );                  break;
    case ParticleBillboardType_VelocityDirectionalPolygon:          m_VertexShaderFlag.Set( _PARTICLE_TYPE_VEL_LOOK_POLYGON );          break;
    case ParticleBillboardType_SimpleYBillboard:                    m_VertexShaderFlag.Set( _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL );      break;
    case ParticleBillboardType_PrimitiveLookBillboard:              m_VertexShaderFlag.Set( _PARTICLE_TYPE_WITH_SCALE_Z );              break;
    case ParticleBillboardType_PrimitiveLookDirectionalBillboard:   m_VertexShaderFlag.Set( _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL );  break;
    default: break;
    }
}

//---------------------------------------------------------------------------
//  パーティクル回転を設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleRotateType( ParticleRotationType type ) NN_NOEXCEPT
{
    switch( type )
    {
    case ParticleRotationType_Xyz:  m_VertexShaderFlag.Set( _ROTATE_XYZ );  break;
    case ParticleRotationType_Zxy:  m_VertexShaderFlag.Set( _ROTATE_ZXY );  break;
    default:                        m_VertexShaderFlag.Set( _ROTATE_YZX );  break;
    }
}

//---------------------------------------------------------------------------
//! @brief  エミッタ追従タイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetEmitterFollowType( ParticleFollowType type ) NN_NOEXCEPT
{
    switch( type )
    {
    case ParticleFollowType_EmitterFull:        m_VertexShaderFlag.Set( _EMITTER_FOLLOW_TYPE_ALL );     break;
    case ParticleFollowType_None:               m_VertexShaderFlag.Set( _EMITTER_FOLLOW_TYPE_NONE );    break;
    case ParticleFollowType_EmitterPosition:    m_VertexShaderFlag.Set( _EMITTER_FOLLOW_TYPE_POS );     break;
    default: break;
    }
}

//---------------------------------------------------------------------------
//  カラー計算タイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleColorCalculationType( ParticleColorCalculationMode color0Type, ParticleColorCalculationMode color1Type ) NN_NOEXCEPT
{
    switch( color0Type )
    {
    case ParticleColorCalculationMode_FixedColor:   m_VertexShaderFlag.Set( _COLOR_0_FIXED );   break;
    case ParticleColorCalculationMode_RandomColor:  m_VertexShaderFlag.Set( _COLOR_0_RANDOM );  break;
    default: break;
    }
    switch( color1Type )
    {
    case ParticleColorCalculationMode_FixedColor:   m_VertexShaderFlag.Set( _COLOR_1_FIXED );   break;
    case ParticleColorCalculationMode_RandomColor:  m_VertexShaderFlag.Set( _COLOR_1_RANDOM );  break;
    default: break;
    }
}

//---------------------------------------------------------------------------
//  パーティクルアルファ計算タイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleAlphaCalculationType( ParticleColorCalculationMode alpha0Type, ParticleColorCalculationMode alpha1Type ) NN_NOEXCEPT
{
    switch( alpha0Type )
    {
    case ParticleColorCalculationMode_FixedColor:   m_VertexShaderFlag.Set( _ALPHA_0_FIXED );   break;
    default: break;
    }
    switch( alpha1Type )
    {
    case ParticleColorCalculationMode_FixedColor:   m_VertexShaderFlag.Set( _ALPHA_1_FIXED );   break;
    default: break;
    }
}

//---------------------------------------------------------------------------
//  パーティクルテクスチャパターンアニメ有効無効を設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleTexturePatternAnim( bool texture0, bool texture1, bool texture2 ) NN_NOEXCEPT
{
    if( texture0 )
    {
        m_VertexShaderFlag.Set( _TEX_0_PATTERN_ANIM );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _TEX_0_PATTERN_ANIM );
    }
    if( texture1 )
    {
        m_VertexShaderFlag.Set( _TEX_1_PATTERN_ANIM );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _TEX_1_PATTERN_ANIM );
    }
    if( texture2 )
    {
        m_VertexShaderFlag.Set( _TEX_2_PATTERN_ANIM );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _TEX_2_PATTERN_ANIM );
    }
}

//---------------------------------------------------------------------------
//  パーティクルテクスチャサーフェイスアニメ有効無効を設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleTextureSurfaceAnim( bool texture0, bool texture1, bool texture2 ) NN_NOEXCEPT
{
    if( texture0 )
    {
        m_PixelShaderFlag2.Set( _TEX_0_SURFACE_ANIM );
    }
    else
    {
        m_PixelShaderFlag2.Cancel( _TEX_0_SURFACE_ANIM );
    }
    if( texture1 )
    {
        m_PixelShaderFlag2.Set( _TEX_1_SURFACE_ANIM );
    }
    else
    {
        m_PixelShaderFlag2.Cancel( _TEX_1_SURFACE_ANIM );
    }
    if( texture2 )
    {
        m_PixelShaderFlag2.Set( _TEX_2_SURFACE_ANIM );
    }
    else
    {
        m_PixelShaderFlag2.Cancel( _TEX_2_SURFACE_ANIM );
    }
}

//---------------------------------------------------------------------------
//  パーティクルテクスチャ座標アニメ有効無効を設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleTextureShiftAnim( bool texture0, bool texture1, bool texture2 ) NN_NOEXCEPT
{
    if( texture0 )
    {
        m_VertexShaderFlag.Set( _TEX_0_SHIFT_ANIM );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _TEX_0_SHIFT_ANIM );
    }
    if( texture1 )
    {
        m_VertexShaderFlag.Set( _TEX_1_SHIFT_ANIM );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _TEX_1_SHIFT_ANIM );
    }
    if( texture2 )
    {
        m_VertexShaderFlag.Set( _TEX_2_SHIFT_ANIM );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _TEX_2_SHIFT_ANIM );
    }
}

//---------------------------------------------------------------------------
//  パーティクルスフィアマップ有効無効を設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleSphereMap( bool texture0, bool texture1, bool texture2 ) NN_NOEXCEPT
{
    if( texture0 )
    {
        m_VertexShaderFlag.Set( _TEX_0_SPHERE_MAP );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _TEX_0_SPHERE_MAP );
    }
    if( texture1 )
    {
        m_VertexShaderFlag.Set( _TEX_1_SPHERE_MAP );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _TEX_1_SPHERE_MAP );
    }
    if( texture2 )
    {
        m_VertexShaderFlag.Set( _TEX_2_SPHERE_MAP );
    }
    else
    {
        m_VertexShaderFlag.Cancel( _TEX_2_SPHERE_MAP );
    }
}

//---------------------------------------------------------------------------
//  ピクセルシェーダを標準設定にします。
//---------------------------------------------------------------------------
void ShaderKey::SetDefaultPixelShaderSetting() NN_NOEXCEPT
{
    m_PixelShaderFlag.Set( _SHADER_TYPE_NORMAL );
    m_PixelShaderFlag.Set( _COLOR_PROCESS_TEXTURE );
    m_PixelShaderFlag.Set( _ALPHA_PROCESS_MOD );
    m_PixelShaderFlag.Set( _TEXTURE1_COLOR_BLEND_MOD );
    m_PixelShaderFlag.Set( _TEXTURE2_COLOR_BLEND_MOD );
    m_PixelShaderFlag.Set( _TEXTURE1_ALPHA_BLEND_MOD );
    m_PixelShaderFlag.Set( _TEXTURE2_ALPHA_BLEND_MOD );
    m_PixelShaderFlag.Set( _PRIMITIVE_COLOR_BLEND_MOD );
    m_PixelShaderFlag.Set( _PRIMITIVE_ALPHA_BLEND_MOD );
    m_PixelShaderFlag.Set( _PRIMITIVE_COLOR_INPUT_SRC );
    m_PixelShaderFlag.Set( _PRIMITIVE_ALPHA_INPUT_A );
}

//---------------------------------------------------------------------------
//  color/alpha 8keyアニメを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetColor8KeyAnim( uint8_t color0Anim, uint8_t color1Anim ) NN_NOEXCEPT
{
    m_VertexColor0AnimKey = color0Anim;
    m_VertexColor1AnimKey = color1Anim;
}

//---------------------------------------------------------------------------
//  揺らぎを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetFluctuation( uint32_t fluctuation ) NN_NOEXCEPT
{
    switch( fluctuation )
    {
    case 1: m_VertexShaderFlag.Set( _FLUCTUATION_SCALE_ENABLE );        break;
    case 2: m_VertexShaderFlag.Set( _FLUCTUATION_ALPHA_ENABLE );        break;
    case 3: m_VertexShaderFlag.Set( _FLUCTUATION_SCALE_Y_SEPARATE );    break;
    default: break;
    }
}
//---------------------------------------------------------------------------
//  全てのパーティクルタイプをクリアする。
//---------------------------------------------------------------------------
void ShaderKey::ClearAllParticleType() NN_NOEXCEPT
{
    m_VertexShaderFlag.Cancel( _PARTICLE_TYPE_BILLBOARD );
    m_VertexShaderFlag.Cancel( _PARTICLE_TYPE_COMPLEX_BILLBOARD );
    m_VertexShaderFlag.Cancel( _PARTICLE_TYPE_Y_BILLBOARD );
    m_VertexShaderFlag.Cancel( _PARTICLE_TYPE_POLYGON_XY );
    m_VertexShaderFlag.Cancel( _PARTICLE_TYPE_POLYGON_XZ );
    m_VertexShaderFlag.Cancel( _PARTICLE_TYPE_VEL_LOOK );
    m_VertexShaderFlag.Cancel( _PARTICLE_TYPE_VEL_LOOK_POLYGON );
    m_VertexShaderFlag.Cancel( _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL );
    m_VertexShaderFlag.Cancel( _PARTICLE_TYPE_WITH_SCALE_Z );
    m_VertexShaderFlag.Cancel( _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL );
}

//---------------------------------------------------------------------------
//  全ての追従タイプをクリアする。
//---------------------------------------------------------------------------
void ShaderKey::ClearAllFollowType() NN_NOEXCEPT
{
    m_VertexShaderFlag.Cancel( _EMITTER_FOLLOW_TYPE_ALL );
    m_VertexShaderFlag.Cancel( _EMITTER_FOLLOW_TYPE_NONE );
    m_VertexShaderFlag.Cancel( _EMITTER_FOLLOW_TYPE_POS );
}

//---------------------------------------------------------------------------
//  全ての挙動計算タイプをクリアする。
//---------------------------------------------------------------------------
void ShaderKey::ClearAllCalculationType() NN_NOEXCEPT
{
    m_VertexShaderFlag.Cancel( _CALC_TYPE_CPU );
    m_VertexShaderFlag.Cancel( _CALC_TYPE_GPU_TIME );
    m_VertexShaderFlag.Cancel( _CALC_TYPE_GPU_SO );
}

//---------------------------------------------------------------------------
//  全ての挙動計算タイプをクリアする。
//---------------------------------------------------------------------------
void ShaderKey::ClearAllFluctuation() NN_NOEXCEPT
{
    m_VertexShaderFlag.Cancel( _FLUCTUATION_SCALE_ENABLE );
    m_VertexShaderFlag.Cancel( _FLUCTUATION_ALPHA_ENABLE );
    m_VertexShaderFlag.Cancel( _FLUCTUATION_SCALE_Y_SEPARATE );
}

} // namespace detail
} // namespace vfx
} // namespace nn
