﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define _CRT_SECURE_NO_WARNINGS

#include "ShaderCodeGeneratorGeneric_PCH.h"
#include "ShaderConverter.h"
#include <nw/eft/eft2_Data.h>
#include <assert.h>

// To enable profiling of the shader conversion, uncomment the line below.
#define ENABLE_PROFILING

// To enable shader binary duplication checking, uncomment the line below.
//#define CHECK_DUP_BINARY

namespace EffectMaker {
namespace ShaderCodeGeneratorGeneric {

//------------------------------------------------------------------------------
//      Constructor
//------------------------------------------------------------------------------
ShaderConverter::ShaderConverter() :
    szVshCodeCompositeBuffer(NULL),
    szFshCodeCompositeBuffer(NULL),
    mIsField(false),
    mReservedShaderIndex(-1)
{
    this->szVshCodeCompositeBuffer = new char[ShaderCodeBufferSize];
    this->szFshCodeCompositeBuffer = new char[ShaderCodeBufferSize];
}

//------------------------------------------------------------------------------
//      初期化処理
//------------------------------------------------------------------------------
void ShaderConverter:: Initialize(ShaderCodeListCpp* pShaderCodes)
{
    mShaderCodes         = pShaderCodes;
    mReservedShaderIndex = -1;
    mCustomShaderIndex   = -1;
}

//------------------------------------------------------------------------------
//      デストラクタ
//------------------------------------------------------------------------------
ShaderConverter::~ShaderConverter()
{// Clean up.
    if (this->szVshCodeCompositeBuffer != NULL)
    {
        delete[] this->szVshCodeCompositeBuffer;
        this->szVshCodeCompositeBuffer = NULL;
    }

    if (this->szFshCodeCompositeBuffer != NULL)
    {
        delete[] this->szFshCodeCompositeBuffer;
        this->szFshCodeCompositeBuffer = NULL;
    }
}

//------------------------------------------------------------------------------
//      カスタムフィールドを取得する
//------------------------------------------------------------------------------
template<class T_out, class T_in> T_out* ShaderConverter::GetCustomField( T_in* binaryData )
{
    T_in* subData = binaryData->GetSubData();
    while( subData )
    {
        if ( ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'F', 'C', 'S', 'F' ) ) )
        {
            return reinterpret_cast<T_out*>( subData->GetBinaryData() );
            break;
        }

        subData = subData->GetNextData();
    }

    return NULL;
}

//------------------------------------------------------------------------------
//      エミッタプラグインを取得する
//------------------------------------------------------------------------------
template<class T> void* ShaderConverter::GetEmitterPlugin( T* binaryData )
{
    T* subData = binaryData->GetSubData();
    while( subData )
    {
        if (   ( subData->GetBinaryTag() == VFX_MAKE_TAG( 'E', 'P', '0', '1' ) )
            || ( subData->GetBinaryTag() == VFX_MAKE_TAG( 'E', 'P', '0', '2' ) )
            || ( subData->GetBinaryTag() == VFX_MAKE_TAG( 'E', 'P', '0', '3' ) )
            || ( subData->GetBinaryTag() == VFX_MAKE_TAG( 'E', 'P', '0', '4' ) )
            || ( subData->GetBinaryTag() == VFX_MAKE_TAG( 'E', 'P', '0', '5' ) )
            || ( subData->GetBinaryTag() == VFX_MAKE_TAG( 'E', 'P', '0', '6' ) )
            || ( subData->GetBinaryTag() == VFX_MAKE_TAG( 'E', 'P', '0', '7' ) )
            || ( subData->GetBinaryTag() == VFX_MAKE_TAG( 'E', 'P', '0', '8' ) ) )
        {
            return ( subData->GetBinaryData() );
            break;
        }

        subData = subData->GetNextData();
    }

    return NULL;
}

//------------------------------------------------------------------------------
//      フィールドデータの存在チェック
//------------------------------------------------------------------------------
template<class T> bool ShaderConverter::IsFieldExist( T* binaryData )
{
    T* subData = binaryData->GetSubData();
    while( subData )
    {
        if ( subData->GetBinaryTag() == EFT_MAKE_TAG( 'F', 'R', 'N', 'D' ) )
        {
            return true;
            break;
        }

        subData = subData->GetNextData();
    }

    return false;
}
//------------------------------------------------------------------------------
//      シェーダソースコードの生成
//------------------------------------------------------------------------------
bool ShaderConverter::GenerateShaderCode( ShaderConverterEmitterData* pEmitterData,
                                          char**  ppVertexShader,
                                          s32* pVertexShaderSize,
                                          char**  ppFragmentShader,
                                          s32* pFragmentShaderSize,
                                          char**  ppComputeShader,
                                          s32* pComputeShaderSize,
                                          u32 userDefineIndex)
{
    nw::eft2::ShaderKey shaderKey;

    // Get the reserved shader index.
    mReservedShaderIndex = pEmitterData->reservedShaderIndex;

    nw::eft2::ResEmitter* emitterData = reinterpret_cast<nw::eft2::ResEmitter*>( pEmitterData->pResEmitter );
    nw::eft2::BinaryData *pBinHeader = reinterpret_cast<nw::eft2::BinaryData*>( pEmitterData->pBinaryHeader );
    void* emitterPluginData = GetEmitterPlugin( pBinHeader );
    nw::eft2::ResFieldCustom* resFieldCustom = GetCustomField<nw::eft2::ResFieldCustom>( pBinHeader );

    u32 fieldFlag = 0;
    if ( IsFieldExist( pBinHeader ) )
    {
        fieldFlag |= EFT_SHADER_KEY_FIELD_RANDOM;
        mIsField = true;
    }
    else
    {
        mIsField = false;
    }


    // TODO: CustomShader Codeを設定すること

    //
    // shaderKeyを作成する。
    //
    s32 reservedShaderIndex = ( mReservedShaderIndex >= 0 ) ? mReservedShaderIndex + 1 : mReservedShaderIndex;

    char *userDefineStr = NULL;
    if ( userDefineIndex == 1 ) userDefineStr = emitterData->shader.userShaderDefine1;
    else if ( userDefineIndex == 2 ) userDefineStr = emitterData->shader.userShaderDefine2;

    // コンバイナシェーダがない場合は、indexを-1にしておく
    if( pEmitterData->iOverrideShaderLength < 1 )
    {
        emitterData->shader.eftCombinerShaderIndex = static_cast<s32>(-1);
    }

    shaderKey.Initialize( emitterData, reservedShaderIndex, emitterPluginData, fieldFlag, resFieldCustom, userDefineStr );
    // Custom Shader Indexの設定
    mCustomShaderIndex = emitterData->shader.customShaderIndex;

    // shader codeを生成する
    char* setting = (char *)shaderKey.GetCompileSetting();
    u32   settingSize = (u32)strlen( setting );

    memset( this->szVshCodeCompositeBuffer, 0, ShaderCodeBufferSize );
    memset( this->szFshCodeCompositeBuffer, 0, ShaderCodeBufferSize );

    // 頂点シェーダコードを生成する
    u32 shaderVshSize = GenerateVertexShaderCode( this->szVshCodeCompositeBuffer,
                                                  setting,
                                                  settingSize,
                                                  NULL,
                                                  0,
                                                  true,
                                                  mCustomShaderIndex,
                                                  mReservedShaderIndex,
                                                  false,
                                                  false);

    // フラグメントシェーダコードを生成する
    u32 shaderFshSize = GenerateFragmentShaderCode( this->szFshCodeCompositeBuffer,
                                                    setting,
                                                    settingSize,
                                                    pEmitterData->szOverrideShader,
                                                    pEmitterData->iOverrideShaderLength,
                                                    true,
                                                    mCustomShaderIndex,
                                                    mReservedShaderIndex,
                                                    false,
                                                    false);

    // Set up the shader code and size.
    *pVertexShaderSize = shaderVshSize;
    *ppVertexShader = new char[shaderVshSize + 1];
    memcpy_s(*ppVertexShader, shaderVshSize + 1, this->szVshCodeCompositeBuffer, shaderVshSize);
    (*ppVertexShader)[shaderVshSize] = '\0';

    *pFragmentShaderSize = shaderFshSize;
    *ppFragmentShader = new char[shaderFshSize + 1];
    memcpy_s(*ppFragmentShader, shaderFshSize + 1, this->szFshCodeCompositeBuffer, shaderFshSize);
    (*ppFragmentShader)[shaderFshSize] = '\0';

    // コンピュートシェーダを生成する (バッファはバーテックスシェーダのものを使いまわす)
    memset( this->szVshCodeCompositeBuffer, 0, ShaderCodeBufferSize );
    u32 shaderCshSize = GenerateVertexShaderCode( this->szVshCodeCompositeBuffer,
                                                  NULL,
                                                  0,
                                                  NULL,
                                                  0,
                                                  true,
                                                  mCustomShaderIndex,
                                                  mReservedShaderIndex,
                                                  true,
                                                  true);
    *pComputeShaderSize = shaderCshSize;
    *ppComputeShader = new char[shaderCshSize + 1];
    memcpy_s(*ppComputeShader, shaderCshSize + 1, this->szVshCodeCompositeBuffer, shaderCshSize);
    (*ppComputeShader)[shaderCshSize] = '\0';

    return true;
}

//------------------------------------------------------------------------------
//      VfxのShaderKeyとシェーダソースコードの生成
//------------------------------------------------------------------------------
bool ShaderConverter::GenerateVfxShaderKeyAndCodes( ShaderConverterEmitterData* pEmitterData,
                                                   char**                      ppShaderKey,
                                                   s32*                        pShaderKeySize,
                                                   char**                      ppVertexShader,
                                                   s32*                        pVertexShaderSize,
                                                   char**                      ppFragmentShader,
                                                   s32*                        pFragmentShaderSize,
                                                   char**                      ppComputeShader,
                                                   s32*                        pComputeShaderSize,
                                                   u32                         userDefineIndex )
{
    nn::vfx::detail::ShaderKey shaderKey;

    mReservedShaderIndex = pEmitterData->reservedShaderIndex;

    nn::vfx::detail::ResEmitter* emitterData = reinterpret_cast<nn::vfx::detail::ResEmitter *>( pEmitterData->pResEmitter );
    nn::vfx::detail::BinaryData *pBinHeader = reinterpret_cast<nn::vfx::detail::BinaryData *>( pEmitterData->pBinaryHeader );
    void* emitterPluginData = GetEmitterPlugin( pBinHeader );
    nn::vfx::detail::ResFieldCustom* resFieldCustom = GetCustomField<nn::vfx::detail::ResFieldCustom>( pBinHeader );

    u32 fieldFlag = 0;
    if ( IsFieldExist( pBinHeader ) )
    {
        fieldFlag |= EFT_SHADER_KEY_FIELD_RANDOM;
        mIsField = true;
    }
    else
    {
        mIsField = false;
    }

    s32 reservedShaderIndex = ( mReservedShaderIndex >= 0 ) ? mReservedShaderIndex + 1 : mReservedShaderIndex;

    char *userDefineStr = NULL;
    if ( userDefineIndex == 1 ) userDefineStr = emitterData->shader.userShaderDefine1;
    else if ( userDefineIndex == 2 ) userDefineStr = emitterData->shader.userShaderDefine2;

    // コンバイナシェーダがない場合は、indexを-1にしておく
    if ( pEmitterData->iOverrideShaderLength < 1 )
    {
        emitterData->shader.eftCombinerShaderIndex = (u32)-1;
    }

    shaderKey.Initialize( emitterData, reservedShaderIndex, emitterPluginData, fieldFlag, resFieldCustom, userDefineStr );

    // ShaderKeyのマクロを保存する
    char* pShaderKeyBuffer     = (char *)shaderKey.GetCompileSetting();
    *pShaderKeySize = (u32)strlen( pShaderKeyBuffer );
    char reservedShaderKey[256];
    if ( mReservedShaderIndex >= 0 )
    {
        sprintf_s(reservedShaderKey, sizeof(reservedShaderKey), "#define RESERVED_SHADER_%d\n", mReservedShaderIndex);
    }

    *ppShaderKey = new char[*pShaderKeySize + strlen(reservedShaderKey)+ 1];
    memcpy_s(*ppShaderKey, *pShaderKeySize, pShaderKeyBuffer, *pShaderKeySize);
    memcpy_s(*ppShaderKey + *pShaderKeySize , strlen(reservedShaderKey) , reservedShaderKey, strlen(reservedShaderKey) );
    *pShaderKeySize+=static_cast<s32>(strlen(reservedShaderKey));
    (*ppShaderKey)[*pShaderKeySize] = '\0';

    // Custom Shader Indexの設定
    mCustomShaderIndex = emitterData->shader.customShaderIndex;

    // offsetを設定する
    nn::vfx::detail::ResShader* shader          = &emitterData->shader;
    pEmitterData->shaderIndexOffset      = (u32)((u64)&shader->shaderIndex - (u64)emitterData);

    // shader codeを生成する
    char* setting = "\n";
    u32   settingSize = (u32)strlen( setting );

    memset( this->szVshCodeCompositeBuffer, 0, ShaderCodeBufferSize );
    memset( this->szFshCodeCompositeBuffer, 0, ShaderCodeBufferSize );

    // 頂点シェーダコードを生成する
    u32 shaderVshSize = GenerateVertexShaderCode( this->szVshCodeCompositeBuffer,
                                                  setting,
                                                  settingSize,
                                                  NULL,
                                                  0,
                                                  true,
                                                  mCustomShaderIndex,
                                                  mReservedShaderIndex,
                                                  false,
                                                  true);

    // フラグメントシェーダコードを生成する
    // Vfxスペックでは1ソースにまとめなくてはいけないため、ここでは付けない
    u32 shaderFshSize = GenerateFragmentShaderCode( this->szFshCodeCompositeBuffer,
                                                    setting,
                                                    settingSize,
                                                    pEmitterData->szOverrideShader,
                                                    0, // pEmitterData->iOverrideShaderLength,
                                                    true,
                                                    mCustomShaderIndex,
                                                    mReservedShaderIndex,
                                                    false,
                                                    true);

    // Set up the shader code and size.
    *pVertexShaderSize = shaderVshSize;
    *ppVertexShader = new char[shaderVshSize + 1];
    memcpy_s(*ppVertexShader, shaderVshSize + 1, this->szVshCodeCompositeBuffer, shaderVshSize);
    (*ppVertexShader)[shaderVshSize] = '\0';

    *pFragmentShaderSize = shaderFshSize;
    *ppFragmentShader = new char[shaderFshSize + 1];
    memcpy_s(*ppFragmentShader, shaderFshSize + 1, this->szFshCodeCompositeBuffer, shaderFshSize);
    (*ppFragmentShader)[shaderFshSize] = '\0';

    // コンピュートシェーダを生成する (バッファはバーテックスシェーダのものを使いまわす)
    memset( this->szVshCodeCompositeBuffer, 0, ShaderCodeBufferSize );
    u32 shaderCshSize = GenerateVertexShaderCode( this->szVshCodeCompositeBuffer,
                                                  NULL,
                                                  0,
                                                  NULL,
                                                  0,
                                                  true,
                                                  mCustomShaderIndex,
                                                  mReservedShaderIndex,
                                                  true,
                                                  true);
    *pComputeShaderSize = shaderCshSize;
    *ppComputeShader = new char[shaderCshSize + 1];
    memcpy_s(*ppComputeShader, shaderCshSize + 1, this->szVshCodeCompositeBuffer, shaderCshSize);
    (*ppComputeShader)[shaderCshSize] = '\0';

    return true;
}

//------------------------------------------------------------------------------
//      頂点シェーダソースの生成
//------------------------------------------------------------------------------
u32 ShaderConverter::GenerateVertexShaderCode(
    char* shaderCodeBuf,
    char* compileSettings,
    u32   settingSize,
    char* overrideVertexShader,
    s32   overrideVertexShaderLength,
    bool  useCustomShader,
    s32   customShaderIndex,
    s32   reservedShaderIndex,
    bool  useStreamOut,
    bool  expandAllShaders )
{
    CopyShaderCode shaderCode( shaderCodeBuf );

    shaderCode.CopyCode( mShaderCodes->mSpecDecShaderSrc.buf,  mShaderCodes->mSpecDecShaderSrc.size  );
    if (!useStreamOut)
    {
        shaderCode.CopyCode( compileSettings, settingSize );
    }

    shaderCode.CopyCode( mShaderCodes->mParticleGlslSrc.buf,   mShaderCodes->mParticleGlslSrc.size   );

    if (!useStreamOut)
    {
        shaderCode.CopyCode( mShaderCodes->mParticleDecVshSrc.buf, mShaderCodes->mParticleDecVshSrc.size );
    }

    if ( overrideVertexShader && overrideVertexShaderLength != 0 )
    {
        shaderCode.CopyCode( overrideVertexShader, overrideVertexShaderLength );
    }

    // Custom shaderのコピー
    if ( useCustomShader == true )
    {
        if ( expandAllShaders == true && useStreamOut == false )
        {
            const char endifPreprocessor[] = "#endif\n";

            // 共通シェーダを展開する.
            if ( mShaderCodes->mGeneralVshSrc.buf  != NULL &&
                 mShaderCodes->mGeneralVshSrc.size != 0 )
            {
                const char shaderMacro[] = "#if CUSTOM_SHADER_NONE_CONVERTER || USR_SETTING_NONE_CONVERTER\n";
                shaderCode.CopyCode( shaderMacro, static_cast<u32>(strlen(shaderMacro)) );
                shaderCode.CopyCode( mShaderCodes->mGeneralVshSrc.buf, mShaderCodes->mGeneralVshSrc.size );
                shaderCode.CopyCode( endifPreprocessor, static_cast<u32>(strlen(endifPreprocessor)) );
            }

            // 全カスタムシェーダを展開する.
            for ( int i = 1; i <= 8; i++ )
            {
                const ShaderCodeParam &shaderCodeParam = mShaderCodes->mCustomVshSrc[i-1];
                if ( shaderCodeParam.buf != NULL &&
                     shaderCodeParam.size != 0 )
                {
                    char shaderMacro[256];
                    sprintf_s(shaderMacro, sizeof(shaderMacro), "#if CUSTOM_SHADER_INDEX_%d_CONVERTER || USR_SETTING_%d_CONVERTER\n", i, i);
                    shaderCode.CopyCode( shaderMacro, static_cast<u32>(strlen(shaderMacro)) );
                    shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
                    shaderCode.CopyCode( endifPreprocessor, static_cast<u32>(strlen(endifPreprocessor)) );
                }
            }
        }
        else
        {
            if ( customShaderIndex <= 0 || useStreamOut == true )
            {
                if ( mShaderCodes->mGeneralVshSrc.buf  != NULL &&
                     mShaderCodes->mGeneralVshSrc.size != 0 )
                {
                    shaderCode.CopyCode( mShaderCodes->mGeneralVshSrc.buf,
                                         mShaderCodes->mGeneralVshSrc.size );
                }
            }
            else
            {
                const ShaderCodeParam &shaderCodeParam =
                    mShaderCodes->mCustomVshSrc[customShaderIndex - 1];

                if ( shaderCodeParam.buf  != NULL &&
                     shaderCodeParam.size != 0 )
                {
                    shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
                }
            }
        }
    }

    if ( useStreamOut == false )
    {
        if ( expandAllShaders == true )
        {
            const char endifPreprocessor[] = "#endif\n";

            for ( int i = 0; i < 8; i++)
            {
                const ShaderCodeParam &shaderCodeParam = mShaderCodes->mReservedVshSrc[i];

                if ( shaderCodeParam.buf != NULL &&
                    shaderCodeParam.size != 0 )
                {
                    char shaderMacro[256];
                    sprintf_s(shaderMacro, sizeof(shaderMacro), "#if RESERVED_SHADER_%d_CONVERTER\n", i);
                    shaderCode.CopyCode( shaderMacro, static_cast<u32>(strlen(shaderMacro)) );
                    shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
                    shaderCode.CopyCode( endifPreprocessor, static_cast<u32>(strlen(endifPreprocessor)) );
                }
            }
        }
        else
        {
            // Reserved shaderのコピー
            if ( reservedShaderIndex >= 0 )
            {
                const ShaderCodeParam &shaderCodeParam =
                    mShaderCodes->mReservedVshSrc[reservedShaderIndex];

                if ( shaderCodeParam.buf  != NULL &&
                     shaderCodeParam.size != 0 )
                {
                    shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
                }
            }
        }
    }

    // Streamout shaderのコピー
    if ( useStreamOut == true )
    {
        shaderCode.CopyCode( mShaderCodes->mStreamOutDecVshSrc.buf,
                             mShaderCodes->mStreamOutDecVshSrc.size );

        if ( mShaderCodes->mGeneralCshSrc.buf != NULL &&
             mShaderCodes->mGeneralCshSrc.size != 0 )
        {
            shaderCode.CopyCode( mShaderCodes->mGeneralCshSrc.buf,
                                 mShaderCodes->mGeneralCshSrc.size );
        }

        shaderCode.CopyCode( mShaderCodes->mStreamOutVshSrc.buf,
                             mShaderCodes->mStreamOutVshSrc.size );
    }

    // 共通コードをコピー (StreamOut shader以外)
    if ( useStreamOut == false )
    {
        shaderCode.CopyCode( mShaderCodes->mVertexShaderSrc.buf,
                             mShaderCodes->mVertexShaderSrc.size );
    }

    return  shaderCode.GetShaderSize();
}

//------------------------------------------------------------------------------
//      フラグメントシェーダソースの生成
//------------------------------------------------------------------------------
u32 ShaderConverter::GenerateFragmentShaderCode(
    char* shaderCodeBuf,
    char* compileSettings,
    u32   settingSize,
    char* overrideFragmentShader,
    s32   overrideFragmentShaderLength,
    bool  useCustomShader,
    s32   customShaderIndex,
    s32   reservedShaderIndex,
    bool  useStreamOut,
    bool  expandAllShaders)
{
    CopyShaderCode shaderCode( shaderCodeBuf );

    shaderCode.CopyCode( mShaderCodes->mSpecDecShaderSrc.buf,  mShaderCodes->mSpecDecShaderSrc.size  );
    if (!useStreamOut)
    {
        shaderCode.CopyCode( compileSettings, settingSize );
    }

    shaderCode.CopyCode( mShaderCodes->mParticleGlslSrc.buf,   mShaderCodes->mParticleGlslSrc.size   );
    shaderCode.CopyCode( mShaderCodes->mParticleDecFshSrc.buf, mShaderCodes->mParticleDecFshSrc.size );

    // Custum shaderのコピー
    if ( useCustomShader == true )
    {
        if ( expandAllShaders == true && useStreamOut == false)
        {
            const char endifPreprocessor[] = "#endif\n";

            // 共通シェーダを展開する.
            if ( mShaderCodes->mGeneralFshSrc.buf  != NULL &&
                 mShaderCodes->mGeneralFshSrc.size != 0 )
            {
                const char shaderMacro[] = "#if CUSTOM_SHADER_NONE_CONVERTER || USR_SETTING_NONE_CONVERTER\n";
                shaderCode.CopyCode( shaderMacro, static_cast<u32>(strlen(shaderMacro)) );
                shaderCode.CopyCode( mShaderCodes->mGeneralFshSrc.buf, mShaderCodes->mGeneralFshSrc.size );
                shaderCode.CopyCode( endifPreprocessor, static_cast<u32>(strlen(endifPreprocessor)) );
            }

            // 全カスタムシェーダを展開する.
            for ( int i = 1; i <= 8; i++ )
            {
                const ShaderCodeParam &shaderCodeParam = mShaderCodes->mCustomFshSrc[i-1];
                if ( shaderCodeParam.buf != NULL &&
                    shaderCodeParam.size != 0 )
                {
                    char shaderMacro[256];
                    sprintf_s(shaderMacro, sizeof(shaderMacro), "#if CUSTOM_SHADER_INDEX_%d_CONVERTER || USR_SETTING_%d_CONVERTER\n", i, i);
                    shaderCode.CopyCode( shaderMacro, static_cast<u32>(strlen(shaderMacro)) );
                    shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
                    shaderCode.CopyCode( endifPreprocessor, static_cast<u32>(strlen(endifPreprocessor)) );
                }
            }
        }
        else
        {
            // 必要なカスタムシェーダだけを、展開する
            if ( customShaderIndex <= 0 || useStreamOut == true)
            {
                if ( mShaderCodes->mGeneralFshSrc.buf  != NULL &&
                     mShaderCodes->mGeneralFshSrc.size != 0 )
                {
                    shaderCode.CopyCode( mShaderCodes->mGeneralFshSrc.buf,
                                         mShaderCodes->mGeneralFshSrc.size );
                }
            }
            else
            {
                const ShaderCodeParam &shaderCodeParam =
                    mShaderCodes->mCustomFshSrc[customShaderIndex - 1];

                if ( shaderCodeParam.buf  != NULL &&
                     shaderCodeParam.size != 0 )
                {
                    shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
                }
            }
        }
    }

    if ( expandAllShaders == true && useStreamOut == false)
    {
        const char endifPreprocessor[] = "#endif\n";

        for ( int i = 0; i < 8; i++)
        {
            const ShaderCodeParam &shaderCodeParam = mShaderCodes->mReservedFshSrc[i];

            if ( shaderCodeParam.buf != NULL &&
                shaderCodeParam.size != 0 )
            {
                char shaderMacro[256];
                sprintf_s(shaderMacro, sizeof(shaderMacro), "#if RESERVED_SHADER_%d_CONVERTER\n", i);
                shaderCode.CopyCode( shaderMacro, static_cast<u32>(strlen(shaderMacro)) );
                shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
                shaderCode.CopyCode( endifPreprocessor, static_cast<u32>(strlen(endifPreprocessor)) );
            }
        }
    }
    else
    {
        // Reserved shaderのコピー
        if ( reservedShaderIndex >= 0 )
        {
            const ShaderCodeParam &shaderCodeParam =
                mShaderCodes->mReservedFshSrc[reservedShaderIndex];

            if ( shaderCodeParam.buf  != NULL &&
                 shaderCodeParam.size != 0 )
            {
                shaderCode.CopyCode( shaderCodeParam.buf, shaderCodeParam.size );
            }
        }
    }

    if ( useStreamOut )
    {
        char nonMainFshShaderCode[] = "void main(){}\n";
        shaderCode.CopyCode( nonMainFshShaderCode,
                             (u32)strlen( nonMainFshShaderCode ) );

    }
    else if ( overrideFragmentShader && overrideFragmentShaderLength != 0 )
    {
        shaderCode.CopyCode( mShaderCodes->mFragShaderSrc.buf,
                             mShaderCodes->mFragShaderSrc.size );
        // CombinerShaderのコピー
        shaderCode.CopyCode( overrideFragmentShader, overrideFragmentShaderLength );
    }
    else
    {
        shaderCode.CopyCode( mShaderCodes->mFragShaderSrc.buf,
                             mShaderCodes->mFragShaderSrc.size );
    }

    return  shaderCode.GetShaderSize();
}

} // ShaderConverter
} // EffectMaker

