﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShaderCodeGeneratorGeneric_ShaderCompileError_H__
#define __ShaderCodeGeneratorGeneric_ShaderCompileError_H__


namespace EffectMaker {
namespace ShaderCodeGeneratorGeneric {

    /// <summary>
    /// Holds a list of error information when shader compile failed.
    /// </summary>
    class LIB_API ShaderCompileErrorList
    {
    private : // Private structure
        /// <summary>
        /// Holds the error information.
        /// </summary>
        struct ShaderCompileError
        {
        public : // Public fields
            char *szEmitterName;
            char *szVertexShader;
            char *szFragmentShader;
            char *szErrorLog;
            bool  isWarning;

        public : // Public methods
            /// <summary>
            /// Default constructor.
            /// </summary>
            ShaderCompileError() :
                szVertexShader(NULL),
                szFragmentShader(NULL),
                szErrorLog(NULL),
                isWarning(false)
            {
            };

            /// <summary>
            /// Set the shaders and the warning log when compile warning occurs.
            /// </summary>
            /// <param name="szEmitterName">The emitter name for the shader.</param>
            /// <param name="szVertexShader">The vertex shader.</param>
            /// <param name="szFragmentShader">The fragment shader.</param>
            /// <param name="szErrorLog">The error log.</param>
            void SetWarningLog( const char *szEmitterName,
                                const char *szVertexShader,
                                const char *szFragmentShader,
                                const char *szErrorLog )
            {
                // This is only a warning.
                this->isWarning = true;
                this->SetLog(szEmitterName, szVertexShader, szFragmentShader, szErrorLog);
            };

            /// <summary>
            /// Set the shaders and the error log when compile error occurs.
            /// </summary>
            /// <param name="szEmitterName">The emitter name for the shader.</param>
            /// <param name="szVertexShader">The vertex shader.</param>
            /// <param name="szFragmentShader">The fragment shader.</param>
            /// <param name="szErrorLog">The error log.</param>
            void SetErrorLog( const char *szEmitterName,
                              const char *szVertexShader,
                              const char *szFragmentShader,
                              const char *szErrorLog )
            {
                // There is an error occurred.
                this->isWarning = false;
                this->SetLog(szEmitterName, szVertexShader, szFragmentShader, szErrorLog);
            };

            /// <summary>
            /// Clear the allocated memory.
            /// </summary>
            void Clear()
            {
                // Release the allocated memory.
                if (this->szEmitterName != NULL)
                {
                    delete[] this->szEmitterName;
                    this->szEmitterName = NULL;
                }

                if (this->szVertexShader != NULL)
                {
                    delete[] this->szVertexShader;
                    this->szVertexShader = NULL;
                }

                if (this->szFragmentShader != NULL)
                {
                    delete[] this->szFragmentShader;
                    this->szFragmentShader = NULL;
                }

                if (this->szErrorLog != NULL)
                {
                    delete[] this->szErrorLog;
                    this->szErrorLog = NULL;
                }
            };

        private :
            /// <summary>
            /// Set the shaders and the warning log.
            /// </summary>
            /// <param name="szEmitterName">The emitter name for the shader.</param>
            /// <param name="szVertexShader">The vertex shader.</param>
            /// <param name="szFragmentShader">The fragment shader.</param>
            /// <param name="szErrorLog">The error log.</param>
            void SetLog( const char *szEmitterName,
                         const char *szVertexShader,
                         const char *szFragmentShader,
                         const char *szErrorLog )
            {
                size_t len;

                // Copy the emitter name.
                if (szEmitterName == NULL)
                {
                    this->szEmitterName = new char[1];
                    this->szEmitterName[0] = '\0';
                }
                else
                {
                    len = strlen(szEmitterName);
                    this->szEmitterName = new char[len + 1];
                    memcpy_s(this->szEmitterName, len + 1, szEmitterName, len + 1);
                }

                // Copy the vertex shader code.
                if (szVertexShader == NULL)
                {
                    this->szVertexShader = new char[1];
                    this->szVertexShader[0] = '\0';
                }
                else
                {
                    len = strlen(szVertexShader);
                    this->szVertexShader = new char[len + 1];
                    memcpy_s(this->szVertexShader, len + 1, szVertexShader, len + 1);
                }

                // Copy the fragment shader code.
                if (szFragmentShader == NULL)
                {
                    this->szFragmentShader = new char[1];
                    this->szFragmentShader[0] = '\0';
                }
                else
                {
                    len = strlen(szFragmentShader);
                    this->szFragmentShader = new char[len + 1];
                    memcpy_s(this->szFragmentShader, len + 1, szFragmentShader, len + 1);
                }

                // Copy the error log.
                if (szErrorLog == NULL)
                {
                    this->szErrorLog = new char[1];
                    this->szErrorLog[0] = '\0';
                }
                else
                {
                    len = strlen(szErrorLog);
                    this->szErrorLog = new char[len + 1];
                    memcpy_s(this->szErrorLog, len + 1, szErrorLog, len + 1);
                }
            };
        };

    public :
        /// <summary>
        /// Constructor.
        /// </summary>
        ShaderCompileErrorList() :
            listSize(0),
            pList(NULL),
            currIndex(0)
        {
        };

        /// <summary>
        /// Destructor.
        /// </summary>
        virtual ~ShaderCompileErrorList()
        {
            // Clear the memory allocated by the list items.
            for (int i = 0; i < this->listSize; ++i)
            {
                this->pList[i].Clear();
            }

            // Clear the list.
            delete[] this->pList;
            this->pList = NULL;
        };

        /// <summary>
        /// Add the shaders and log messages when compile warning occurs.
        /// </summary>
        /// <param name="szEmitterName">The emitter name for the shader.</param>
        /// <param name="szVertexShader">The vertex shader.</param>
        /// <param name="szFragmentShader">The fragment shader.</param>
        /// <param name="szErrorLog">The warning log.</param>
        void AddWarningLog( const char *szEmitterName,
                            const char *szVertexShader,
                            const char *szFragmentShader,
                            const char *szErrorLog )
        {
            // Allocate memory for the new list.
            ShaderCompileError *pNewList = new ShaderCompileError[this->listSize + 1];

            // Copy the items in the old list to the newly allocated one.
            if (this->listSize > 0 && this->pList != NULL)
            {
                const size_t ItemSize = sizeof(ShaderCompileError);

                size_t oldSize = ItemSize * this->listSize;
                size_t newSize = oldSize + ItemSize;

                memcpy_s(pNewList, newSize, this->pList, oldSize);

                // Release the memory the old list occupies.
                delete[] this->pList;
            }

            // Set up the new item.
            pNewList[this->listSize].SetWarningLog(szEmitterName,
                                                   szVertexShader,
                                                   szFragmentShader,
                                                   szErrorLog);

            // Save the new list.
            this->pList = pNewList;

            // Increment the counter.
            ++this->listSize;
        };

        /// <summary>
        /// Add the shaders and the error log when compile error occurs.
        /// </summary>
        /// <param name="szEmitterName">The emitter name for the shader.</param>
        /// <param name="szVertexShader">The vertex shader.</param>
        /// <param name="szFragmentShader">The fragment shader.</param>
        /// <param name="szErrorLog">The error log.</param>
        void AddErrorLog( const char *szEmitterName,
                          const char *szVertexShader,
                          const char *szFragmentShader,
                          const char *szErrorLog )
        {
            // Allocate memory for the new list.
            ShaderCompileError *pNewList = new ShaderCompileError[this->listSize + 1];

            // Copy the items in the old list to the newly allocated one.
            if (this->listSize > 0 && this->pList != NULL)
            {
                const size_t ItemSize = sizeof(ShaderCompileError);

                size_t oldSize = ItemSize * this->listSize;
                size_t newSize = oldSize + ItemSize;

                memcpy_s(pNewList, newSize, this->pList, oldSize);

                // Release the memory the old list occupies.
                delete[] this->pList;
            }

            // Set up the new item.
            pNewList[this->listSize].SetErrorLog(szEmitterName,
                                                 szVertexShader,
                                                 szFragmentShader,
                                                 szErrorLog);

            // Save the new list.
            this->pList = pNewList;

            // Increment the counter.
            ++this->listSize;
        };

        /// <summary>
        /// Reset the enumeration head.
        /// After this, GetNextItem() returns the first item.
        /// </summary>
        void BeginEnumeration()
        {
            this->currIndex = 0;
        };

        /// <summary>
        /// Get the next error information.
        /// This method returns the first item after BeginEnumeration() is called,
        /// then the second, the third, and so forth.
        /// </summary>
        /// <param name="pIsWarning">A pointer to the flag indicating if the log is just a warning.</param>
        /// <param name="szEmitterName">The emitter name for the shader.</param>
        /// <param name="szVertexShader">The vertex shader.</param>
        /// <param name="szFragmentShader">The fragment shader.</param>
        /// <param name="szErrorLog">The error log.</param>
        bool GetNextItem( bool *pIsWarning,
                          const char * &szEmitterName,
                          const char * &szVertexShader,
                          const char * &szFragmentShader,
                          const char * &szErrorLog )
        {
            if (this->currIndex < 0 || this->currIndex >= this->listSize)
            {
                return false;
            }

            const ShaderCompileError &item = this->pList[this->currIndex];

            *pIsWarning      = item.isWarning;
            szEmitterName    = item.szEmitterName;
            szVertexShader   = item.szVertexShader;
            szFragmentShader = item.szFragmentShader;
            szErrorLog       = item.szErrorLog;

            ++this->currIndex;

            return true;
        };

    private :
        int                 listSize;
        ShaderCompileError *pList;

        int                 currIndex;
    };

} // namespace ShaderCodeGeneratorGeneric
} // namespace EffectMaker

#endif // __ShaderCodeGeneratorGeneric_ShaderCompileError_H__
