﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.EffectCombinerEditor;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.EffectCombinerBridge.Properties;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.EffectCombinerBridge.EffectCombinerBridge
{
    /// <summary>
    /// Manager class for the communications between EffectCombinerEditor and EffectMaker.
    /// </summary>
    class EffectCombinerCommunicationBridge : IEffectCombinerCommunicationBridge
    {
        /// <summary>
        /// A dictionary stores the combiner editor project paths and
        /// the generated shader source codes.
        /// </summary>
        private Dictionary<string, string> GeneratedShaderCodes = new Dictionary<string, string>();

        /// <summary>
        /// A dictionary stores the shader source codes received from connected effect combiner.
        /// </summary>
        private Dictionary<string, string> ReceivedShaderCodes = new Dictionary<string, string>();

        /// <summary>The communication task.</summary>
        private Task runningTask;

        /// <summary>The communications status.</summary>
        private EffectCombinerCommunicationStatus status = EffectCombinerCommunicationStatus.Stopped;

        /// <summary>The communication server.</summary>
        private Communication.Core.CommunicationServer<Communication.Contracts.IShaderTransferService> communicationServer = null;

        /// <summary>The effect combiner assembly.</summary>
        private Assembly combinerAssembly = null;

        /// <summary>The type information for ShaderGenSettings class.</summary>
        private Type shaderGenSettingsType = null;

        /// <summary>The type information for PathContainer class.</summary>
        private Type pathContainerType = null;

        /// <summary>The type information for ShaderGen class.</summary>
        private Type shaderGenType = null;

        /// <summary>The information for PathContainer.Add method.</summary>
        private MethodInfo pathContainerAddMethod = null;

        /// <summary>The information for ShaderGen.Initialize method.</summary>
        private MethodInfo shaderGenInitializeMethod = null;

        /// <summary>The information for ShaderGen.ProduceShaderCodes method.</summary>
        private MethodInfo shaderGenProduceShaderCodesMethod = null;

        /// <summary>The information for ShaderGenSettings.EffectDefinitionsPaths property.</summary>
        private PropertyInfo shaderGenSettingsEffectDefinitionsPathsProperty = null;

        /// <summary>
        /// Event fired when communication status changed.
        /// </summary>
        public event EventHandler<EffectCombinerCommunicationStatusEventArgs> StatusChanged;

        /// <summary>
        /// Event fired when shader code changed.
        /// </summary>
        public event EventHandler<EffectCombinerShaderChangedEventArgs> ShaderChanged;

        /// <summary>
        /// Get the current communication status.
        /// </summary>
        public EffectCombinerCommunicationStatus Status
        {
            get
            {
                return status;
            }

            private set
            {
                if (status == value)
                {
                    return;
                }

                var oldStatus = status;
                status = value;

                OnStatusChanged(
                    new EffectCombinerCommunicationStatusEventArgs(oldStatus, status));
            }
        }

        /// <summary>
        /// Get the error while communication.
        /// </summary>
        public Exception Error { get; private set; }

        /// <summary>
        /// Initialize.
        /// </summary>
        public void Init()
        {
            if (OptionStore.RuntimeOptions.IsCommandLineMode == true)
            {
                return;
            }

            if (runningTask != null)
            {
                return;
            }

            // Create the communication server.
            communicationServer = new Communication.Core.CommunicationServer<Communication.Contracts.IShaderTransferService>();

            runningTask = communicationServer.Start(new EffectCombinerShaderTransferService(this))
                .ContinueWith(t =>
                {
                    if (t.IsFaulted)
                    {
                        Error = t.Exception;
                        Status = EffectCombinerCommunicationStatus.Failed;
                    }
                    else
                    {
                        Error = null;
                        Status = EffectCombinerCommunicationStatus.Running;
                    }
                });
        }

        /// <summary>
        /// Deinitialize.
        /// </summary>
        public void Deinit()
        {
            // Stop the communication server.
            if (Status == EffectCombinerCommunicationStatus.Running)
            {
                communicationServer.Stop();
            }

            Error = null;
            Status = EffectCombinerCommunicationStatus.Stopped;

            runningTask = null;
        }

        /// <summary>
        /// Set shader source code that is received from connected effect combiner editor.
        /// </summary>
        /// <param name="combinerProjectPath">The combiner editor project file path.</param>
        /// <param name="shaderSourceCode">The generated shader code.</param>
        public void SetReceivedShaderSourceCode(
            string combinerProjectPath,
            string shaderSourceCode)
        {
            ReceivedShaderCodes[combinerProjectPath] = shaderSourceCode;
            if (ShaderChanged != null)
            {
                ShaderChanged(
                    null,
                    new EffectCombinerShaderChangedEventArgs(combinerProjectPath));
            }
        }

        /// <summary>
        /// 指定したパスのキャッシュを破棄して、もう一度ファイルからシェーダが生成されるようにします。
        /// </summary>
        /// <param name="combinerProjectPath">コンバイナシェーダのファイルパス</param>
        public void DiscardCache(string combinerProjectPath)
        {
            if (ReceivedShaderCodes.ContainsKey(combinerProjectPath))
            {
                ReceivedShaderCodes.Remove(combinerProjectPath);
            }

            if (GeneratedShaderCodes.ContainsKey(combinerProjectPath))
            {
                GeneratedShaderCodes.Remove(combinerProjectPath);
            }
        }

        /// <summary>
        /// Clear all the shader source code received from connected effect combiner editor.
        /// </summary>
        public void ClearReceivedShaderSourceCode()
        {
            var combinerProjectPaths = ReceivedShaderCodes.Keys.ToArray();
            ReceivedShaderCodes.Clear();
            if (ShaderChanged != null)
            {
                foreach (string path in combinerProjectPaths)
                {
                    ShaderChanged(
                        null,
                        new EffectCombinerShaderChangedEventArgs(path));
                }
            }
        }

        /// <summary>
        /// Get the shader source code that is generated from
        /// the given combiner editor project.
        /// </summary>
        /// <param name="combinerProjectPath">The combiner editor project file path.</param>
        /// <returns>The shader source code if found, or empty string is returned.</returns>
        public string GetShaderSourceCode(string combinerProjectPath)
        {
            string shader;
            if (ReceivedShaderCodes.TryGetValue(combinerProjectPath, out shader) == false)
            {
                if (GeneratedShaderCodes.TryGetValue(combinerProjectPath, out shader) == false)
                {
                    // The shader has not been generated yet, try to generate it.
                    shader = GenerateShaderFromProject(combinerProjectPath);
                    if (string.IsNullOrEmpty(shader) == true)
                    {
                        // The shader generation is failed.
                        return string.Empty;
                    }

                    // キャッシュに追加
                    GeneratedShaderCodes[combinerProjectPath] = shader;
                }
            }

            return shader;
        }

        /// <summary>
        /// Initialize the effect combiner library for generating shader code from
        /// combiner editor project.
        /// </summary>
        /// <returns>True on success.</returns>
        private bool InitializeEffectCombinerLibrary()
        {
            // Compose the effect combiner library path.
            string libraryPath = Path.Combine(
                IOConstants.CoreModulesPath,
                "EffectCombiner.Generator.dll");

            if (File.Exists(libraryPath) == false)
            {
                Logger.Log(LogLevels.Error, "The EffectCombiner library cannot be found at " + libraryPath);
                return false;
            }

            // Load the library assembly.
            try
            {
                combinerAssembly = Assembly.LoadFrom(libraryPath);
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "Failed loading EffectCombiner library.");
                Logger.Log(LogLevels.Error, ex.Message);
                return false;
            }

            // Find the necessary types in the library.
            shaderGenSettingsType =
                combinerAssembly.GetType("EffectCombiner.Generator.ShaderGenSettings");

            pathContainerType =
                combinerAssembly.GetType("EffectCombiner.Generator.PathContainer");

            shaderGenType =
                combinerAssembly.GetType("EffectCombiner.Generator.ShaderGen");

            // Find the method and property information.
            pathContainerAddMethod = pathContainerType.GetMethod(
                "Add",
                new Type[] { typeof(string), typeof(SearchOption), typeof(string[]) });

            shaderGenInitializeMethod =
                shaderGenType.GetMethod("Initialize");

            shaderGenProduceShaderCodesMethod =
                shaderGenType.GetMethod("ProduceShaderCodes");

            shaderGenSettingsEffectDefinitionsPathsProperty =
                shaderGenSettingsType.GetProperty("EffectDefinitionsPaths");

            return true;
        }

        /// <summary>
        /// Generate shader source code from the specified combiner editor project.
        /// </summary>
        /// <param name="combinerProjectPath">The combiner editor project file path.</param>
        /// <returns>The shader source code if successful, or empty string is returned.</returns>
        private string GenerateShaderFromProject(string combinerProjectPath)
        {
            if (combinerAssembly == null)
            {
                // Initialize EffectCombiner library for shader generation.
                if (InitializeEffectCombinerLibrary() == false)
                {
                    return string.Empty;
                }
            }

            // エフェクトコンバイナの定義がロードされていない時は生成を止める
            if (!OptionStore.ProjectConfig.IsEftCombinerEditorEnabled)
            {
                Logger.Log("LogView,Console", LogLevels.Warning, Resources.WarningEffectCombinerDefinitionNotDefined);
                return string.Empty;
            }

            // Create setting for shader generation.
            object settings = Activator.CreateInstance(shaderGenSettingsType);

            // Get effect definition paths.
            object pathContainer =
                shaderGenSettingsEffectDefinitionsPathsProperty.GetValue(settings);

            // Add the shader block definition path for shader generation.
            foreach (string path in OptionStore.ProjectConfig.EffectCombinerBlockDefinitionPaths)
            {
                string folderPath = PathUtility.ToAbsolutePath(path, Path.GetDirectoryName(OptionStore.ProjectConfig.ConfigFilePath));
                if (!Directory.Exists(folderPath))
                {
                    Logger.Log(
                        "LogView,Console",
                        LogLevels.Warning,
                        Resources.WarningEffectCombinerDefinitionNotFound,
                        folderPath);
                    return string.Empty;
                }

                pathContainerAddMethod.Invoke(
                    pathContainer,
                    new object[] { folderPath, SearchOption.AllDirectories, new string[0] });
            }

            // Create the shader generator.
            object shaderGen = Activator.CreateInstance(shaderGenType, settings);

            // Initialize the shader generator.
            shaderGenInitializeMethod.Invoke(shaderGen, null);

            object result;
            try
            {
                // Generate the shader codes.
                result = shaderGenProduceShaderCodesMethod.Invoke(
                    shaderGen,
                    new object[] { combinerProjectPath, "main" });
            }
            catch (Exception e)
            {
                Logger.Log("LogView,Console", LogLevels.Warning, "Failed generating shader from EffectCombiner project " + combinerProjectPath);
                Logger.LogException(e);
                return string.Empty;
            }

            var codes = result as string[];
            if (codes == null || codes.Length <= 0)
            {
                Logger.Log("LogView,Console", LogLevels.Warning, "The EffectCombiner did not generate any shader from " + combinerProjectPath);
                return string.Empty;
            }

            // Only return the first shader code.
            return codes[0];
        }

        /// <summary>
        /// Helper method for firing communication status changed event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        private void OnStatusChanged(
            EffectCombinerCommunicationStatusEventArgs e)
        {
            var handler = StatusChanged;
            if (handler != null)
            {
                handler(null, e);
            }
        }
    }
}
