﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.Specifics.Behaviors;
using EffectMaker.UIDialogs.MessageDialogs;
using EffectMaker.UILogic.Manager;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// User basic options.
    /// </summary>
    [DisplayOrder(0)]
    public partial class UserBasicOptionPane : UserControl, IOptionPane
    {
        /// <summary>
        /// Initializes the UserBasicOptionPane instance.
        /// </summary>
        public UserBasicOptionPane()
        {
            this.InitializeComponent();

            this.chkUndoUnlimited.CheckedChanged += this.OnUndoUnlimitedCheckBoxCheckedChanged;

            this.btnWorkspaceFilesAutoLoadClear.Click += this.OnWorkspaceFilesAutoLoadClearButtonClick;
            this.btnWorkspaceFilesAutoLoadLoad.Click += this.OnWorkspaceFilesAutoLoadLoadButtonClick;

            this.btnPresetLoad.Click += this.OnPresetLoadButtonClick;
            this.btnPresetReset.Click += this.OnPresetClearButtonClick;

            this.btnDocumentLoad.Click += this.OnDocumentLoadButtonClick;
            this.btnDocumentReset.Click += this.OnDocumentClearButtonClick;

            OptionUtil.SetCommentLabelState(this.lblPresetFolderComment);

            OptionUtil.SetCommentLabelState(this.lblDocumentFolderComment);

            this.btnWorkspaceFilesAutoLoadClear.Enabled = false;

            // ワークスペースのファイルパスが変更された場合は、ファイルの有無をチェックして、ツリービューを更新する.
            this.txtWorkspaceFilesAutoLoad.FilePathChanged += (s, e) =>
            {
                this.btnWorkspaceFilesAutoLoadClear.Enabled = !string.IsNullOrEmpty(this.txtWorkspaceFilesAutoLoad.FilePath);
                OptionUtil.OnCheckFilePathExisting(s, e);
                if (this.DrawTreeView != null)
                {
                    this.DrawTreeView();
                }
            };

            this.btnPresetReset.Enabled = false;

            // プリセットのフォルダパスが変更された場合は、フォルダの有無をチェックして、ツリービューを更新する.
            this.txtPresetFolder.FilePathChanged += (s, e) =>
            {
                this.btnPresetReset.Enabled = !string.Equals(this.txtPresetFolder.FilePath, DefaultPresetFolderPath);
                OptionUtil.OnCheckFilePathExisting(s, e);
                if (this.DrawTreeView != null)
                {
                    this.DrawTreeView();
                }
            };

            // ドキュメントパスが変更された場合は、フォルダの有無をチェックして、ツリービューを更新する.
            this.txtDocumentFolder.FilePathChanged += (s, e) =>
            {
                this.btnDocumentReset.Enabled = !string.Equals(this.txtDocumentFolder.FilePath, IOConstants.DefaultDocumentPath);
                OptionUtil.OnCheckFilePathExisting(s, e);
                if (this.DrawTreeView != null)
                {
                    this.DrawTreeView();
                }
            };

            this.txtWorkspaceFilesAutoLoad.BackColor = Color.White;
            this.txtPresetFolder.BackColor = Color.White;
            this.txtDocumentFolder.BackColor = Color.White;
        }

        /// <summary>
        /// デフォルトのプリセットフォルダパスを返します。
        /// </summary>
        public static string DefaultPresetFolderPath
        {
            get
            {
                return Path.Combine(
                    Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location),
                    "Presets");
            }
        }

        /// <summary>
        /// オプションウィンドウのTreeViewを再描画するdelegate
        /// </summary>
        public Action DrawTreeView { get;　set; }

        /// <summary>
        /// プロジェクトコンフィグの表示内容を更新するdelegate
        /// </summary>
        public Action TriggerProjectSetting { get; set; }
　
        /// <summary>
        /// Gets the option pane identifier.
        /// </summary>
        public string Identifier
        {
            get { return "UserConfig"; }
        }

        /// <summary>
        /// Gets the identifier of the parent, if any.
        /// </summary>
        public string ChildOf
        {
            get { return null; }
        }

        /// <summary>
        /// Gets the display text of the option pane.
        /// </summary>
        public string DisplayName
        {
            get { return Properties.Resources.OptionPaneUserBasic; } // TODO: make it translatable
        }

        /// <summary>
        /// テキストボックスへのドラッグアンドドロップ時の挙動について設定します.
        /// </summary>
        public void InitializeDragAndDrop()
        {
            // 起動時に読み込むワークスペースファイルのドラッグ&ドロップ処理を設定
            {
                // ワークスペースのファイル以外はドラッグ&ドロップできないようにする
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowExtensions = RegexUtility.ExtractExtensions(Properties.Resources.OptionDialogFilterWorkspaceFiles);
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    // ワークスペースのxmlファイルであるかチェックする.
                    if (!IOConstants.CheckRootElement(dragDropBehavior.DropResult, "WorkspaceStorageData"))
                    {
                        ThreadSafeMsgBox.Show(
                            Resources.InvalidSettingFileWorkspace,
                            Resources.WarningCaption,
                            MessageBoxButtons.OK,
                            MessageBoxIcon.Warning);
                    }
                    else
                    {
                        txtWorkspaceFilesAutoLoad.FilePath = dragDropBehavior.DropResult;
                    }
                };

                this.txtWorkspaceFilesAutoLoad.Behaviors.Add(dragDropBehavior);
            }

            // プリセットフォルダのドラッグ&ドロップ処理を設定
            {
                // 全てのファイルとディレクトリのドラッグ&ドロップを許可する
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowAllExtensions = true;
                dragDropBehavior.AllowDirectory = true;
                dragDropBehavior.ConvertToDirectoryPath = true;
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    this.txtPresetFolder.FilePath = dragDropBehavior.DropResult;
                };

                this.txtPresetFolder.Behaviors.Add(dragDropBehavior);
            }

            // ドキュメントファイルのドラッグ&ドロップ処理を設定
            {
                // ドキュメントパスは、フォルダとhtmlファイルのみドラッグ&ドロップを許可する
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowExtensions = RegexUtility.ExtractExtensions(Properties.Resources.OptionDialogFilterDocumentPath);
                dragDropBehavior.AllowDirectory = true;
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    this.txtDocumentFolder.FilePath = dragDropBehavior.DropResult;
                };

                this.txtDocumentFolder.Behaviors.Add(dragDropBehavior);
            }
        }

        /// <summary>
        /// Called when initialized.
        /// </summary>
        public void OnInitialize()
        {
            BasicOptions options = OptionStore.RootOptions.Basic;

            this.chkUndoUnlimited.Checked = options.Undo.IsEnabled;
            this.numUndoMaximum.Value = options.Undo.MaximumStackItems;

            // ドラッグアンドドロップ時の設定
            this.InitializeDragAndDrop();

            // プリセットフォルダ
            this.txtPresetFolder.FilePath = OptionStore.RootOptions.FileEvent.PresetFolderPath;

            // ドキュメント起動パス
            this.txtDocumentFolder.FilePath = OptionStore.RootOptions.Basic.DocumentPath;

            // 起動時に読み込むワークスペース
            this.txtWorkspaceFilesAutoLoad.FilePath = options.WorkspaceFilesAutoLoad;

            OptionalMsgBoxSetting setting;
            var settingMap = OptionStore.RootOptions.Interface.OptionalMsgBoxSettings;

            // 探索範囲外のテクスチャを読み込んだ時の警告On/Off
            if (settingMap.TryGetValue("WarningOnLoadTexture", out setting) == false)
            {
                setting = new OptionalMsgBoxSetting() { IsEnabled = true, DialogResult = DialogResult.OK };
                settingMap.Add("WarningOnLoadTexture", setting);
            }

            this.chkOptionsEmittersShowWarningIconLinearTexture.Checked = options.Emitters.ShowWarningIconLinearTexture;

            this.chkOptionsTextureShowWarningOnLoad.Checked = setting.IsEnabled;
            this.chkOptionsTextureAutoReloadWhenChanged.Checked = options.Textures.AutoReload;

            // リニア編集モードの設定と異なるテクスチャを読み込んだ時に警告On/Off
            if (settingMap.TryGetValue("LinearEditTextureWarning", out setting) == false)
            {
                setting = new OptionalMsgBoxSetting() { IsEnabled = true, DialogResult = DialogResult.OK };
                settingMap.Add("LinearEditTextureWarning", setting);
            }

            this.chkOptionsWarningLoadingTextureDifferentMode.Checked = setting.IsEnabled;

            // テクスチャが未設定のエミッタセットを保存する時に警告を表示On/Off
            if (settingMap.TryGetValue("WarningTextureIsNotAssigned", out setting) == false)
            {
                setting = new OptionalMsgBoxSetting() { IsEnabled = true, DialogResult = DialogResult.OK };
                settingMap.Add("WarningTextureIsNotAssigned", setting);
            }

            this.chkOptionsWarningTextureIsNotAssigned.Checked = setting.IsEnabled;

            // 探索範囲外のプリミティブを読み込んだ時の警告On/Off
            if (settingMap.TryGetValue("WarningOnLoadPrimitive", out setting) == false)
            {
                setting = new OptionalMsgBoxSetting() { IsEnabled = true, DialogResult = DialogResult.OK };
                settingMap.Add("WarningOnLoadPrimitive", setting);
            }

            this.chkOptionsPrimitivesShowWarningOnLoad.Checked = setting.IsEnabled;

            this.chkOptionsPrimitivesAutoReloadWhenChanged.Checked = options.Primitives.AutoReload;

            this.chkOptionsPreviewIncludeSavingTargetOnAllSave.Checked = options.Preview.IncludeAllSave;

            ////this.chkCombinerIsEnable.Checked = options.IsEftCombinerEditorEnabled;
            ////OptionUtil.IsEnableEffectCombinerEditor = options.IsEftCombinerEditorEnabled;

            ////if (BusinessLogic.EffectCombinerEditor.EffectCombinerCommunicationManager.CommunicationBridge != null &&
            ////    Directory.EnumerateFiles(IOConstants.ExecutableFolderPath, "UseEffectCombinerEditor.*").Any())
            ////{
            ////    // チェックボックスのON/OFFでコンバイナシェーダ定義フォルダUIがON/OFFする仕込み
            ////    this.chkCombinerIsEnable.CheckedChanged += (sender, args) =>
            ////    {
            ////        OptionUtil.IsEnableEffectCombinerEditor = this.chkCombinerIsEnable.Checked;
            ////        if (this.TriggerProjectSetting != null)
            ////        {
            ////            this.TriggerProjectSetting();
            ////        }
            ////    };
            ////}
            ////else
            {
                // コンバイナ連携に関するUIを隠す（コメントアウトによる対処）
                this.flowLayoutPanel1.Controls.Remove(this.lblCombiner);
                this.flowLayoutPanel1.Controls.Remove(this.chkCombinerIsEnable);

                int collapsedHegiht =
                    this.lblCombiner.Height + this.lblCombiner.Margin.Top +
                    this.chkCombinerIsEnable.Height + this.chkCombinerIsEnable.Margin.Top;
                this.grpOptions.Height -= collapsedHegiht;
                this.Height -= collapsedHegiht;

                this.grpPresetFolder.Location = new System.Drawing.Point(this.grpPresetFolder.Location.X,
                                                                         this.grpPresetFolder.Location.Y - collapsedHegiht);
                this.grpDocumentPath.Location = new System.Drawing.Point(this.grpDocumentPath.Location.X,
                                                                         this.grpDocumentPath.Location.Y - collapsedHegiht);
            }
        }

        /// <summary>
        /// Called when terminated.
        /// </summary>
        public void OnTerminate()
        {
        }

        /// <summary>
        /// Called when the Accept button is clicked.
        /// </summary>
        /// <returns>入力に不正がなければtrue,入力が不正であればfalse.</returns>
        public bool OnAccept()
        {
            BasicOptions options = OptionStore.RootOptions.Basic;

            // コマンド上限を変更したか?
            if (options.Undo.MaximumStackItems != (int)this.numUndoMaximum.Value || options.Undo.IsEnabled != this.chkUndoUnlimited.Checked)
            {
                EffectMaker.UIDialogs.MessageDialogs.ThreadSafeMsgBox.Show(Properties.Resources.OptionUndoMessage, Properties.Resources.WarningCaption, MessageBoxButtons.OK);
            }

            OptionStore.RootOptions.FileEvent.PresetFolderPath = this.txtPresetFolder.FilePath;
            options.DocumentPath = this.txtDocumentFolder.FilePath;

            options.Undo.IsEnabled = this.chkUndoUnlimited.Checked;
            options.Undo.MaximumStackItems = (int)this.numUndoMaximum.Value;

            options.WorkspaceFilesAutoLoad = this.txtWorkspaceFilesAutoLoad.FilePath;

            var settingMap = OptionStore.RootOptions.Interface.OptionalMsgBoxSettings;

            // エミッタ
            settingMap["WarningTextureIsNotAssigned"].IsEnabled = this.chkOptionsWarningTextureIsNotAssigned.Checked;
            options.Emitters.ShowWarningIconLinearTexture = this.chkOptionsEmittersShowWarningIconLinearTexture.Checked;

            // テクスチャ
            settingMap["WarningOnLoadTexture"].IsEnabled = this.chkOptionsTextureShowWarningOnLoad.Checked;
            options.Textures.AutoReload = this.chkOptionsTextureAutoReloadWhenChanged.Checked;
            settingMap["LinearEditTextureWarning"].IsEnabled = this.chkOptionsWarningLoadingTextureDifferentMode.Checked;

            // プリミティブ
            settingMap["WarningOnLoadPrimitive"].IsEnabled = this.chkOptionsPrimitivesShowWarningOnLoad.Checked;
            options.Primitives.AutoReload = this.chkOptionsPrimitivesAutoReloadWhenChanged.Checked;

            // プレビュー
            options.Preview.IncludeAllSave = this.chkOptionsPreviewIncludeSavingTargetOnAllSave.Checked;

            ////options.IsEftCombinerEditorEnabled = this.chkCombinerIsEnable.Checked;

            return true;
        }

        /// <summary>
        /// Called when the Cancel button is clicked.
        /// </summary>
        public void OnCancel()
        {
        }

        /// <summary>
        /// ユーザーコンフィグが読み込まれた際に、その内容にUIを更新します。
        /// </summary>
        public void OnProjectSettingChanged()
        {
        }

        /// <summary>
        /// パス設定項目をチェックします。
        /// </summary>
        /// <returns>
        /// 有効ならtrue、無効ならfalse.
        /// </returns>
        public bool OnValidationCheck()
        {
            return this.txtWorkspaceFilesAutoLoad.BackColor == Color.White && this.txtPresetFolder.BackColor == Color.White
                && (this.txtDocumentFolder.FilePath == IOConstants.DefaultDocumentPath || this.txtDocumentFolder.BackColor == Color.White);
        }

        /// <summary>
        /// プロジェクトコンフィグをファイル出力する際に各UIの内容を出力用インスタンスに収集します。
        /// </summary>
        public void OnExportProjectSetting()
        {
        }

        /// <summary>
        /// Called when the WorkspaceFilesAutoLoadLoad Button is clicked.
        /// </summary>
        /// <param name="sender">The WorkspaceFilesAutoLoadLoad Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnWorkspaceFilesAutoLoadLoadButtonClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterWorkspaceFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
            };

            if (dlg.ShowDialog(this) != DialogResult.OK)
            {
                return;
            }

            if (!IOConstants.CheckRootElement(dlg.FileName, "WorkspaceStorageData"))
            {
                ThreadSafeMsgBox.Show(
                    Resources.InvalidSettingFileWorkspace,
                    Resources.WarningCaption,
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Warning);
                return;
            }

            this.txtWorkspaceFilesAutoLoad.FilePath = dlg.FileName;
        }

        /// <summary>
        /// Called when the WorkspaceFilesAutoLoadClear Button is clicked.
        /// </summary>
        /// <param name="sender">The WorkspaceFilesAutoLoadClear Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnWorkspaceFilesAutoLoadClearButtonClick(object sender, EventArgs e)
        {
            this.txtWorkspaceFilesAutoLoad.FilePath = string.Empty;
        }

        /// <summary>
        /// Called when the UndoUnlimited CheckBox is checked or unchecked.
        /// </summary>
        /// <param name="sender">The UndoUnlimited CheckBox.</param>
        /// <param name="e">Event argument.</param>
        private void OnUndoUnlimitedCheckBoxCheckedChanged(object sender, EventArgs e)
        {
            this.numUndoMaximum.Enabled = !this.chkUndoUnlimited.Checked;
        }

        /// <summary>
        /// プリセットフォルダパスの選択
        /// </summary>
        /// <param name="sender">The Preset folder Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnPresetLoadButtonClick(object sender, EventArgs e)
        {
            string originalPath = string.Empty;
            if (string.IsNullOrEmpty(this.txtPresetFolder.FilePath) == false)
            {
                originalPath =
                    System.IO.Path.GetDirectoryName(this.txtPresetFolder.FilePath + Path.DirectorySeparatorChar);
            }

            var dlg = new FolderBrowserDialog
            {
                ShowNewFolderButton = true,
                SelectedPath = originalPath,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.txtPresetFolder.FilePath = dlg.SelectedPath;
        }

        /// <summary>
        /// プリセットフォルダパスの初期化
        /// </summary>
        /// <param name="sender">The Preset folder Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnPresetClearButtonClick(object sender, EventArgs e)
        {
            this.txtPresetFolder.FilePath = DefaultPresetFolderPath;
        }

        /// <summary>
        /// ドキュメントパスの選択
        /// </summary>
        /// <param name="sender">The Preset folder Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnDocumentLoadButtonClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterDocumentPath,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
            };

            if (dlg.ShowDialog(this) != DialogResult.OK)
            {
                return;
            }

            this.txtDocumentFolder.FilePath = dlg.FileName;
        }

        /// <summary>
        /// ドキュメントパスの初期化
        /// </summary>
        /// <param name="sender">The Preset folder Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnDocumentClearButtonClick(object sender, EventArgs e)
        {
            this.txtDocumentFolder.FilePath = IOConstants.DefaultDocumentPath;
        }
    }
}
