﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.Specifics.Behaviors;
using EffectMaker.UIDialogs.MessageDialogs;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// Option panel for Emitter Extension Parameters.
    /// </summary>
    [DisplayOrder(8)]
    public partial class EmitterExtParamsOptionPane : UserControl, IOptionPane
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public EmitterExtParamsOptionPane()
        {
            this.InitializeComponent();

            this.resetButton.Enabled = false;
            this.btnEmitterExtParamsReload.Enabled = false;

            this.filePathInput.BackColor = Color.White;
            this.filePathInput.FilePathChanged += (s, e) =>
            {
                this.resetButton.Enabled = !string.Equals(this.filePathInput.FilePath, DefaultEmitterExtParamsPath);
                this.btnEmitterExtParamsReload.Enabled = !string.Equals(this.filePathInput.FilePath, DefaultEmitterExtParamsPath);
            };

            foreach (var box in this.Controls.OfType<OptionGroupBox>())
            {
                box.BorderColor = Color.FromArgb(130, 150, 185);
            }
        }

        /// <summary>
        /// デフォルトのエミッタ拡張パラメータ定義パスを返します。
        /// </summary>
        public static string DefaultEmitterExtParamsPath
        {
            get
            {
                string exeFolder = EffectMaker.BusinessLogic.IO.IOConstants.ExecutableFolderPath;
                string addinFolder = Path.Combine(exeFolder, "Addins");
                return Path.Combine(addinFolder, "EmitterExtParams.xml");
            }
        }

        /// <summary>
        /// オプションウィンドウのTreeViewを再描画するdelegate
        /// </summary>
        public Action DrawTreeView { get; set; }

        /// <summary>
        /// Unique identifier of the option pane
        /// </summary>
        public string Identifier
        {
            get { return "EmitterExtParamsOptionPane"; }
        }

        /// <summary>
        /// Set the identifier of the parent if the pane becomes a child pane.
        /// Set to null if the pane is a root pane.
        /// </summary>
        public string ChildOf
        {
            get { return "ProjectConfig"; }
        }

        /// <summary>
        /// NameWithExpander to be displayed in the user interface.
        /// </summary>
        public string DisplayName
        {
            get { return Properties.Resources.OptionPaneEmitterExtParams; }
        }

        /// <summary>
        /// Method called just after the option pane has been instanced.
        /// </summary>
        public void OnInitialize()
        {
            EffectMakerProjectConfig config = OptionStore.ProjectConfig;

            // 起動時に読み込むエミッタ拡張パラメータ
            this.filePathInput.FilePath = config.EmitterExtParamsPath;
            this.CheckAndUpdateEmitterExtParams();

            // エミッタ拡張パラメータ定義ファイルのドラッグ&ドロップ処理を設定
            {
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowExtensions = RegexUtility.ExtractExtensions(Properties.Resources.OptionDialogFilterXmlFiles);
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    this.UpdateFilePathInputTxt(dragDropBehavior.DropResult);
                };

                this.filePathInput.Behaviors.Add(dragDropBehavior);
            }

            this.btnEmitterExtParamsReload.Click += this.OnEmitterExtParamsReloadButtonClick;

            OptionUtil.SetCommentLabelState(this.lblProjectConfigOverwrite);

            OptionUtil.IsNeedReloadEmitterExtParams = !this.OnValidationCheck();
        }

        /// <summary>
        /// Method called when option window is closed.
        /// </summary>
        public void OnTerminate()
        {
            OptionUtil.IsNeedReloadEmitterExtParams = false;
        }

        /// <summary>
        /// Method called when the user click the Accept button of the options dialog.
        /// </summary>
        /// <returns>入力に不正がなければtrue,入力が不正であればfalse.</returns>
        public bool OnAccept()
        {
            if (OptionUtil.IsNeedReloadEmitterExtParams)
            {
                EffectMakerProjectConfig config = OptionStore.ProjectConfig;
                EffectMakerProjectConfig tempConfig = OptionUtil.TempConfig;

                // エミッタ拡張パラメータだけを変更したときはconfigを使用するが
                // プロジェクト設定を変更したときはtempConfigで上書きされるので両方書き換える
                config.EmitterExtParamsPath = this.filePathInput.FilePath;
                tempConfig.EmitterExtParamsPath = this.filePathInput.FilePath;

                // Load the Emitter Extension Parameters.
                var def = UserDataHelper.Instance.LoadEmitterExtParamsUserData(config.EmitterExtParamsPath, true);

                if (def == null)
                {
                    // 警告を表示する.
                    this.ShowAllWarningsInPane();

                    return false;
                }

                // Clear command stack.
                CommandManager.ClearCommandStack();
            }

            return true;
        }

        /// <summary>
        /// Method called when the user click the Cancel button of the options dialog.
        /// </summary>
        public void OnCancel()
        {
        }

        /// <summary>
        /// ユーザーコンフィグが読み込まれた際に、その内容にUIを更新します。
        /// </summary>
        public void OnProjectSettingChanged()
        {
            this.filePathInput.FilePath = OptionUtil.TempConfig.EmitterExtParamsPath;
            if (string.IsNullOrEmpty(this.filePathInput.FilePath))
            {
                this.OnClearDefFileButtonClicked(null, EventArgs.Empty);
            }
            else
            {
                this.CheckAndUpdateEmitterExtParams();
            }
        }

        /// <summary>
        /// カスタムアクションの設定が有効か無効かを返す。
        /// </summary>
        /// <returns>
        /// 有効ならtrue、無効ならfalse.
        /// </returns>
        public bool OnValidationCheck()
        {
            return this.filePathInput.BackColor == Color.White;
        }

        /// <summary>
        /// プロジェクトコンフィグをファイル出力する際に各UIの内容を出力用インスタンスに収集します。
        /// </summary>
        public void OnExportProjectSetting()
        {
            OptionUtil.TempConfig.EmitterExtParamsPath = this.filePathInput.FilePath;
        }

        /// <summary>
        /// プロジェクト設定ファイルの内容が不正だった場合のメッセージボックスを表示します.
        /// </summary>
        private void ShowWarningMessageBox()
        {
            ThreadSafeMsgBox.Show(
                Resources.InvalidSettingFileEmitterExtParams,
                Resources.WarningCaption,
                MessageBoxButtons.OK,
                MessageBoxIcon.Warning);
        }

        /// <summary>
        /// カスタムシェーダpane内の全警告を表示する.
        /// </summary>
        private void ShowAllWarningsInPane()
        {
            this.fileTreeView.Clear();
            this.filePathInput.BackColor = Color.Pink;
            if (this.DrawTreeView != null)
            {
                this.DrawTreeView();
            }

            // 最後のメッセージボックスを表示する.
            this.ShowWarningMessageBox();
        }

        /// <summary>
        /// カスタムアクションをチェックして読み込む.
        /// </summary>
        private void CheckAndUpdateEmitterExtParams()
        {
            if (string.IsNullOrEmpty(this.filePathInput.FilePath))
            {
                return;
            }

            if (!File.Exists(this.filePathInput.FilePath)
                || !IOConstants.CheckRootElement(this.filePathInput.FilePath, "ArrayOfEmitterExtParamsDefinition"))
            {
                this.ShowAllWarningsInPane();
            }
            else
            {
                this.UpdateFileHierarchyTree(this.filePathInput.FilePath);
            }

            OptionUtil.IsNeedReloadEmitterExtParams = true;
        }

        /// <summary>
        /// Handle Click event for the load button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnLoadDefFileButtonClicked(object sender, EventArgs e)
        {
            string initDir = null;
            if (!string.IsNullOrEmpty(this.filePathInput.FilePath))
            {
                string path = Path.GetDirectoryName(this.filePathInput.FilePath);
                if (Directory.Exists(path) == true)
                {
                    initDir = path;
                }
            }

            if (initDir == null)
            {
                initDir = ApplicationIOManager.GetLastAccessedDir(
                    LastAccessDirectoryTypes.EmitterExtParamsList);
            }

            var dlg = new OpenFileDialog()
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterXmlFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
                FileName = this.filePathInput.FilePath,
                InitialDirectory = initDir,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.UpdateFilePathInputTxt(dlg.FileName);
        }

        /// <summary>
        /// Handle Click event for the clear button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnClearDefFileButtonClicked(object sender, EventArgs e)
        {
            // Set the selected file path to the path text box.
            this.filePathInput.FilePath = DefaultEmitterExtParamsPath;

            // Update the file hierarchy tree view.
            this.UpdateFileHierarchyTree(DefaultEmitterExtParamsPath);
            OptionUtil.IsNeedReloadEmitterExtParams = true;
            this.filePathInput.BackColor = Color.White;
            if (this.DrawTreeView != null)
            {
                this.DrawTreeView();
            }
        }

        /// <summary>
        /// filePathInput.FilePathを更新する
        /// </summary>
        /// <param name="fileName">更新するファイル名(パス)</param>
        /// <returns>filePathInput.FilePathを更新した場合はtrueを返す. 更新しなかった場合はfalseを返す.</returns>
        private bool UpdateFilePathInputTxt(string fileName)
        {
            if (!IOConstants.CheckRootElement(fileName, "ArrayOfEmitterExtParamsDefinition"))
            {
                this.ShowWarningMessageBox();
                return false;
            }

            // Update the last accessed Emitter Extension Parameters list file path.
            ApplicationIOManager.SetLastAccessedDir(
                LastAccessDirectoryTypes.EmitterExtParamsList,
                fileName);

            // Set the selected file path to the path text box.
            this.filePathInput.FilePath = fileName;

            // Update the file hierarchy tree view.
            this.UpdateFileHierarchyTree(this.filePathInput.FilePath);

            // オプションウィンドウを閉じる時に、エミッタ拡張パラメータをロードさせる.
            OptionUtil.IsNeedReloadEmitterExtParams = true;

            return true;
        }

        /// <summary>
        /// The on Emitter Extension Parameters reload button click.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void OnEmitterExtParamsReloadButtonClick(object sender, EventArgs e)
        {
            this.CheckAndUpdateEmitterExtParams();
        }

        /// <summary>
        /// Load Emitter Extension Parameters definition file from the given path and
        /// update the file hierarchy tree.
        /// </summary>
        /// <param name="emitterExtParamsDefFilePath">The Emitter Extension Parameters definition file path.</param>
        private void UpdateFileHierarchyTree(string emitterExtParamsDefFilePath)
        {
            this.fileTreeView.Clear();

            // Is the Emitter Extension Parameters definition data loaded?
            if (File.Exists(emitterExtParamsDefFilePath) == false)
            {
                this.fileTreeView.AddText(
                    null,
                    Properties.Resources.OptionEmitterExtParamsDefFileNotSelected);

                // 警告を表示する.
                this.ShowAllWarningsInPane();
                return;
            }

            // ツリー表示用にカスタムアクション設定を読み込む
            var data = UserDataHelper.Instance.LoadEmitterExtParamsUserData(emitterExtParamsDefFilePath, false);

            // Emitter Extension Parametersのロードに失敗した場合はここで終了
            if (data == null)
            {
                // 警告を表示する.
                this.ShowAllWarningsInPane();
                return;
            }

            this.filePathInput.BackColor = Color.White;

            // The Emitter Extension Parameters definition file.
            var root = this.fileTreeView.AddFile(null, null, emitterExtParamsDefFilePath);

            // The Emitter Extension Parameters files.
            var emitterExtParamsNode = this.fileTreeView.AddText(
                    root,
                    Properties.Resources.OptionEmitterExtParamsNode);

            if (data.UsingUdd2)
            {
                this.fileTreeView.AddFile(
                    emitterExtParamsNode,
                    Resources.OptionCustomUserDataDefinitionFile + " : " + data.Caption,
                    data.Udd2FullFilePath);
            }
            else
            {
                // Data model source file.
                this.fileTreeView.AddFile(
                    emitterExtParamsNode,
                    Properties.Resources.OptionCustomActionDataModelFile,
                    data.DataModelFullFilePath);

                // XAML file.
                this.fileTreeView.AddFile(
                    emitterExtParamsNode,
                    Properties.Resources.OptionCustomActionXamlFile,
                    data.XamlFullFilePath);

                // Binary conversion info source file.
                this.fileTreeView.AddFile(
                    emitterExtParamsNode,
                    Properties.Resources.OptionCustomActionConversionInfoFile,
                    data.BinaryConversionInfoFullPath);
            }

            if (this.DrawTreeView != null)
            {
                this.DrawTreeView();
            }
        }
    }
}
