﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Windows.Forms;

using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Core;
using EffectMaker.UIControls.Specifics.Behaviors;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// THIS IS NOT AN OPTION PANEL!!!
    /// An user control for input effect combiner shader block definition file path.
    /// Used in FileEventOptionPane.
    /// </summary>
    public partial class EffectCombinerDefInput : UserControl
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public EffectCombinerDefInput()
        {
            this.InitializeComponent();

            // コンバイナ定義ファイルのドラッグ&ドロップ処理を設定
            {
                // 全てのファイルとディレクトリのドラッグアンドドロップを許可する
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowAllExtensions = true;
                dragDropBehavior.AllowDirectory = true;
                dragDropBehavior.ConvertToDirectoryPath = true;
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    this.txtCombinerDefPath.Text = dragDropBehavior.DropResult;
                };

                this.txtCombinerDefPath.Behaviors.Add(dragDropBehavior);
            }
        }

        /// <summary>
        /// Event triggers when the add combiner definition button is clicked.
        /// </summary>
        public event EventHandler AddButtonClicked;

        /// <summary>
        /// Event triggers when the remove combiner definition button is clicked.
        /// </summary>
        public event EventHandler RemoveButtonClicked;

        /// <summary>
        /// Get or set the flag indicating whether the add button is visible.
        /// </summary>
        public bool IsAddButtonVisible
        {
            get { return this.btnAddCombinerDef.Visible; }
            set { this.btnAddCombinerDef.Visible = value; }
        }

        /// <summary>
        /// Get or set the flag indicating whether the remove button is visible.
        /// </summary>
        public bool IsRemoveButtonVisible
        {
            get { return this.btnDelCombinerDef.Visible; }
            set { this.btnDelCombinerDef.Visible = value; }
        }

        /// <summary>
        /// Get or set the combiner definition folder path.
        /// </summary>
        public string CombinerDefFolderPath
        {
            get { return this.txtCombinerDefPath.Text; }
            set { this.txtCombinerDefPath.Text = value; }
        }

        /// <summary>
        /// Handle Click event for the browse button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnBrowseButtonClicked(object sender, EventArgs e)
        {
            var path = this.txtCombinerDefPath.Text;
            if (string.IsNullOrEmpty(path) == true)
            {
                path = ApplicationIOManager.GetLastAccessedDir(
                    LastAccessDirectoryTypes.EffectCombinerDefinition);
            }
            else if (!Path.IsPathRooted(path))
            {
                path = Path.GetDirectoryName(OptionStore.ProjectConfig.ConfigFilePath);
            }

            var dlg = new FolderBrowserDialog
            {
                SelectedPath = path,
                ShowNewFolderButton = false,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            ApplicationIOManager.SetLastAccessedDir(
                LastAccessDirectoryTypes.EffectCombinerDefinition,
                dlg.SelectedPath);

            this.txtCombinerDefPath.Text = dlg.SelectedPath;
        }

        /// <summary>
        /// Handle Click event for the add button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnAddButtonClicked(object sender, EventArgs e)
        {
            if (this.IsAddButtonVisible == true &&
                this.AddButtonClicked != null)
            {
                this.AddButtonClicked(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle Click event for the remove button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnRemoveButtonClicked(object sender, EventArgs e)
        {
            if (this.RemoveButtonClicked != null)
            {
                this.RemoveButtonClicked(this, EventArgs.Empty);
            }
        }
    }
}
